#include "StdAfx.h"
#include "Testing/FeatureTester.h"
#include "Utility/CryWatch.h"
#include "Utility/StringUtils.h"
#include "Utility/DesignerWarning.h"
#include "Player.h"
#include "IPlayerInput.h"
#include "GameCodeCoverage/GameCodeCoverageManager.h"
#include "GameRulesModules/IGameRulesSpawningModule.h"
#include "GameRules.h"
#include "Testing/AutoTester.h"
#include "FrontEnd/FlashFrontEnd.h"

#if ENABLE_FEATURE_TESTER

//===================================================================================================
// Macros (including auto-enum defines)
//===================================================================================================

#define CHECK_FOR_THIS_MANY_DUPLICATED_STRINGS_WHEN_READING_XML    512

// After reading the file, the actual amount of memory required will be allocated and the buffer will be disposed of... it's just for during the loading procedure
#define BUFFER_SIZE_FOR_READING_INSTRUCTIONS                       4096

#define FeatureTestDataTypeDetails(f)                                                                                                     \
	f(kFTVT_Command,  EFeatureTestCommand, (val >= 0 && val < kFTC_Num),           "%s",         s_featureTestCommandNames[val]           ) \
	f(kFTVT_Float,    float,               true,                                   "%f",         val                                      ) \
	f(kFTVT_Int,      int,                 true,                                   "%i",         val                                      ) \
	f(kFTVT_Bool,     bool,                (val == true || val == false),          "%s",         val ? "TRUE" : "FALSE"                   ) \
	f(kFTVT_SuitMode, ENanoSuitMode,       (val >= 0 && val < eNanoSuitMode_Last), "%d \"%s\"",  val, CNanoSuit::GetNanoSuitModeName(val) ) \
	f(kFTVT_Text,     const char *,        (val != NULL),                          "\"%s\"",     val                                      ) \
	f(kFTVT_HitResponse, EFTCheckpointHitResponse, (val >= 0 && val < kFTCHR_num), "\"%s\"",     s_featureTestHitResponseNames[val]       ) \
	f(kFTVT_EntityFlags, uint32,                   true,                           "%u",         val                                      ) \

#define FeatureTesterLog(...)       CryLogAlways("$3[FEATURETESTER]$1 %s%s", CFeatureTester::GetContextString().c_str(), string().Format(__VA_ARGS__).c_str())
#define FeatureTesterWarning(...)   DesignerWarning(false, "FEATURE-TESTER WARNING! %s%s", CFeatureTester::GetContextString().c_str(), string().Format(__VA_ARGS__).c_str())

#if defined(USER_timf) || defined(USER_martinsh)
#define FeatureTesterSpam(...)      FeatureTesterLog("SPAM! " __VA_ARGS__)
#else
#define FeatureTesterSpam(...)      (void)(0)
#endif

#define MAKE_UNION_CONTENTS(enumVal, theType, ...)        theType m_ ## enumVal;
#define MAKE_INSTRUCTION_FUNC_POINTER(instructionName)    &CFeatureTester::Instruction_ ## instructionName,

#define MAKE_FUNCTIONS(enumVal, theType, isValid, strFormat, ...)                          \
	ILINE bool AddData (theType val)                                                         \
	{                                                                                        \
	  m_type = enumVal;                                                                      \
		m_data.m_ ## enumVal = val;                                                            \
		return isValid;                                                                        \
	}                                                                                        \
	ILINE theType GetData_ ## enumVal () const                                               \
	{                                                                                        \
		theType val = m_data.m_ ## enumVal;                                                    \
		if (m_type != enumVal)		     																	                       \
		{																									                                     \
			CRY_ASSERT_TRACE(0, ("Expected next item to be of type %s but it's %s!",             \
				s_featureTestValTypeNames[enumVal], s_featureTestValTypeNames[m_type]));           \
		}                                                                                      \
		else if (! (isValid))                                                                  \
		{																									                                     \
			CRY_ASSERT_TRACE(0, ("Next item is right type (%s) but value is invalid!",           \
				s_featureTestValTypeNames[enumVal]));                                              \
		}                                                                                      \
		else                                                                                   \
		{																							                                         \
			FeatureTesterSpam("Getting " #enumVal " (" #theType ") " strFormat, __VA_ARGS__);    \
		}                                                                                      \
		return val;                                                                            \
	}

// TODO: Enum for the type of failure... did the test fail to run (error in data) or did the test run OK and find a problem with the game?
#define CheckFeatureTestFailure(condition, ...)        ((condition) || FeatureTestFailureFunc(#condition, string().Format(__VA_ARGS__)))
#define FeatureTestFailure(...)                        FeatureTestFailureFunc("No condition specified; please see message", string().Format(__VA_ARGS__))

//===================================================================================================
// Data types, structures and variable initialization
//===================================================================================================

AUTOENUM_BUILDENUMWITHTYPE_WITHINVALID_WITHNUM(EFeatureTestValType, FeatureTestDataTypeDetails, kFTVT_Invalid, kFTVT_Num);

static AUTOENUM_BUILDNAMEARRAY(s_featureTestCommandNames, FeatureTestCommandList);
static AUTOENUM_BUILDNAMEARRAY(s_featureTestValTypeNames, FeatureTestDataTypeDetails);
static AUTOENUM_BUILDNAMEARRAY(s_featureTestPauseReasonNames, FeatureTestPauseReasonList);
static AUTOENUM_BUILDNAMEARRAY(s_featureTestRequirementNames, FeatureTestRequirementList);
static AUTOENUM_BUILDNAMEARRAY(s_featureTestHitResponseNames, FeatureTestCheckpointHitResponseList);

CFeatureTester::InstructionFunc CFeatureTester::s_instructionFunctions[] =
{
	FeatureTestCommandList(MAKE_INSTRUCTION_FUNC_POINTER)
};

struct SFeatureTestInstructionOrParam
{
	EFeatureTestValType m_type;

	union
	{
		FeatureTestDataTypeDetails(MAKE_UNION_CONTENTS)
	} m_data;

	FeatureTestDataTypeDetails(MAKE_FUNCTIONS)
};

struct SFeatureTestDataLoadWorkspace
{
	private:
	SFeatureTestInstructionOrParam m_instructionBuffer[BUFFER_SIZE_FOR_READING_INSTRUCTIONS];
	int m_count;

	public:
	SFeatureTestDataLoadWorkspace()
	{
		m_count = 0;
	}

	ILINE int GetNumInstructions()                                       { return m_count;             }
	ILINE const SFeatureTestInstructionOrParam * GetInstructionBuffer()  { return m_instructionBuffer; }

	template <class T>
	bool AddData(T data)
	{
		if (m_count < BUFFER_SIZE_FOR_READING_INSTRUCTIONS)
		{
			return m_instructionBuffer[m_count ++].AddData(data);
		}
		FeatureTesterWarning("Out of memory reading feature tester data - please increase BUFFER_SIZE_FOR_READING_INSTRUCTIONS (currently %d)", BUFFER_SIZE_FOR_READING_INSTRUCTIONS);
		return false;
	}
};

class CFeatureTestArgumentAutoComplete : public IConsoleArgumentAutoComplete
{
	virtual int GetCount() const { return CFeatureTester::s_instance->m_numTests; }
	virtual const char* GetValue( int nIndex ) const { return CFeatureTester::s_instance->m_featureTestArray[nIndex].m_testName; }
};

class CFeatureTestFilenameAutoComplete : public IConsoleArgumentAutoComplete
{
	public:
	CFeatureTestFilenameAutoComplete()
	{
		m_lastFrameUsed = -1;
		m_nonConstThis = this;
	}

	private:
	int m_lastFrameUsed;
	std::vector<string> m_filenames;
	CFeatureTestFilenameAutoComplete * m_nonConstThis;

	virtual int GetCount() const
	{
		int curFrame = gEnv->pRenderer->GetFrameID(false);
		if (m_lastFrameUsed != curFrame)
		{
			m_nonConstThis->FindAllFiles(curFrame);
		}

		return m_filenames.size();
	}
	virtual const char* GetValue( int nIndex ) const { return m_filenames[nIndex].c_str(); }

	void FindAllFiles(int curFrame)
	{
		m_lastFrameUsed = curFrame;
		m_filenames.clear();

		ICryPak *pPak = gEnv->pCryPak;
		_finddata_t fd;
		intptr_t handle = pPak->FindFirst("Scripts/FeatureTests/*.xml", &fd);

		if (handle > -1)
		{
			do
			{
				char buffer[64];
				size_t charsCopied = cry_copyStringUntilFindChar(buffer, fd.name, sizeof(buffer), '.');
				if (charsCopied && 0 == stricmp(fd.name + charsCopied, "xml"))
				{
					string addThis = buffer;
					m_filenames.push_back(addThis);
				}
			}
			while (pPak->FindNext(handle, &fd) >= 0);

			pPak->FindClose(handle);
		}
	}
};

CFeatureTester * CFeatureTester::s_instance = NULL;
static CFeatureTestArgumentAutoComplete s_featureTestArgumentAutoComplete;
static CFeatureTestFilenameAutoComplete s_featureTestFilenameAutoComplete;

//===================================================================================================
// Functions
//===================================================================================================

//-------------------------------------------------------------------------------
CFeatureTester::CFeatureTester() :
	REGISTER_GAME_MECHANISM(CFeatureTester),
	m_featureTestArray(NULL)
{
	FeatureTesterLog ("Creating feature tester instance");

	m_informAutoTesterOfResults = NULL;
	m_currentTest = NULL;
	m_currentTestNextInstruction = NULL;
	m_singleBufferContainingAllInstructions = NULL;

	m_numWatchedCheckpoints = 0;
	m_waitUntilCCCPointHit_numStillToHit = 0;
	m_runFeatureTestStack.m_count = 0;
	m_numOverriddenInputs = 0;
	m_numTests = 0;
	m_numFeatureTestsLeftToAutoRun = 0;
	m_abortUntilNextFrame = false;
	m_pause_enableCountdown = false;
	m_timeSinceCurrentTestBegan = 0.f;
	m_saveScreenshotWhenFail = 0;

	memset(& m_iterateOverParams, 0, sizeof(m_iterateOverParams));
	memset(& m_nextIteration, 0, sizeof(m_nextIteration));

	assert (s_instance == NULL);
	s_instance = this;

	IConsole * console = GetISystem()->GetIConsole();
	assert (console);

	console->AddCommand("ft_startTest", CmdStartTest, VF_CHEAT, "FEATURE TESTER: Start a feature test");
	console->AddCommand("ft_reload", CmdReload, VF_CHEAT, "FEATURE TESTER: Reload current feature tester data file");
	console->AddCommand("ft_load", CmdLoad, VF_CHEAT, "FEATURE TESTER: Load a feature tester data file");
	console->AddCommand("ft_runAll", CmdRunAll, VF_CHEAT, "FEATURE TESTER: Run all enabled feature tests");
	console->RegisterAutoComplete("ft_startTest", & s_featureTestArgumentAutoComplete);
	console->RegisterAutoComplete("ft_load", & s_featureTestFilenameAutoComplete);
	console->Register("ft_saveScreenshotWhenFail", & m_saveScreenshotWhenFail, m_saveScreenshotWhenFail, VF_CHEAT, "FEATURE TESTER: When non-zero, a screenshot will be saved whenever a feature test fails");
}

//-------------------------------------------------------------------------------
CFeatureTester::~CFeatureTester()
{
	UnloadTestData();

	IConsole * console = GetISystem()->GetIConsole();
	if (console)
	{
		console->RemoveCommand("ft_startTest");
		console->RemoveCommand("ft_reload");
		console->RemoveCommand("ft_load");
		console->RemoveCommand("ft_runAll");
		console->UnregisterVariable("ft_saveScreenshotWhenFail");
	}

	assert (s_instance == this);
	s_instance = NULL;
}

//-------------------------------------------------------------------------------
string CFeatureTester::GetContextString()
{
	string reply;
	if (s_instance == NULL)
	{
		reply.Format("(No instance) ");
	}
	else
	{
		if (s_instance->m_timeSinceCurrentTestBegan > 0.f)
		{
			reply.Format("@%.3f ", s_instance->m_timeSinceCurrentTestBegan);
		}

		if (s_instance->m_currentTest)
		{
			reply.append("(Test: ");

			if (s_instance->m_iterateOverParams.m_numParams)
			{
				for (int i = 0; i < s_instance->m_iterateOverParams.m_numParams; ++ i)
				{
					reply.append(i ? ", " : "<");
					reply.append(s_instance->m_iterateOverParams.m_currentParams[i]);
				}
				reply.append("> ");
			}

			reply.append(s_instance->m_currentTest->m_testName);
		
			for (int i = 0; i < s_instance->m_runFeatureTestStack.m_count; ++ i)
			{
				reply.append("=>");
				reply.append(s_instance->m_runFeatureTestStack.m_info[i].m_calledTest->m_testName);
			}

			reply.append(") ");
		}
	}
	return reply;
}

//-------------------------------------------------------------------------------
int CFeatureTester::PreprocessTestSet(const IItemParamsNode *testsListNode)
{
	int numTests = 0;
	int numNodesInThisSet = testsListNode->GetChildCount();
	const char * setName = testsListNode->GetAttribute("setName");

	m_singleAllocTextBlock.IncreaseSizeNeeded(setName);

	// Quickly run through all test data to calculate the amount of memory we need to allocate for strings
	for (int readChildNum = 0; readChildNum < numNodesInThisSet; ++ readChildNum)
	{
		const IItemParamsNode * oneTest = testsListNode->GetChild(readChildNum);

		if (0 == stricmp (oneTest->GetName(), "FeatureTest"))
		{
			++ numTests;

			int numCommandsInTest = oneTest->GetChildCount();
			m_singleAllocTextBlock.IncreaseSizeNeeded(oneTest->GetAttribute("name"));
			m_singleAllocTextBlock.IncreaseSizeNeeded(oneTest->GetAttribute("description"));
			m_singleAllocTextBlock.IncreaseSizeNeeded(oneTest->GetAttribute("iterateOverParams"));

			for (int readCommandNum = 0; readCommandNum < numCommandsInTest; ++ readCommandNum)
			{
				const IItemParamsNode * cmdParams = oneTest->GetChild(readCommandNum);
				m_singleAllocTextBlock.IncreaseSizeNeeded(cmdParams->GetAttribute("command"));
				m_singleAllocTextBlock.IncreaseSizeNeeded(cmdParams->GetAttribute("checkpointName"));
				m_singleAllocTextBlock.IncreaseSizeNeeded(cmdParams->GetAttribute("inputName"));
				m_singleAllocTextBlock.IncreaseSizeNeeded(cmdParams->GetAttribute("className"));
				m_singleAllocTextBlock.IncreaseSizeNeeded(cmdParams->GetAttribute("testName"));
			}
		}
		else
		{
			FeatureTesterWarning ("Found unexpected tag of type '%s' while reading test set '%s'", oneTest->GetName(), setName);
		}
	}

	return numTests;
}

//-------------------------------------------------------------------------------
bool CFeatureTester::ReadTestSet(const IItemParamsNode *testsListNode, SFeatureTestDataLoadWorkspace * loadWorkspace)
{
	bool bOk = true;

	const char * setName = m_singleAllocTextBlock.StoreText(testsListNode->GetAttribute("setName"));
	int numChildrenInThisSet = testsListNode->GetChildCount();

	for (int readChildNum = 0; bOk && (readChildNum < numChildrenInThisSet); ++ readChildNum)
	{
		const IItemParamsNode * oneTest = testsListNode->GetChild(readChildNum);

		if (0 == stricmp (oneTest->GetName(), "FeatureTest"))
		{
			int numCommandsInTest = oneTest->GetChildCount();

			SFeatureTest & createTest = m_featureTestArray[m_numTests ++];
			createTest.m_offsetIntoInstructionBuffer = loadWorkspace->GetNumInstructions();
			createTest.m_enabled = false;
			createTest.m_autoRunThis = false;
			createTest.m_setName = setName;
			createTest.m_testName = m_singleAllocTextBlock.StoreText(oneTest->GetAttribute("name"));
			createTest.m_testDescription = m_singleAllocTextBlock.StoreText(oneTest->GetAttribute("description"));
			createTest.m_iterateOverParams = m_singleAllocTextBlock.StoreText(oneTest->GetAttribute("iterateOverParams"));
			createTest.m_requirementBitfield = AutoEnum_GetBitfieldFromString(oneTest->GetAttribute("require"), s_featureTestRequirementNames, FeatureTestRequirementList_numBits);
			createTest.m_maxTime = 60.f;

			oneTest->GetAttribute("maxTime", createTest.m_maxTime);

			if (createTest.m_requirementBitfield & kFTReq_localPlayerExists)     createTest.m_requirementBitfield |= kFTReq_inLevel;
			if (createTest.m_requirementBitfield & kFTReq_remotePlayerExists)    createTest.m_requirementBitfield |= kFTReq_inLevel;

			int readEnabled;
			if (oneTest->GetAttribute("enabled", readEnabled))
			{
				createTest.m_enabled = (readEnabled != 0);
			}

			for (int readCommandNum = 0; bOk && (readCommandNum < numCommandsInTest); ++ readCommandNum)
			{
				const IItemParamsNode * cmdParams = oneTest->GetChild(readCommandNum);
				const char * cmdName = cmdParams->GetName();
				int commandIndex = -1;

				if (AutoEnum_GetEnumValFromString(cmdName, s_featureTestCommandNames, kFTC_Num, & commandIndex))
				{
					EFeatureTestCommand cmdID = (EFeatureTestCommand)commandIndex;
					if (! AddInstructionAndParams(cmdID, cmdParams, loadWorkspace))
					{
						bOk = false;
						FeatureTesterWarning("While reading feature test '%s' set '%s' instruction number %d: error while adding instruction '%s'", createTest.m_testName, setName, 1 + readCommandNum, cmdName);
					}
				}
				else
				{
					bOk = false;
					FeatureTesterWarning("While reading feature test '%s' set '%s' instruction number %d: '%s' isn't a valid instruction name", createTest.m_testName, setName, 1 + readCommandNum, cmdName);
				}
			}

			if (bOk && ! AddInstructionAndParams(kFTC_End, NULL, loadWorkspace))
			{
				bOk = false;
				FeatureTesterWarning("While reading feature test '%s' set '%s': error while adding instruction 'kTFC_End'", createTest.m_testName, setName);
			}

			FeatureTesterLog ("Finished reading test '%s' in set '%s' - ID = #%04d, size = %d, test enabled = %s, still ok = %s", createTest.m_testName, setName, m_numTests, loadWorkspace->GetNumInstructions() - createTest.m_offsetIntoInstructionBuffer, createTest.m_enabled ? "YES" : "NO", bOk ? "YES" : "NO");

#if 0 && defined(USER_timf)
			cry_displayMemInHexAndAscii(string().Format("[FEATURETESTER] Test %d \"%s\": ", readChildNum + 1, createTest.m_testName), loadWorkspace->m_instructionBuffer + createTest.m_offsetIntoInstructionBuffer, (loadWorkspace->m_count - createTest.m_offsetIntoInstructionBuffer) * sizeof(loadWorkspace->m_instructionBuffer[0]), CCryLogOutputHandler(), sizeof(loadWorkspace->m_instructionBuffer[0]));
#endif
		}
	}

	return bOk;
}

//-------------------------------------------------------------------------------
void CFeatureTester::LoadTestData(const char * filenameNoPathOrExtension)
{
	m_currentlyLoadedFileName = filenameNoPathOrExtension;

	string filenameStr;
	filenameStr.Format("Scripts/FeatureTests/%s.xml", filenameNoPathOrExtension);
	const char * filename = filenameStr.c_str();

	FeatureTesterLog ("Loading feature test data from file '%s' i.e. '%s'", filenameNoPathOrExtension, filename);

	assert (m_currentTest == NULL);
	assert (m_currentTestNextInstruction == NULL);
	assert (m_numWatchedCheckpoints == 0);
	assert (m_waitUntilCCCPointHit_numStillToHit == 0);
	assert (m_numOverriddenInputs == 0);
	assert (m_numTests == 0);

	XmlNodeRef rootNode = gEnv->pSystem->LoadXmlFile(filename);
	const char * failureReason = NULL;
	
	if (rootNode && 0 == strcmpi(rootNode->getTag(), "FeatureTester"))
	{
		SFeatureTestDataLoadWorkspace loadWorkspace;

		assert (loadWorkspace.GetNumInstructions() == 0);

		CSingleAllocTextBlock::SReuseDuplicatedStrings duplicatedStringsWorkspace[CHECK_FOR_THIS_MANY_DUPLICATED_STRINGS_WHEN_READING_XML];
		m_singleAllocTextBlock.SetDuplicatedStringWorkspace(duplicatedStringsWorkspace, CHECK_FOR_THIS_MANY_DUPLICATED_STRINGS_WHEN_READING_XML);

		IItemParamsNode *paramNode = g_pGame->GetIGameFramework()->GetIItemSystem()->CreateParams();
		paramNode->ConvertFromXML(rootNode);

		int numChildNodes = paramNode->GetChildCount();
		int numTestsToRead = 0;

		for (int childNodeNum = 0; childNodeNum < numChildNodes; ++ childNodeNum)
		{
			const IItemParamsNode * thisNode = paramNode->GetChild(childNodeNum);
			const char * nodeName = thisNode->GetName();

			if (0 == stricmp (nodeName, "settings"))
			{
				// TODO: read things from here if needed...
			}
			else if (0 == stricmp (nodeName, "tests"))
			{
				numTestsToRead += PreprocessTestSet (thisNode);
			}
			else
			{
				FeatureTesterWarning ("Found unexpected tag of type '%s' while reading '%s'", nodeName, filename);
			}
		}

		// Allocate memory
		m_singleAllocTextBlock.Allocate();
		m_featureTestArray = new SFeatureTest[numTestsToRead];
		ASSERT_IS_NOT_NULL(m_featureTestArray);

		// Run through all test data again and store stuff in the allocated memory
		for (int childNodeNum = 0; failureReason == NULL && (childNodeNum < numChildNodes); ++ childNodeNum)
		{
			const IItemParamsNode *testsListNode = paramNode->GetChild(childNodeNum);
			const char * nodeName = testsListNode->GetName();

			if (0 == stricmp (nodeName, "tests"))
			{
				if (! ReadTestSet(testsListNode, & loadWorkspace))
				{
					failureReason = "error reading a set of tests";
				}
			}
		}

		paramNode->Release();

		if (failureReason == NULL)
		{
			if (numTestsToRead != m_numTests)
			{
				failureReason = numTestsToRead < m_numTests ? "read more feature tests than expected" : "read fewer feature tests than expected";
			}
			else
			{
				// Success!

				assert (m_singleBufferContainingAllInstructions == NULL);
				m_singleBufferContainingAllInstructions = new SFeatureTestInstructionOrParam[loadWorkspace.GetNumInstructions()];

				if (m_singleBufferContainingAllInstructions == NULL)
				{
					failureReason = "couldn't allocate memory to store instructions";
				}
				else
				{
					size_t numBytes = loadWorkspace.GetNumInstructions() * sizeof(SFeatureTestInstructionOrParam);
					FeatureTesterLog ("Allocated %d bytes of memory for storing %d instructions/parameters", numBytes, loadWorkspace.GetNumInstructions());
					memcpy (m_singleBufferContainingAllInstructions, loadWorkspace.GetInstructionBuffer(), numBytes);
				}
			}
		}
	}
	else
	{
		failureReason = "file could not be parsed";
	}

	if (failureReason == NULL)
	{
		m_singleAllocTextBlock.Lock();
		FeatureTesterLog ("Loaded all data for feature tester successfully!");
	}
	else
	{
		FeatureTesterWarning ("Failed to load '%s': %s", filename, failureReason);
		UnloadTestData();
	}
}

//-------------------------------------------------------------------------------
bool CFeatureTester::AddInstructionAndParams(EFeatureTestCommand cmd, const IItemParamsNode * paramsNode, SFeatureTestDataLoadWorkspace * loadWorkspace)
{
	bool bOk = loadWorkspace->AddData(cmd);

	switch (cmd)
	{
		case kFTC_MovePlayerToOtherEntity:
		{
			int useLocalPlayerInt = false;
			bOk &= paramsNode->GetAttribute("localPlayer", useLocalPlayerInt);
			bool useLocalPlayer = (useLocalPlayerInt != 0);
			bOk &= loadWorkspace->AddData(useLocalPlayer);

			bOk &= loadWorkspace->AddData(m_singleAllocTextBlock.StoreText(paramsNode->GetAttribute("className")));

			// TODO: Read flags from XML instead of hardcoding!
			uint32 requireFlags = ENTITY_FLAG_ON_RADAR;
			uint32 skipWithFlags = 0;
			bOk &= loadWorkspace->AddData(requireFlags);
			bOk &= loadWorkspace->AddData(skipWithFlags);

			// TODO: Read offset pos from XML!
		}
		break;

		case kFTC_TrySpawnPlayer:
		case kFTC_WaitUntilPlayerIsAlive:
		{
			int useLocalPlayerInt = false;
			bOk &= paramsNode->GetAttribute("localPlayer", useLocalPlayerInt);
			bool useLocalPlayer = (useLocalPlayerInt != 0);
			bOk &= loadWorkspace->AddData(useLocalPlayer);
		}
		break;

		case kFTC_RunFeatureTest:
		bOk &= loadWorkspace->AddData(m_singleAllocTextBlock.StoreText(paramsNode->GetAttribute("testName")));
		break;

		case kFTC_DoConsoleCommand:
		case kFTC_DoMenuCommand:
		bOk &= loadWorkspace->AddData(m_singleAllocTextBlock.StoreText(paramsNode->GetAttribute("command")));
		break;

		case kFTC_WatchCCCPoint:
		bOk &= loadWorkspace->AddData(m_singleAllocTextBlock.StoreText(paramsNode->GetAttribute("checkpointName")));
		break;

		case kFTC_SetResponseToHittingCCCPoint:
		{
			int responseAsInt;
			bOk &= AutoEnum_GetEnumValFromString(paramsNode->GetAttribute("response"), s_featureTestHitResponseNames, kFTCHR_num, & responseAsInt);
			EFTCheckpointHitResponse response = (EFTCheckpointHitResponse) responseAsInt;
			bOk &= loadWorkspace->AddData(m_singleAllocTextBlock.StoreText(paramsNode->GetAttribute("checkpointName")));
			bOk &= loadWorkspace->AddData((EFTCheckpointHitResponse) responseAsInt);

			if (response == kFTCHR_restartTest)
			{
				float delay = 0.f;
				paramsNode->GetAttribute("restartDelay", delay);
				bOk &= loadWorkspace->AddData(delay);
			}
		}
		break;

		case kFTC_WaitUntilHitAllExpectedCCCPoints:
		{
			float timeout = 10.f;
			bOk &= paramsNode->GetAttribute("timeout", timeout);
			bOk &= loadWorkspace->AddData(timeout);
		}
		break;

		case kFTC_SetItem:
		bOk &= loadWorkspace->AddData(m_singleAllocTextBlock.StoreText(paramsNode->GetAttribute("className")));
		break;

		case kFTC_OverrideButtonInput_Press:
		case kFTC_OverrideButtonInput_Release:
		bOk &= loadWorkspace->AddData(m_singleAllocTextBlock.StoreText(paramsNode->GetAttribute("inputName")));
		break;

		case kFTC_OverrideAnalogInput:
		{
			float amount;
			bOk &= paramsNode->GetAttribute("value", amount);
			bOk &= loadWorkspace->AddData(m_singleAllocTextBlock.StoreText(paramsNode->GetAttribute("inputName")));
			bOk &= loadWorkspace->AddData(amount);
		}
		break;

		case kFTC_Wait:
		{
			float duration;
			bOk &= paramsNode->GetAttribute("duration", duration);
			bOk &= loadWorkspace->AddData(duration);
		}
		break;

		case kFTC_CheckNumCCCPointHits:
		{
			int expectedNumHits = 0;
			bOk &= paramsNode->GetAttribute("expectedNumHits", expectedNumHits);
			bOk &= loadWorkspace->AddData(m_singleAllocTextBlock.StoreText(paramsNode->GetAttribute("checkpointName")));
			bOk &= loadWorkspace->AddData(expectedNumHits);
		}
		break;

		case kFTC_SetSuitMode:
		{
			const char * modeName = paramsNode->GetAttribute("modeName");
			ENanoSuitMode new_suit_mode = eNanoSuitMode_Invalid;

			if (modeName)
			{
				for (int i = 0; i < eNanoSuitMode_Last; ++ i)
				{
					if (0 == stricmp(modeName, CNanoSuit::GetNanoSuitModeName(i)))
					{
						new_suit_mode = (ENanoSuitMode) i;
						break;
					}
				}
			}

			if (! loadWorkspace->AddData(new_suit_mode))
			{
				bOk = false;
				FeatureTesterWarning ("%s nanosuit mode parameter '%s' while loading a feature test!", (new_suit_mode == eNanoSuitMode_Invalid) ? "Found invalid" : "Failed to add", modeName);
			}
		}
		break;
	}

	return bOk;
}

//-------------------------------------------------------------------------------
void CFeatureTester::SendInputToLocalPlayer(const char * inputName, EActionActivationMode mode, float value)
{
	CPlayer *pPlayer = static_cast<CPlayer *>(gEnv->pGame->GetIGameFramework()->GetClientActor());
	if(pPlayer && pPlayer->GetPlayerInput())
	{
		pPlayer->GetPlayerInput()->OnAction(inputName, mode, value);

		const char * releaseName = inputName;

		// TODO: Define these special cases (where the 'press' and 'release' actions have different names) in data file...
		if (0 == strcmp(inputName, "suitmode_menu_open"))
		{
			releaseName = "suitmode_menu_close";
		}

		// Also remember what inputs we've overridden, so we can reset them to defaults when we stop the test...
		int foundAtIndex = 0;
		while (foundAtIndex < m_numOverriddenInputs && strcmp(releaseName, m_currentlyOverriddenInputs[foundAtIndex].m_inputName))
		{
			++ foundAtIndex;
		}

		if (! ((mode == eAAM_Always && value == 0.f) || (mode == eAAM_OnRelease)))
		{
			// Store it
			if (foundAtIndex == m_numOverriddenInputs)
			{
				if (foundAtIndex < kMaxSimultaneouslyOverriddenInputs)
				{
					m_currentlyOverriddenInputs[foundAtIndex].m_inputName = releaseName;
					m_currentlyOverriddenInputs[foundAtIndex].m_mode = mode;
					++ m_numOverriddenInputs;
				}
				else
				{
					FeatureTesterWarning ("%d inputs currently overridden, can't remember '%s' too...", m_numOverriddenInputs, releaseName);
				}
			}
			else
			{
				assert (m_currentlyOverriddenInputs[foundAtIndex].m_mode == mode);
			}
		}
		else if (foundAtIndex < m_numOverriddenInputs)
		{
			// Remove it from the list
			m_currentlyOverriddenInputs[foundAtIndex] = m_currentlyOverriddenInputs[-- m_numOverriddenInputs];
		}
	}
}

//-------------------------------------------------------------------------------
const SFeatureTestInstructionOrParam * CFeatureTester::GetNextInstructionOrParam()
{
	assert (m_currentTestNextInstruction);
	return (m_currentTestNextInstruction++);
}

//-------------------------------------------------------------------------------
const char * CFeatureTester::GetTextParam()
{
	const char * param = GetNextInstructionOrParam()->GetData_kFTVT_Text();

	if (param && param[0] == '%' && param[1] >= '1' && param[1] < ('1' + m_iterateOverParams.m_numParams) && param[2] == '\0')
	{
		int paramNum = param[1] - '1';
		assert (paramNum < m_iterateOverParams.m_numParams);
		FeatureTesterSpam ("Found text '%s' so substituting parameter %d i.e. '%s'", param, paramNum, m_iterateOverParams.m_currentParams[paramNum]);
		param = m_iterateOverParams.m_currentParams[paramNum];
		assert(param);
	}

	return param;
}

//-------------------------------------------------------------------------------
void CFeatureTester::Instruction_kFTC_OverrideAnalogInput()
{
	const char * inputName = GetTextParam();
	float theValue = GetNextInstructionOrParam()->GetData_kFTVT_Float();
	SendInputToLocalPlayer(inputName, eAAM_Always, theValue);
}

//-------------------------------------------------------------------------------
void CFeatureTester::Instruction_kFTC_End()
{
	CompleteSubroutine();
}

//-------------------------------------------------------------------------------
void CFeatureTester::Instruction_kFTC_Fail()
{
	FeatureTestFailure("Reached an explicit 'Fail' command in the list of test instructions");
}

//-------------------------------------------------------------------------------
void CFeatureTester::Instruction_kFTC_WaitSingleFrame()
{
	PauseExecution();
}

//-------------------------------------------------------------------------------
void CFeatureTester::SetPauseStateAndTimeout(EFTPauseReason pauseReason, float timeOut)
{
	assert (timeOut >= 0.f);
	assert (pauseReason != kFTPauseReason_none || timeOut == 0.f);

	m_pause_enableCountdown = false;
	m_pause_state = pauseReason;
	m_pause_timeLeft = timeOut;
	m_pause_originalTimeOut = timeOut;
}

//-------------------------------------------------------------------------------
void CFeatureTester::Instruction_kFTC_Wait()
{
	SetPauseStateAndTimeout(kFTPauseReason_untilTimeHasPassed, GetNextInstructionOrParam()->GetData_kFTVT_Float());
}

//-------------------------------------------------------------------------------
void CFeatureTester::Instruction_kFTC_WaitUntilHitAllExpectedCCCPoints()
{
	float timeOut = GetNextInstructionOrParam()->GetData_kFTVT_Float();

	if (m_waitUntilCCCPointHit_numStillToHit > 0)
	{
		SetPauseStateAndTimeout(kFTPauseReason_untilCCCPointsHit, timeOut);
		FeatureTesterLog ("Waiting until %s %s hit (timeout = %.1f seconds)", GetListOfCheckpointsExpected().c_str(), (m_waitUntilCCCPointHit_numStillToHit == 1) ? "is" : "are", timeOut);
	}
	else
	{
		FeatureTesterLog ("No checkpoints still to be hit by the time we reached the WaitUntilHitAllExpectedCCCPoints instruction; continuing immediately");
	}
}

//-------------------------------------------------------------------------------
void CFeatureTester::Instruction_kFTC_ResetCCCPointHitCounters()
{
	for (int i = 0; i < m_numWatchedCheckpoints; ++ i)
	{
		m_checkpointCountArray[i].m_timesHit = 0;
	}
}

//-------------------------------------------------------------------------------
void CFeatureTester::Instruction_kFTC_SetItem()
{
	const char * itemClassName = GetTextParam();
	CActor * pClientActor = (CActor*)gEnv->pGame->GetIGameFramework()->GetClientActor();

	if(pClientActor)
	{
		IInventory * pInventory = pClientActor->GetInventory();
		IEntityClass* itemClassPtr = gEnv->pEntitySystem->GetClassRegistry()->FindClass(itemClassName);

		if (CheckFeatureTestFailure(itemClassPtr != NULL, "There's no class called '%s'", itemClassName))
		{
			EntityId itemId = pInventory->GetItemByClass(itemClassPtr);

			// If we fail to switch to the specified item we can try to add a new one to the player's inventory...
			if (itemId == 0)
			{
				gEnv->pConsole->ExecuteString(string().Format("i_giveitem %s", itemClassName));
				itemId = pInventory->GetItemByClass(itemClassPtr);
			}

			if (CheckFeatureTestFailure(itemId != 0, "%s isn't carrying a '%s' (and i_giveitem failed to create one)", pClientActor->GetEntity()->GetName(), itemClassName))
			{
				pClientActor->SelectItem(itemId, false);
				SetPauseStateAndTimeout(kFTPauseReason_untilWeaponIsReadyToUse, 10.f);
			}
		}
	}
}

//-------------------------------------------------------------------------------
void CFeatureTester::Instruction_kFTC_OverrideButtonInput_Press()
{
	SendInputToLocalPlayer(GetTextParam(), eAAM_OnPress, 1.f);
}

//-------------------------------------------------------------------------------
void CFeatureTester::Instruction_kFTC_OverrideButtonInput_Release()
{
	SendInputToLocalPlayer(GetTextParam(), eAAM_OnRelease, 0.f);
}

//-------------------------------------------------------------------------------
void CFeatureTester::Instruction_kFTC_WatchCCCPoint()
{
	WatchCheckpoint(GetTextParam());
}

//-------------------------------------------------------------------------------
void CFeatureTester::Instruction_kFTC_CheckNumCCCPointHits()
{
	const char * checkpointName = GetTextParam();
	int expectedNumTimesHit = GetNextInstructionOrParam()->GetData_kFTVT_Int();
	int actualNumTimesHit = 0;

	for (int i = 0; i < m_numWatchedCheckpoints; ++ i)
	{
		if (0 == stricmp(checkpointName, m_checkpointCountArray[i].m_checkpointName))
		{
			actualNumTimesHit = m_checkpointCountArray[i].m_timesHit;
			break;
		}
	}

	CheckFeatureTestFailure(actualNumTimesHit == expectedNumTimesHit, "Expected to have hit '%s' %d times but have actually hit it %d times", checkpointName, expectedNumTimesHit, actualNumTimesHit);
}

//-------------------------------------------------------------------------------
void CFeatureTester::Instruction_kFTC_SetSuitMode()
{
	CPlayer *pPlayer = static_cast<CPlayer *>(gEnv->pGame->GetIGameFramework()->GetClientActor());
	CNanoSuit * nanosuit = pPlayer ? pPlayer->GetNanoSuit() : NULL;
	ENanoSuitMode new_suit_mode = GetNextInstructionOrParam()->GetData_kFTVT_SuitMode();

	if (nanosuit && new_suit_mode != eNanoSuitMode_Invalid)
	{
		nanosuit->ActivateMode(new_suit_mode);
	}
}

//-------------------------------------------------------------------------------
void CFeatureTester::Instruction_kFTC_SetResponseToHittingCCCPoint()
{
	const char * cpName = GetTextParam();
	EFTCheckpointHitResponse response = GetNextInstructionOrParam()->GetData_kFTVT_HitResponse();
	SCheckpointCount * watchedCheckpoint = WatchCheckpoint(cpName);

	if (CheckFeatureTestFailure(watchedCheckpoint, "Feature tester isn't watching a game code coverage checkpoint called '%s'", cpName))
	{
		SetCheckpointHitResponse(watchedCheckpoint, response);

		if (response == kFTCHR_restartTest)
		{
			watchedCheckpoint->m_restartDelay = GetNextInstructionOrParam()->GetData_kFTVT_Float();
		}
	}
}

//-------------------------------------------------------------------------------
void CFeatureTester::Instruction_kFTC_DoConsoleCommand()
{
	const char * cmd = GetTextParam();
	FeatureTesterLog("Current feature test '%s' is executing console command '%s'", m_currentTest->m_testName, cmd);
	
	// TODO: Might be a good idea to add a 'defer' bool to XML...
	bool defer = false;
	gEnv->pConsole->ExecuteString(string(cmd), false, defer);
	PauseExecution(defer);
}

//-------------------------------------------------------------------------------
void CFeatureTester::Instruction_kFTC_RunFeatureTest()
{
	const char * testName = GetTextParam();
	FeatureTesterLog("Current feature test '%s' is triggering another feature test '%s'", m_currentTest->m_testName, testName);

	if (CheckFeatureTestFailure (m_runFeatureTestStack.m_count < m_runFeatureTestStack.k_stackSize, "Out of room on nested-feature-test execution stack (size=%d) when trying to run '%s'", m_runFeatureTestStack.k_stackSize, testName))
	{
		SFeatureTest * test = FindTestByName(testName);

		if (CheckFeatureTestFailure(test, "There's no feature test called '%s'", testName))
		{
			SStackedTestCallInfo * newInfo = & m_runFeatureTestStack.m_info[m_runFeatureTestStack.m_count ++];
			newInfo->m_calledTest = test;
			newInfo->m_returnToHereWhenDone = m_currentTestNextInstruction;
			FeatureTesterSpam ("Subroutine %s (@%u)", test->m_testName, test->m_offsetIntoInstructionBuffer);
			m_currentTestNextInstruction = m_singleBufferContainingAllInstructions + test->m_offsetIntoInstructionBuffer;
		}
	}
}

//-------------------------------------------------------------------------------
void CFeatureTester::Instruction_kFTC_DoMenuCommand()
{
	const char * allText = GetTextParam();
	char stuffBeforeFirstSpace[256];
	size_t numBytesCopied = cry_copyStringUntilFindChar(stuffBeforeFirstSpace, allText, sizeof(stuffBeforeFirstSpace), ' ');
	const char * cmd = numBytesCopied ? stuffBeforeFirstSpace : allText;
	const char * params = numBytesCopied ? allText + numBytesCopied : NULL;

	// g_pGame->GetMenu() returns NULL on a dedicated build...
	CFlashFrontEnd * menuObject = g_pGame->GetFlashMenu();
	if (CheckFeatureTestFailure(menuObject, "Can't execute menu command '%s' params '%s' - game has no menu object!", cmd, params))
	{
		FeatureTesterLog("Current feature test '%s' is executing front-end command '%s' params '%s'", m_currentTest->m_testName, cmd, params);
		menuObject->HandleFSCommand(cmd, params);
	}
}

//-------------------------------------------------------------------------------
void CFeatureTester::Instruction_kFTC_MovePlayerToOtherEntity()
{
	bool useLocalPlayer = GetNextInstructionOrParam()->GetData_kFTVT_Bool();
	IActor *pPlayer = useLocalPlayer ? gEnv->pGame->GetIGameFramework()->GetClientActor() : GetNthNonLocalActor(0);
	const char * className = GetTextParam();
	IEntityClass* classPtr = gEnv->pEntitySystem->GetClassRegistry()->FindClass(className);
	uint32 requireFlags = GetNextInstructionOrParam()->GetData_kFTVT_EntityFlags();
	uint32 skipWithFlags = GetNextInstructionOrParam()->GetData_kFTVT_EntityFlags();

	if (CheckFeatureTestFailure(pPlayer, "There's no %s player", useLocalPlayer ? "local" : "non-local"))
	{
		if (CheckFeatureTestFailure(classPtr, "There's no class by the name of '%s'", className))
		{
			IEntity * foundEntity = NULL;
			IEntityIt * itEntity = gEnv->pEntitySystem->GetEntityIterator();
			if (itEntity)
			{
				itEntity->MoveFirst();
				while (!itEntity ->IsEnd())
				{
					IEntity * pEntity = itEntity->Next();
					if (pEntity && pEntity->GetClass() == classPtr)
					{
						uint32 entityFlags = pEntity->GetFlags();
						bool hasRequiredFlags = (entityFlags & requireFlags) == requireFlags;
						bool hasAnySkipFlags = (entityFlags & skipWithFlags) != 0;
						FeatureTesterSpam("Entity %u: %s '%s' flags=%u%s%s%s (%.2f %.2f %.2f)%s%s", pEntity->GetId(), pEntity->GetClass()->GetName(), pEntity->GetName(), entityFlags, pEntity->IsHidden() ? ", HIDDEN" : "", pEntity->IsInvisible() ? ", INVISIBLE" : "", pEntity->IsActive() ? ", ACTIVE" : "", pEntity->GetWorldPos().x, pEntity->GetWorldPos().y, pEntity->GetWorldPos().z, hasAnySkipFlags ? ", skip" : "", hasRequiredFlags ? ", suitable" : "");
						if (hasRequiredFlags && ! hasAnySkipFlags)
						{
							// TODO: Perhaps if foundEntity != NULL then only set to pEntity if it's closer?
							foundEntity = pEntity;
						}
					}
				}
			}

			if (CheckFeatureTestFailure(foundEntity, "Didn't find an appropriate entity (was looking for class '%s', with flags %u but without flags %u)", className, requireFlags, skipWithFlags))
			{
				char command[512];
				Vec3 moveToPos = foundEntity->GetWorldPos();
				moveToPos.x += 1.f;
				moveToPos.y += 1.f;
				sprintf(command, "%splayerGoto %.2f %.2f %.2f 0 0 135", useLocalPlayer ? "" : "sv_sendConsoleCommand", moveToPos.x, moveToPos.y, moveToPos.z);
				gEnv->pConsole->ExecuteString(command, false, false);
			}
		}
	}
}

//-------------------------------------------------------------------------------
void CFeatureTester::Instruction_kFTC_TrySpawnPlayer()
{
	bool useLocalPlayer = GetNextInstructionOrParam()->GetData_kFTVT_Bool();
	IActor *pPlayer = useLocalPlayer ? gEnv->pGame->GetIGameFramework()->GetClientActor() : GetNthNonLocalActor(0);

	if (CheckFeatureTestFailure(pPlayer, "There's no %s player", useLocalPlayer ? "local" : "non-local"))
	{
		if (!g_pGame->GetGameRules()->IsPlayerActivelyPlaying(pPlayer->GetEntityId(), true))
		{
			IGameRulesSpawningModule *pSpawningModule = g_pGame->GetGameRules() ? g_pGame->GetGameRules()->GetSpawningModule() : NULL;

			if (CheckFeatureTestFailure(pSpawningModule, "No spawning module present"))
			{
				EntityId id = pPlayer->GetEntityId();
				int teamNum = g_pGame->GetGameRules()->GetTeam(id);
				FeatureTesterLog("%s player '%s' is dead - trying to revive it (team %d)", useLocalPlayer ? "Local" : "Non-local", pPlayer->GetEntity()->GetName(), teamNum);
				pSpawningModule->PerformRevive(id, teamNum);
			}
		}
	}
}

//-------------------------------------------------------------------------------
void CFeatureTester::Instruction_kFTC_WaitUntilPlayerIsAlive()
{
	bool useLocalPlayer = GetNextInstructionOrParam()->GetData_kFTVT_Bool();
	IActor *pPlayer = useLocalPlayer ? gEnv->pGame->GetIGameFramework()->GetClientActor() : GetNthNonLocalActor(0);
	
	if (CheckFeatureTestFailure(pPlayer, "There's no %s player", useLocalPlayer ? "local" : "non-local"))
	{
		if (!g_pGame->GetGameRules()->IsPlayerActivelyPlaying(pPlayer->GetEntityId(), true))
		{
			FeatureTesterLog("%s player '%s' is dead - waiting until it is revived", useLocalPlayer ? "Local" : "Non-local", pPlayer->GetEntity()->GetName());
			SetPauseStateAndTimeout(kFTPauseReason_untilPlayerIsAlive, 10.f);
			m_pausedInfo.m_waitUntilPlayerIsAlive_localPlayer = useLocalPlayer;
		}
	}
}

//-------------------------------------------------------------------------------
CFeatureTester::SCheckpointCount * CFeatureTester::WatchCheckpoint(const char * cpName)
{
	SCheckpointCount * watchedCheckpoint = FindWatchedCheckpointDataByName(cpName, m_runFeatureTestStack.m_count);

	if (watchedCheckpoint == NULL)
	{
		if (m_numWatchedCheckpoints < kMaxWatchedCheckpoints)
		{
			watchedCheckpoint = & m_checkpointCountArray[m_numWatchedCheckpoints];
			++ m_numWatchedCheckpoints;

			memset (watchedCheckpoint, 0, sizeof(SCheckpointCount));
			watchedCheckpoint->m_checkpointName = cpName;
			watchedCheckpoint->m_stackLevelAtWhichAdded = m_runFeatureTestStack.m_count;
		}
		else
		{
			FeatureTestFailure("Can't watch checkpoint '%s', already watching %d checkpoints", cpName, kMaxWatchedCheckpoints);
		}
	}

	return watchedCheckpoint;
}

//-------------------------------------------------------------------------------
void CFeatureTester::SetCheckpointHitResponse(SCheckpointCount * checkpoint, EFTCheckpointHitResponse response)
{
	if (response != checkpoint->m_hitResponse)
	{
		FeatureTesterSpam ("Checkpoint %s hit response changing from %s to %s [%u = %s]", checkpoint->m_checkpointName, s_featureTestHitResponseNames[checkpoint->m_hitResponse], s_featureTestHitResponseNames[response], m_waitUntilCCCPointHit_numStillToHit, GetListOfCheckpointsExpected().c_str());

		EFTCheckpointHitResponse oldResponse = checkpoint->m_hitResponse;
		checkpoint->m_hitResponse = kFTCHR_nothing;

		switch (oldResponse)
		{
			case kFTCHR_expectedNext:
			assert (m_waitUntilCCCPointHit_numStillToHit > 0);
			if ((-- m_waitUntilCCCPointHit_numStillToHit) == 0)
			{
				if (m_pause_state == kFTPauseReason_untilCCCPointsHit)
				{
					SetPauseStateAndTimeout(kFTPauseReason_none, 0.f);
				}
			}
			else
			{
				FeatureTesterSpam ("Still waiting for %u (%s)", m_waitUntilCCCPointHit_numStillToHit, GetListOfCheckpointsExpected().c_str());
			}
			break;

			case kFTCHR_restartTest:
			checkpoint->m_restartDelay = 0.f;
			break;
		}
	
		checkpoint->m_hitResponse = response;

		switch (checkpoint->m_hitResponse)
		{
			case kFTCHR_expectedNext:
			++ m_waitUntilCCCPointHit_numStillToHit;
			FeatureTesterSpam ("Ah-ha! Upping number of 'expected next' checkpoints to %d: %s", m_waitUntilCCCPointHit_numStillToHit, GetListOfCheckpointsExpected().c_str());
			break;
		}
	}
}

//-------------------------------------------------------------------------------
bool CFeatureTester::FeatureTestFailureFunc(const char * conditionTxt, const char * messageTxt)
{
	string reply = messageTxt;

	if (m_runFeatureTestStack.m_count > 0)
	{
		reply.append(" [in ");
		reply.append(m_runFeatureTestStack.m_info[0].m_calledTest->m_testName);

		for (int i = 1; i < m_runFeatureTestStack.m_count; ++ i)
		{
			reply.append("=>");
			reply.append(m_runFeatureTestStack.m_info[i].m_calledTest->m_testName);
		}
		reply.append("]");
	}

	if (m_informAutoTesterOfResults == NULL)
	{
		DesignerWarningFail(conditionTxt, string().Format("Feature test '%s' failed: %s", m_currentTest->m_testName, reply.c_str()));
	}
	FeatureTesterLog("Aborting feature test '%s' because it failed: %s", m_currentTest->m_testName, messageTxt);

	StopTest(reply.c_str());
	return false;
}

//-------------------------------------------------------------------------------
void CFeatureTester::Update(float dt)
{
	for (int i = 0; i < m_numOverriddenInputs; ++ i)
	{
		CryWatch ("$7[FEATURETESTER]$o Overridden input '%s' mode=%d", m_currentlyOverriddenInputs[i].m_inputName, m_currentlyOverriddenInputs[i].m_mode);
	}

	if (m_currentTestNextInstruction == NULL)
	{
		if (m_nextIteration.m_test)
		{
			StartTest(m_nextIteration.m_test, "repeat", 0.f, m_nextIteration.m_charOffset);
		}	
		else if (m_numFeatureTestsLeftToAutoRun)
		{
			while (m_numFeatureTestsLeftToAutoRun)
			{
				int i = m_numTests - m_numFeatureTestsLeftToAutoRun;
				SFeatureTest * test = & m_featureTestArray[i];

				FeatureTesterLog ("Test %d/%d = '%s'... %s", i + 1, m_numTests, test->m_testName, test->m_enabled ? (test->m_autoRunThis ? "needs to be run" : "being skipped") : "disabled");

				if (! test->m_autoRunThis)
				{
					-- m_numFeatureTestsLeftToAutoRun;
				}
				else
				{
					if (StartTest(test, "auto-start", 0.f))
					{
					}
					else
					{
						m_timeSinceCurrentTestBegan += dt;
						if (m_timeSinceCurrentTestBegan > test->m_maxTime)
						{
							SubmitResultToAutoTester(test, m_timeSinceCurrentTestBegan, string().Format("Did not start (failed to meet requirements for over %.3f seconds)", test->m_maxTime));
							m_timeSinceCurrentTestBegan = 0.f;
							test->m_autoRunThis = false;
							-- m_numFeatureTestsLeftToAutoRun;
						}
					}
					break;
				}
			}
		}
	}

	if (m_currentTestNextInstruction)
	{
		assert (m_currentTest);

		m_pause_enableCountdown = (m_pause_state != kFTPauseReason_none);
		m_abortUntilNextFrame = false;
		m_timeSinceCurrentTestBegan += dt;

		CheckFeatureTestFailure (m_timeSinceCurrentTestBegan <= m_currentTest->m_maxTime, "Test has taken over %.2f seconds, aborting!", m_currentTest->m_maxTime);

		while (! m_abortUntilNextFrame)
		{
			switch (m_pause_state)
			{
				case kFTPauseReason_none:
				{
					EFeatureTestCommand cmd = GetNextInstructionOrParam()->GetData_kFTVT_Command();

					if (cmd >= 0 && cmd < kFTC_Num)
					{
						(this->*s_instructionFunctions[cmd])();
					}
					else
					{
						FeatureTestFailure("Invalid feature tester command number %u", cmd);
					}
				}
				break;

				case kFTPauseReason_untilWeaponIsReadyToUse:
				{
					CActor * pClientActor = (CActor*)gEnv->pGame->GetIGameFramework()->GetClientActor();
					IItem * currentItem = pClientActor ? pClientActor->GetCurrentItem() : NULL;

					if (currentItem && currentItem->IsBusy() == false)
					{
						SetPauseStateAndTimeout(kFTPauseReason_none, 0.f);
					}
				}
				break;

				case kFTPauseReason_untilPlayerIsAlive:
				{
					IActor *pPlayer = m_pausedInfo.m_waitUntilPlayerIsAlive_localPlayer ? gEnv->pGame->GetIGameFramework()->GetClientActor() : GetNthNonLocalActor(0);

					if (pPlayer && g_pGame->GetGameRules()->IsPlayerActivelyPlaying(pPlayer->GetEntityId(), true))
					{
						SetPauseStateAndTimeout(kFTPauseReason_none, 0.f);
					}

					FeatureTesterSpam("Waiting for player '%s' to be alive... %s", pPlayer ? pPlayer->GetEntity()->GetName() : "NULL", (m_pause_state == kFTPauseReason_none) ? "hooray!" : "hasn't happened yet");
					PauseExecution();
				}
				break;
			}

			if (m_pause_state != kFTPauseReason_none)
			{
				PauseExecution();
			}

			CryWatch ("$7[FEATURETESTER]$o %sWait=%s Time=%.1f/%.1f", GetContextString().c_str(), s_featureTestPauseReasonNames[m_pause_state], m_pause_timeLeft, m_pause_originalTimeOut);

			if (m_pause_enableCountdown)
			{
				if ((m_pause_timeLeft -= dt) <= 0.f)
				{
					switch (m_pause_state)
					{
						case kFTPauseReason_untilTimeHasPassed:
						SetPauseStateAndTimeout(kFTPauseReason_none, 0.f);
						break;

						case kFTPauseReason_untilCCCPointsHit:
						FeatureTestFailure("Timed out! Spent over %.1f seconds waiting to hit %s", m_pause_originalTimeOut, GetListOfCheckpointsExpected().c_str());
						break;

						default:
						FeatureTestFailure("Timed out! Spent over %.1f seconds in '%s' state", m_pause_originalTimeOut, s_featureTestPauseReasonNames[m_pause_state]);
						break;
					}
				}
			}
		}

		for (int i = 0; i < m_numWatchedCheckpoints; ++ i)
		{
			EFTCheckpointHitResponse response = m_checkpointCountArray[i].m_hitResponse;
			CryWatch ("  $%c[%s]$%c %s#%d", response + '2', s_featureTestHitResponseNames[response], (m_checkpointCountArray[i].m_stackLevelAtWhichAdded == m_runFeatureTestStack.m_count) ? '1' : '9', m_checkpointCountArray[i].m_checkpointName, m_checkpointCountArray[i].m_stackLevelAtWhichAdded);
		}
	}
}

//-------------------------------------------------------------------------------
/* static */ void CFeatureTester::CmdStartTest(IConsoleCmdArgs *pArgs)
{
	if (pArgs->GetArgCount() == 2)
	{
		const char * name = pArgs->GetArg(1);

		SFeatureTest * test = s_instance->FindTestByName(name);

		if (test)
		{
			s_instance->StartTest(test, "start", 0.f);
		}
		else
		{
			FeatureTesterLog("No feature test found called \"%s\"", name);
		}
	}
	else
	{
		for (int i = 0; i < s_instance->m_numTests; ++ i)
		{
			char colourChar = s_instance->m_featureTestArray[i].m_enabled ? '5' : '9';
			FeatureTesterLog("$%c%s$o: %s", colourChar, s_instance->m_featureTestArray[i].m_testName, s_instance->m_featureTestArray[i].m_testDescription);
		}
	}
}

//-------------------------------------------------------------------------------
/* static */ void CFeatureTester::CmdReload(IConsoleCmdArgs *pArgs)
{
	if (! s_instance->m_currentlyLoadedFileName.empty())
	{
		s_instance->UnloadTestData();
		s_instance->LoadTestData(s_instance->m_currentlyLoadedFileName.c_str());
	}
	else
	{
		FeatureTesterLog ("Can't reload feature tests - there's nothing currently loaded!");
	}
}

//-------------------------------------------------------------------------------
/* static */ void CFeatureTester::CmdLoad(IConsoleCmdArgs * pArgs)
{
	if (pArgs->GetArgCount() == 2)
	{
		s_instance->UnloadTestData();
		s_instance->LoadTestData(pArgs->GetArg(1));
	}
}

//-------------------------------------------------------------------------------
/* static */ void CFeatureTester::CmdRunAll(IConsoleCmdArgs *pArgs)
{
	if (s_instance->m_numFeatureTestsLeftToAutoRun)
	{
		FeatureTesterLog ("Already auto-running feature tests!");
	}
	else
	{
		if (pArgs->GetArgCount() == 2)
		{
			const char * setNames = pArgs->GetArg(1);
			const char * parseSetNames = setNames;
			char oneSetName[64];
			int total = 0;

			while (parseSetNames)
			{
				size_t skipChars = cry_copyStringUntilFindChar(oneSetName, parseSetNames, sizeof(oneSetName), '+');
				parseSetNames = skipChars ? (parseSetNames + skipChars) : NULL;
				int countInThisSet = 0;

				for (int i = 0; i < s_instance->m_numTests; ++ i)
				{
					SFeatureTest * testData = & s_instance->m_featureTestArray[i];
					if (testData->m_enabled && 0 == stricmp (testData->m_setName, oneSetName))
					{
						testData->m_autoRunThis = true;
						++ countInThisSet;
					}
				}

				FeatureTesterLog ("Number of enabled auto-tests in set '%s' = %d", oneSetName, countInThisSet);
				total += countInThisSet;
			}

			if (total > 0)
			{
				s_instance->m_numFeatureTestsLeftToAutoRun = s_instance->m_numTests;
			}
		}

		if (s_instance->m_numFeatureTestsLeftToAutoRun == 0)
		{
			static const int kMaxSetNamesToList = 32;
			const char * setNamesDisplayed[kMaxSetNamesToList];
			const char * cmdName = pArgs->GetArg(0);
			int numSetNamesDisplayed = 0;
			FeatureTesterLog("Please specify which feature test set(s) to run, e.g.");
			for (int i = 0; i < s_instance->m_numTests; ++ i)
			{
				int j;
				const char * mySetName = s_instance->m_featureTestArray[i].m_setName;

				for (j = 0; j < numSetNamesDisplayed; ++ j)
				{
					if (setNamesDisplayed[j] == mySetName)
					{
						break;
					}
				}

				if (j == numSetNamesDisplayed && numSetNamesDisplayed < kMaxSetNamesToList)
				{
					setNamesDisplayed[j] = mySetName;
					FeatureTesterLog ("  %s %s", cmdName, mySetName);
					++ numSetNamesDisplayed;
				}
			}
			if (numSetNamesDisplayed >= 2)
			{
				FeatureTesterLog ("  %s %s+%s", cmdName, setNamesDisplayed[0], setNamesDisplayed[1]);
			}
		}
	}
}

//-------------------------------------------------------------------------------
CFeatureTester::SFeatureTest * CFeatureTester::FindTestByName(const char * name)
{
	for (int i = 0; i < m_numTests; ++ i)
	{
		if (0 == stricmp(name, m_featureTestArray[i].m_testName))
		{
			return & m_featureTestArray[i];
		}
	}

	return NULL;
}

//-------------------------------------------------------------------------------
void CFeatureTester::UnloadTestData()
{
	if (m_informAutoTesterOfResults && m_currentTest)
	{
		FeatureTestFailure("Did not finish");
	}

	InterruptCurrentTestIfOneIsRunning();

	FeatureTesterLog ("Unloading all feature test data");

	if (m_informAutoTesterOfResults && m_singleBufferContainingAllInstructions)
	{
		while (m_numFeatureTestsLeftToAutoRun)
		{
			SFeatureTest * test = & m_featureTestArray[m_numTests - m_numFeatureTestsLeftToAutoRun];

			if (test->m_autoRunThis)
			{
				SubmitResultToAutoTester(test, -1.f, "Did not start");
				test->m_autoRunThis = false;
			}

			-- m_numFeatureTestsLeftToAutoRun;
		}

		m_informAutoTesterOfResults->Stop();
		m_informAutoTesterOfResults = NULL;
	}

	SAFE_DELETE_ARRAY(m_featureTestArray);
	SAFE_DELETE_ARRAY(m_singleBufferContainingAllInstructions);

	m_numTests = 0;
	m_numFeatureTestsLeftToAutoRun = 0;
	m_timeSinceCurrentTestBegan = 0.f;

	m_singleAllocTextBlock.Reset();
}

//-------------------------------------------------------------------------------
void CFeatureTester::RemoveWatchedCheckpointsAddedAtCurrentStackLevel(const char * reason)
{
	FeatureTesterSpam ("%s %s... removing all checkpoints of level %d [%u = %s]", m_runFeatureTestStack.m_count ? "Subroutine" : "Entire test", reason, m_runFeatureTestStack.m_count, m_waitUntilCCCPointHit_numStillToHit, GetListOfCheckpointsExpected().c_str());

	while (m_numWatchedCheckpoints > 0 && m_checkpointCountArray[m_numWatchedCheckpoints - 1].m_stackLevelAtWhichAdded == m_runFeatureTestStack.m_count)
	{
		SetCheckpointHitResponse(& m_checkpointCountArray[m_numWatchedCheckpoints - 1], kFTCHR_nothing);
		-- m_numWatchedCheckpoints;
		FeatureTesterSpam ("%s %s... no longer watching array[%d] %s#%d", m_runFeatureTestStack.m_count ? "Subroutine" : "Entire test", reason, m_numWatchedCheckpoints, m_checkpointCountArray[m_numWatchedCheckpoints].m_checkpointName, m_runFeatureTestStack.m_count);
	}
}

//-------------------------------------------------------------------------------
bool CFeatureTester::CompleteSubroutine()
{
	RemoveWatchedCheckpointsAddedAtCurrentStackLevel("completed");

	if (m_runFeatureTestStack.m_count)
	{
		SStackedTestCallInfo * info = & m_runFeatureTestStack.m_info[-- m_runFeatureTestStack.m_count];
		m_currentTestNextInstruction = info->m_returnToHereWhenDone;
		FeatureTesterSpam ("Finished using feature test '%s' as a subroutine (wait state was %s)", info->m_calledTest->m_testName, s_featureTestPauseReasonNames[m_pause_state]);
		SetPauseStateAndTimeout(kFTPauseReason_none, 0.f);
		return true;
	}

	StopTest("");
	return false;
}

//-------------------------------------------------------------------------------
void CFeatureTester::SubmitResultToAutoTester(const SFeatureTest * test, float timeTaken, const char * failureMessage)
{
	CryFixedStringT<1024> actuallySendFailureMessage;
	CGameRules * gameRules = g_pGame->GetGameRules();

	if (failureMessage && failureMessage[0])
	{
		actuallySendFailureMessage = failureMessage;

		if (gameRules)
		{
			IActorSystem * pActorSystem = gEnv->pGame->GetIGameFramework()->GetIActorSystem();
			IActorIteratorPtr pIter = pActorSystem->CreateActorIterator();

			while (IActor * iActor = pIter->Next())
			{
				CryFixedStringT<256> singleLine;

				CActor * pActor = (CActor *) iActor;
				IEntity * pEntity = pActor->GetEntity();
				Vec3 pos = pEntity->GetWorldPos();
				const Vec3 & dir = pEntity->GetForwardDir();
				int teamNum = gameRules->GetTeam(pEntity->GetId());
				singleLine.Format("%s '%s' (%s, team=%d '%s', %s%s, health=%d%s) pos=<%.2f %.2f %.2f> dir=<%.2f %.2f %.2f>", pEntity->GetClass()->GetName(), pEntity->GetName(),
					pActor->IsPlayer() ? "player" : "NPC", teamNum, teamNum ? gameRules->GetTeamName(teamNum) : "none", pActor->IsClient() ? "client, " : "", pActor->IsDead() ? "dead" : "alive", pActor->GetHealth(),
					pActor->GetActorStats()->isRagDoll ? ", RAGDOLL" : "", pos.x, pos.y, pos.z, dir.x, dir.y, dir.z);

				FeatureTesterLog("FYI: %s", singleLine.c_str());
				actuallySendFailureMessage.append("\n").append(singleLine);
			}
		}

		if (m_saveScreenshotWhenFail && gEnv->pRenderer && m_informAutoTesterOfResults)
		{
			CryFixedStringT<256> screenShotFileName;
			screenShotFileName.Format("featureTestFailureImages/%s_%s/%s_%s", m_currentlyLoadedFileName.c_str(), m_informAutoTesterOfResults->GetTestName(), test->m_setName, test->m_testName);
			if (gEnv->pRenderer->ScreenShot(screenShotFileName.c_str()))
			{
				actuallySendFailureMessage.append("\nScreenshot saved to: ").append(screenShotFileName);
			}
		}

		failureMessage = actuallySendFailureMessage.c_str();
	}

	if (m_informAutoTesterOfResults)
	{
		string testName(test->m_setName);
		testName.append(": ");
		testName.append(test->m_testName);

		if (test == m_currentTest && m_iterateOverParams.m_numParams)
		{
			for (int i = 0; i < m_iterateOverParams.m_numParams; ++ i)
			{
				testName.append(i ? ", " : " <");
				testName.append(m_iterateOverParams.m_currentParams[i]);
			}
			testName.append(">");
		}

		testName.append(" - ");
		testName.append(test->m_testDescription);

		m_informAutoTesterOfResults->AddSimpleTestCase((string().Format("%s", m_currentlyLoadedFileName.c_str())).c_str(), testName.c_str(), timeTaken, failureMessage);
		m_informAutoTesterOfResults->WriteResults(m_informAutoTesterOfResults->kWriteResultsFlag_unfinished);
	}
}

//-------------------------------------------------------------------------------
void CFeatureTester::StopTest(const char * failureMessage)
{
	CRY_ASSERT_TRACE (m_currentTest, ("Should only call StopTest function when there's a test running!"));
	CRY_ASSERT_TRACE (m_currentTestNextInstruction, ("Should only call StopTest function when there's a test running!"));

	if (failureMessage)
	{
		memset(& m_nextIteration, 0, sizeof(m_nextIteration));

		SubmitResultToAutoTester(m_currentTest, m_timeSinceCurrentTestBegan, failureMessage);

		if (failureMessage[0] == '\0')
		{
			FeatureTesterLog("Test '%s' complete!", m_currentTest->m_testName);
		}

		if (m_iterateOverParams.m_nextIterationCharOffset)
		{
			FeatureTesterLog ("Test '%s' is finished (%s) but has further parameters so will restart soon!", m_currentTest->m_testName, failureMessage[0] ? failureMessage : "OK");
			m_nextIteration.m_test = m_currentTest;
			m_nextIteration.m_charOffset = m_iterateOverParams.m_nextIterationCharOffset;
		}
		else if (m_numFeatureTestsLeftToAutoRun)
		{
			if (m_currentTest == & m_featureTestArray[m_numTests - m_numFeatureTestsLeftToAutoRun])
			{
				FeatureTesterLog ("Finished automatically run test '%s' (%s) so reducing 'num tests left to run' value from %d to %d", m_currentTest->m_testName, failureMessage[0] ? failureMessage : "OK", m_numFeatureTestsLeftToAutoRun, m_numFeatureTestsLeftToAutoRun - 1);

				m_featureTestArray[m_numTests - m_numFeatureTestsLeftToAutoRun].m_autoRunThis = false;
				-- m_numFeatureTestsLeftToAutoRun;
			}
		}

		m_timeSinceCurrentTestBegan = 0.f;
	}

	m_currentTest = NULL;
	m_currentTestNextInstruction = NULL;
	m_numWatchedCheckpoints = 0;
	m_waitUntilCCCPointHit_numStillToHit = 0;
	m_runFeatureTestStack.m_count = 0;
	PauseExecution();

	for (int i = 0; i < m_iterateOverParams.m_numParams; ++ i)
	{
		assert (m_iterateOverParams.m_currentParams[i]);
		SAFE_DELETE_ARRAY (m_iterateOverParams.m_currentParams[i]);
	}
	memset(& m_iterateOverParams, 0, sizeof(m_iterateOverParams));

	CGameCodeCoverageManager::GetInstance()->DisableListener(this);

	// Revert all inputs which are currently overridden by this feature test...
	CPlayer *pPlayer = static_cast<CPlayer *>(gEnv->pGame->GetIGameFramework()->GetClientActor());
	if(pPlayer && pPlayer->GetPlayerInput())
	{
		for (int i = 0; i < m_numOverriddenInputs; ++ i)
		{
			SCurrentlyOverriddenInput & overriddenInput = m_currentlyOverriddenInputs[i];

			EActionActivationMode currentMode = overriddenInput.m_mode;
			FeatureTesterSpam("Test finished while '%s' is still being overridden (mode 0x%x) - reverting it now!", overriddenInput.m_inputName, currentMode);
			CRY_ASSERT_TRACE(currentMode == eAAM_OnPress || currentMode == eAAM_Always, ("Unexpected mode 0x%x", currentMode));

			EActionActivationMode stopMode = (currentMode == eAAM_OnPress) ? eAAM_OnRelease : eAAM_Always;
			const char * releaseName = overriddenInput.m_inputName;
			pPlayer->GetPlayerInput()->OnAction(releaseName, stopMode, 0.f);
		}
	}

	m_numOverriddenInputs = 0;
}

//-------------------------------------------------------------------------------
void CFeatureTester::InterruptCurrentTestIfOneIsRunning()
{
	if (m_currentTest)
	{
		FeatureTesterLog("Test '%s' interrupted!", m_currentTest->m_testName);
		StopTest(NULL);
		assert (m_currentTestNextInstruction == NULL);
	}
}

//-------------------------------------------------------------------------------
bool CFeatureTester::StartTest(const SFeatureTest * test, const char * actionName, float delay, int offsetIntoIterationList)
{
	InterruptCurrentTestIfOneIsRunning();

	assert (test);

	const char * failReason = NULL;
	TBitfield requirements = test->m_requirementBitfield;

	if (requirements & kFTReq_inLevel)
	{
		CGameRules * rules = g_pGame->GetGameRules();
		failReason = rules ? NULL : "there are no game rules set";

		if (failReason == NULL && (requirements & kFTReq_localPlayerExists))
		{
			IActor * pActor = gEnv->pGame->GetIGameFramework()->GetClientActor();
			CPlayer * pPlayer = (pActor && pActor->IsPlayer()) ? (static_cast<CPlayer *>(pActor)) : NULL;
			IPlayerInput * inputs = pPlayer ? pPlayer->GetPlayerInput() : NULL;

			failReason = (pActor == NULL)   ? "there's no local actor" :
			             (pPlayer == NULL)  ? "local actor isn't a player" :
			             (inputs == NULL)   ? "local player has no player input class" :
									 NULL;
		}

		if (failReason == NULL && (requirements & kFTReq_remotePlayerExists))
		{
			IActor * remotePlayer = GetNthNonLocalActor(0);
			failReason = remotePlayer ? NULL : "there's no remote player";
		}
	}
	else if ((requirements & kFTReq_noLevelLoaded) && g_pGame->GetGameRules())
	{
		failReason = "a level is loaded";
	}

	if (failReason == NULL)
	{
		m_currentTest = test;
		PauseExecution();
		assert (m_singleBufferContainingAllInstructions != NULL);
		m_currentTestNextInstruction = m_singleBufferContainingAllInstructions + test->m_offsetIntoInstructionBuffer;

		if (delay > 0.f)
		{
			SetPauseStateAndTimeout(kFTPauseReason_untilTimeHasPassed, delay);
			FeatureTesterLog("Test '%s' will %s in %.1f seconds!", test->m_testName, actionName, delay);
		}
		else
		{
			SetPauseStateAndTimeout(kFTPauseReason_none, 0.f);
			FeatureTesterLog("Test '%s' %sed!", test->m_testName, actionName);
		}

		CGameCodeCoverageManager::GetInstance()->EnableListener(this);

		assert (m_iterateOverParams.m_numParams == 0);
		assert (m_iterateOverParams.m_nextIterationCharOffset == 0);

		if (test->m_iterateOverParams)
		{
			assert (offsetIntoIterationList < strlen(test->m_iterateOverParams));
			const char * paramsStartHere = test->m_iterateOverParams + offsetIntoIterationList;
			char curParams[256];
			size_t readChars = cry_copyStringUntilFindChar(curParams, paramsStartHere, sizeof(curParams), ';');
			FeatureTesterLog("Test '%s' has parameter list '%s', this time using '%s'", test->m_testName, test->m_iterateOverParams, curParams);
			// TODO: split on ',' and support multiple parameters
			{
				size_t paramLength = strlen(curParams) + 1;
				m_iterateOverParams.m_currentParams[m_iterateOverParams.m_numParams] = new char[paramLength];
				cry_strncpy(m_iterateOverParams.m_currentParams[m_iterateOverParams.m_numParams], curParams, paramLength);
				++ m_iterateOverParams.m_numParams;
			}

			if (readChars)
			{
				m_iterateOverParams.m_nextIterationCharOffset = offsetIntoIterationList + readChars;
			}
		}
	}
	else
	{
		FeatureTesterLog("Test '%s' didn't %s because %s", test->m_testName, actionName, failReason);
	}

	return failReason == NULL;
}

//-------------------------------------------------------------------------------
CFeatureTester::SCheckpointCount * CFeatureTester::FindWatchedCheckpointDataByName(const char * name, int stackLevel)
{
	for (int i = 0; i < m_numWatchedCheckpoints; ++ i)
	{
		if ((stackLevel == m_checkpointCountArray[i].m_stackLevelAtWhichAdded) && (0 == stricmp(name, m_checkpointCountArray[i].m_checkpointName)))
		{
			return & m_checkpointCountArray[i];
		}
	}

	return NULL;
}

//-------------------------------------------------------------------------------
string CFeatureTester::GetListOfCheckpointsExpected()
{
	string reply;
	int countFound = 0;

	for (int i = 0; i < m_numWatchedCheckpoints; ++ i)
	{
		if (m_checkpointCountArray[i].m_hitResponse == kFTCHR_expectedNext)
		{
			if (countFound == 0)
			{
				reply = m_checkpointCountArray[i].m_checkpointName;
			}
			else if (countFound == m_waitUntilCCCPointHit_numStillToHit - 1)
			{
				reply.append(" and ");
				reply.append(m_checkpointCountArray[i].m_checkpointName);
			}
			else
			{
				reply.append(", ");
				reply.append(m_checkpointCountArray[i].m_checkpointName);
			}
			++ countFound;
		}
	}

	assert (countFound == m_waitUntilCCCPointHit_numStillToHit);

	return reply.empty() ? "none" : reply;
}

//-------------------------------------------------------------------------------
void CFeatureTester::InformCodeCoverageCheckpointHit(CGameCodeCoverageCheckPoint * cp)
{
	int countNumDealtWith = 0;

	for (int i = 0; i <= m_runFeatureTestStack.m_count; ++ i)
	{
		SCheckpointCount * watchedCheckpoint = FindWatchedCheckpointDataByName(cp->GetLabel(), i);

		assert (this == s_instance);

		if (watchedCheckpoint)
		{
			FeatureTesterLog ("Watched checkpoint '%s#%d' has been hit (response is \"%s\") while state=%s", cp->GetLabel(), i, s_featureTestHitResponseNames[watchedCheckpoint->m_hitResponse], s_featureTestPauseReasonNames[m_pause_state]);
			++ countNumDealtWith;
			++ watchedCheckpoint->m_timesHit;

			float oldTimeLeft = m_pause_timeLeft;
			float oldTotalTime = m_pause_originalTimeOut;
			EFTPauseReason oldPauseReason = m_pause_state;
			EFTCheckpointHitResponse response = watchedCheckpoint->m_hitResponse;
			SetCheckpointHitResponse(watchedCheckpoint, kFTCHR_nothing);

			if (m_pause_state == kFTPauseReason_none && oldPauseReason == kFTPauseReason_untilCCCPointsHit)
			{
				FeatureTesterLog ("Hit all checkpoints with %.1f/%.1f seconds left on the clock", oldTimeLeft, oldTotalTime);
			}

			switch (response)
			{
				case kFTCHR_restartTest:
				StartTest(m_currentTest, "restart", watchedCheckpoint->m_restartDelay);
				break;

				case kFTCHR_restartSubroutine:
				if (i == m_runFeatureTestStack.m_count)
				{
					RemoveWatchedCheckpointsAddedAtCurrentStackLevel("restarting");
					const SFeatureTest * runningSubroutine = m_runFeatureTestStack.m_count ? m_runFeatureTestStack.m_info[m_runFeatureTestStack.m_count - 1].m_calledTest : m_currentTest;
					FeatureTesterSpam ("Running subroutine '%s' (@%u) again from the start...", runningSubroutine->m_testName, runningSubroutine->m_offsetIntoInstructionBuffer);
					m_currentTestNextInstruction = m_singleBufferContainingAllInstructions + runningSubroutine->m_offsetIntoInstructionBuffer;
					SetPauseStateAndTimeout(kFTPauseReason_none, 0.f);
					PauseExecution();
				}
				else
				{
					FeatureTesterWarning("There's currently no support for restarting any subroutine other than the one currently being executed");
				}
				break;

				case kFTCHR_completeTest:
				StopTest("");
				break;

				case kFTCHR_completeSubroutine:
				if (i == m_runFeatureTestStack.m_count)
				{
					CompleteSubroutine();
				}
				else
				{
					FeatureTesterWarning("There's currently no support for completing any subroutine other than the one currently being executed");
				}
				break;

				case kFTCHR_failTest:
				FeatureTestFailure ("Checkpoint '%s#%d' was hit while it was flagged as a reason for failing the test", cp->GetLabel(), i);
				break;

				case kFTCHR_expectedNext:
				break;

				default:
				if (i == m_runFeatureTestStack.m_count)
				{
					CheckFeatureTestFailure (m_pause_state != kFTPauseReason_untilCCCPointsHit, "Checkpoint %s was hit while waiting to hit %s!", cp->GetLabel(), GetListOfCheckpointsExpected().c_str());
				}
				break;
			}
		}
	}

	if (countNumDealtWith == 0)
	{
		FeatureTesterLog ("Unwatched checkpoint '%s' has been hit while state=%s", cp->GetLabel(), s_featureTestPauseReasonNames[m_pause_state]);
	}
}

//-------------------------------------------------------------------------------
IActor * CFeatureTester::GetNthNonLocalActor(int skipThisManyBeforeReturning)
{
	IActorSystem * pActorSystem = gEnv->pGame->GetIGameFramework()->GetIActorSystem();
	IActor * localPlayerActor = gEnv->pGame->GetIGameFramework()->GetClientActor();
	IActorIteratorPtr pIter = pActorSystem->CreateActorIterator();

	while (IActor * pActor = pIter->Next())
	{
		if (pActor != localPlayerActor)
		{
			if (-- skipThisManyBeforeReturning < 0)
			{
				return pActor;
			}
		}
	}

	return NULL;
}

#endif // ENABLE_FEATURE_TESTER
