#ifndef __FEATURETESTER_H__
#define __FEATURETESTER_H__

#include "AutoEnum.h"
#include "GameCodeCoverage/IGameCodeCoverageListener.h"
#include "Utility/SingleAllocTextBlock.h"
#include "GameCodeCoverage/GameCodeCoverageEnabled.h"
#include "GameMechanismManager/GameMechanismBase.h"

#ifdef _RELEASE
	// Final release - don't enable feature tester!
	#define ENABLE_FEATURE_TESTER            0
#else
	// Feel free to turn this on/off or add more conditions (platform, _DEBUG etc.) here...
	#define ENABLE_FEATURE_TESTER            ENABLE_GAME_CODE_COVERAGE
#endif

#if ENABLE_FEATURE_TESTER

struct SFeatureTestInstructionOrParam;
struct SFeatureTestDataLoadWorkspace;
class CAutoTester;

#define FeatureTestPauseReasonList(f)         \
	f(kFTPauseReason_none)                      \
	f(kFTPauseReason_untilTimeHasPassed)        \
	f(kFTPauseReason_untilCCCPointsHit)         \
	f(kFTPauseReason_untilWeaponIsReadyToUse)   \
	f(kFTPauseReason_untilPlayerIsAlive)        \

#define FeatureTestCommandList(f)             \
	f(kFTC_End)                                 \
	f(kFTC_Fail)                                \
	f(kFTC_SetItem)                             \
	f(kFTC_SetSuitMode)                         \
	f(kFTC_Wait)                                \
	f(kFTC_WaitSingleFrame)                     \
	f(kFTC_WaitUntilHitAllExpectedCCCPoints)    \
	f(kFTC_OverrideButtonInput_Press)           \
	f(kFTC_OverrideButtonInput_Release)         \
	f(kFTC_OverrideAnalogInput)                 \
	f(kFTC_WatchCCCPoint)                       \
	f(kFTC_ResetCCCPointHitCounters)            \
	f(kFTC_CheckNumCCCPointHits)                \
	f(kFTC_SetResponseToHittingCCCPoint)        \
	f(kFTC_DoConsoleCommand)                    \
	f(kFTC_DoMenuCommand)                       \
	f(kFTC_RunFeatureTest)                      \
	f(kFTC_TrySpawnPlayer)                      \
	f(kFTC_WaitUntilPlayerIsAlive)              \
	f(kFTC_MovePlayerToOtherEntity)             \

#define FeatureTestRequirementList(f)         \
	f(kFTReq_noLevelLoaded)                     \
	f(kFTReq_inLevel)                           \
	f(kFTReq_localPlayerExists)                 \
	f(kFTReq_remotePlayerExists)                \

#define FeatureTestCheckpointHitResponseList(f)   \
	f(kFTCHR_nothing)                               \
	f(kFTCHR_failTest)                              \
	f(kFTCHR_completeTest)                          \
	f(kFTCHR_completeSubroutine)                    \
	f(kFTCHR_restartTest)                           \
	f(kFTCHR_restartSubroutine)                     \
	f(kFTCHR_expectedNext)                          \

AUTOENUM_BUILDENUMWITHTYPE(EFTPauseReason, FeatureTestPauseReasonList);
AUTOENUM_BUILDENUMWITHTYPE_WITHINVALID_WITHNUM(EFeatureTestCommand, FeatureTestCommandList, kFTC_Invalid, kFTC_Num);
AUTOENUM_BUILDFLAGS_WITHZERO(FeatureTestRequirementList, kFTReq_none);
AUTOENUM_BUILDENUMWITHTYPE_WITHNUM(EFTCheckpointHitResponse, FeatureTestCheckpointHitResponseList, kFTCHR_num);

#define DO_COMMAND_PROTOTYPE(name)        void Instruction_ ## name ();

class CFeatureTester : public CGameMechanismBase, public IGameCodeCoverageListener
{
	friend class CFeatureTestArgumentAutoComplete;

	private:
	static const int kMaxWatchedCheckpoints = 16;
	static const int kMaxSimultaneouslyOverriddenInputs = 8;

	typedef void (CFeatureTester::*InstructionFunc)();

	struct SCheckpointCount
	{
		const char *              m_checkpointName;
		int                       m_timesHit;
		int                       m_stackLevelAtWhichAdded;
		float                     m_restartDelay;
		EFTCheckpointHitResponse  m_hitResponse;
	};

	union UPausedInfo
	{
		struct
		{
			bool m_waitUntilPlayerIsAlive_localPlayer;
		};
	};

	struct SFeatureTest
	{
		const char * m_setName;
		const char * m_testName;
		const char * m_testDescription;
		const char * m_iterateOverParams;
		float m_maxTime;
		TBitfield m_requirementBitfield;
		uint16 m_offsetIntoInstructionBuffer;
		bool m_enabled;
		bool m_autoRunThis;
	};

	struct SStackedTestCallInfo
	{
		const SFeatureTest *                    m_calledTest;
		const SFeatureTestInstructionOrParam *  m_returnToHereWhenDone;
	};

	struct SStack
	{
		static const int          k_stackSize = 3;
		int                       m_count;
		SStackedTestCallInfo      m_info[k_stackSize];
	};

	struct SCurrentlyOverriddenInput
	{
		const char * m_inputName;
		EActionActivationMode m_mode;
	};

	public:
	CFeatureTester();
	~CFeatureTester();
	static string GetContextString();

	ILINE static CFeatureTester * GetInstance()
	{
		return s_instance;
	}

	ILINE bool GetIsActive()
	{
		return m_currentTest || m_numFeatureTestsLeftToAutoRun;
	}

	ILINE void InformAutoTesterOfResults(CAutoTester * autoTester)
	{
		m_informAutoTesterOfResults = autoTester;
	}

	private:
	void Update(float dt);
	int PreprocessTestSet(const IItemParamsNode *testsListNode);
	bool ReadTestSet(const IItemParamsNode *testsListNode, SFeatureTestDataLoadWorkspace * loadWorkspace);
	void LoadTestData(const char * filename);
	void UnloadTestData();
	bool AddInstructionAndParams(EFeatureTestCommand cmd, const IItemParamsNode * paramsNode, SFeatureTestDataLoadWorkspace * loadWorkspace);
	bool StartTest(const SFeatureTest * test, const char * actionName, float delay, int offsetIntoIterationList = 0);
	void InterruptCurrentTestIfOneIsRunning();
	bool CompleteSubroutine();
	void SendInputToLocalPlayer(const char * inputName, EActionActivationMode mode, float value);
	const SFeatureTestInstructionOrParam * GetNextInstructionOrParam();
	SCheckpointCount * WatchCheckpoint(const char * cpName);
	SCheckpointCount * FindWatchedCheckpointDataByName(const char * name, int stackLevel);
	SFeatureTest * FindTestByName(const char * name);
	bool FeatureTestFailureFunc(const char * conditionTxt, const char * messageTxt);
	IActor * GetNthNonLocalActor(int skipThisManyBeforeReturning);
	void SetCheckpointHitResponse(SCheckpointCount * checkpoint, EFTCheckpointHitResponse response);
	void SetPauseStateAndTimeout(EFTPauseReason pauseReason, float timeOut);
	void RemoveWatchedCheckpointsAddedAtCurrentStackLevel(const char * reason);
	void SubmitResultToAutoTester(const SFeatureTest * test, float timeTaken, const char * failureMessage);
	string GetListOfCheckpointsExpected();
	const char * GetTextParam();

	ILINE void PauseExecution(bool pause = true)
	{
		m_abortUntilNextFrame |= (pause == true);
	}

	// StopTest:
	// Pass in NULL to stop quietly, "" to count as a successful run or any other char* ptr for a test failure
	void StopTest(const char * failureMsg);

	// Static functions triggered by console commands...
	static void CmdStartTest(IConsoleCmdArgs *pArgs);
	static void CmdLoad(IConsoleCmdArgs *pArgs);
	static void CmdReload(IConsoleCmdArgs *pArgs);
	static void CmdRunAll(IConsoleCmdArgs *pArgs);

	// IGameCodeCoverageListener
	void InformCodeCoverageCheckpointHit(CGameCodeCoverageCheckPoint * cp);

	// Instructions
	FeatureTestCommandList(DO_COMMAND_PROTOTYPE)

	// Static private data
	static CFeatureTester *                 s_instance;
	static InstructionFunc                  s_instructionFunctions[];

	// Non-static member vars
	CSingleAllocTextBlock                   m_singleAllocTextBlock;
	SCheckpointCount                        m_checkpointCountArray[kMaxWatchedCheckpoints];
	SCurrentlyOverriddenInput               m_currentlyOverriddenInputs[kMaxSimultaneouslyOverriddenInputs];
	UPausedInfo                             m_pausedInfo;
	SStack                                  m_runFeatureTestStack;
	string                                  m_currentlyLoadedFileName;

	const SFeatureTestInstructionOrParam *  m_currentTestNextInstruction;
	SFeatureTest *                          m_featureTestArray;
	const SFeatureTest *                    m_currentTest;
	CAutoTester *                           m_informAutoTesterOfResults;
	SFeatureTestInstructionOrParam *        m_singleBufferContainingAllInstructions;
	EFTPauseReason                          m_pause_state;
	float                                   m_pause_timeLeft;
	float                                   m_pause_originalTimeOut;
	float                                   m_timeSinceCurrentTestBegan;
	int                                     m_numWatchedCheckpoints;
	int                                     m_numOverriddenInputs;
	int                                     m_numTests;
	int                                     m_numFeatureTestsLeftToAutoRun;
	int                                     m_saveScreenshotWhenFail;
	bool                                    m_pause_enableCountdown;
	bool                                    m_abortUntilNextFrame;
	uint8                                   m_waitUntilCCCPointHit_numStillToHit;

	struct
	{
		int                                   m_numParams;
		int                                   m_nextIterationCharOffset;
		char *                                m_currentParams[9];
	} m_iterateOverParams;

	struct
	{
		const SFeatureTest *                  m_test;
		int                                   m_charOffset;
	} m_nextIteration;
};

#endif

#endif //__FEATURETESTER_H__