#ifndef __packetcommon_h__
#define __packetcommon_h__

#pragma once

#include <string>
#include <vector>

namespace Packet
{
	typedef std::string string;
	typedef uint16 PacketType;
	typedef uint32 PacketLenType;
	const uint32 PacketHeaderSize = sizeof(PacketType)+sizeof(PacketLenType);

	//////////////////////////////////////////////////////////////////////////

	template<typename T> inline uint32 SizeT(T& data)
	{
		return (uint32)sizeof(data);
	}

	template<> inline uint32 SizeT<const string>(const string& data)
	{
		return (uint32)(data.size() + sizeof(uint32));
	}

	template<> inline uint32 SizeT<string>(string& data)
	{
		return SizeT((const string&)data);
	}

	template<typename T> inline uint32 SizeVectorT(const std::vector<T>& list)
	{
		uint32 len = sizeof(uint32);
		for(std::vector<T>::const_iterator it = list.begin(); it != list.end(); it++)
			len += SizeT(*it);
		return len;
	}

	template<typename T> inline uint32 SizeVectorT(std::vector<T>& list)
	{
		return SizeVectorT((const std::vector<T>&)list);
	}

	//////////////////////////////////////////////////////////////////////////

	template<typename T> inline bool SerializeT(const T& data, char* buffer, int& offset, int bufferLen)
	{
		memcpy(&buffer[offset], (const char*)&data, SizeT(data));
		offset += SizeT(data);
		return true;
	}

	template<> inline bool SerializeT<string>(const string& data, char* buffer, int& offset, int bufferLen)
	{
		uint32 len = (signed)data.size();
		SerializeT(len, buffer, offset, bufferLen);
		memcpy(&buffer[offset], data.c_str(), len);
		offset += len;
		return true;
	}

	template<typename T> inline bool SerializeVectorT(const std::vector<T>& list, char* buffer, int& offset, int bufferLen)
	{
		uint32 len = (signed)list.size();
		SerializeT(len, buffer, offset, bufferLen);
		for (uint32 i=0; i<len; i++)
			SerializeT(list[i], buffer, offset, bufferLen);
		return true;
	}

	//////////////////////////////////////////////////////////////////////////

	template<typename T> inline bool DeserializeT(T& data, const char* buffer, int& offset, int bufferLen)
	{
		memcpy(&data, &buffer[offset], SizeT(data));
		offset += SizeT(data);
		return true;
	}

	template<> inline bool DeserializeT<string>(string& data, const char* buffer, int& offset, int bufferLen)
	{
		uint32 len = 0;
		DeserializeT(len, buffer, offset, bufferLen);
		data.assign(&buffer[offset], len);
		offset += len;
		return true;
	}

	template<typename T> inline bool DeserializeVectorT(std::vector<T>& list, const char* buffer, int& offset, int bufferLen)
	{
		int leftBufferLen = bufferLen;
		uint32 len = 0;
		DeserializeT(len, buffer, offset, bufferLen);
		for (size_t i=0; i<len; i++)
		{
			if (leftBufferLen <= 0)
				return false;

			T element;
			DeserializeT(element, buffer, offset, bufferLen);
			list.push_back(element);
			leftBufferLen -= SizeT(element);
		}
		return true;
	}
	
	//////////////////////////////////////////////////////////////////////////

	static void WriteHeader(PacketType type, PacketLenType len, char* buffer, int& offset, int bufferLen)
	{
		SerializeT(len, buffer, offset, bufferLen);
		SerializeT(type, buffer, offset, bufferLen);
	}

	static void SkipHeader(const char* buffer, int& offset, int bufferLen)
	{
		PacketLenType dummyLen;
		PacketType dummyType; 
		DeserializeT(dummyLen, buffer, offset, bufferLen);
		DeserializeT(dummyType, buffer, offset, bufferLen);
	}

	static void ReadPacketHeaderWithoutOffsetAdvance(PacketType& type, PacketLenType& len, const char* buffer, int& offset, int bufferLen)
	{
		DeserializeT(len, buffer, offset, bufferLen);
		DeserializeT(type, buffer, offset, bufferLen);
		offset -= PacketHeaderSize;
	}

	static PacketType GetPacketType(const char* buffer, int& offset, int bufferLen)
	{
		PacketType packetType; 
		PacketLenType packetLen;
		ReadPacketHeaderWithoutOffsetAdvance(packetType, packetLen, buffer, offset, bufferLen);
		return packetType;
	}

	static PacketLenType GetPacketLength(const char* buffer, int& offset, int bufferLen)
	{
		PacketType packetType; 
		PacketLenType packetLen;
		ReadPacketHeaderWithoutOffsetAdvance(packetType, packetLen, buffer, offset, bufferLen);
		return packetLen;
	}

} // namespace Packet

#endif // __packetcommon_h__