#include "StdAfx.h"
#include "GameFramework.h"

#include <ICryAnimation.h>
#include <IEntitySystem.h>
#include <IGame.h>
#include <IMovieSystem.h>
#include <ITextModeConsole.h>

#include "LevelSystem.h"
#include "ActorSystem.h"

#ifndef USE_EXTERNAL_FRAMEWORK_DLL

// CrySystem entry point
#ifdef XENON
#define DLL_INITFUNC_SYSTEM (LPCSTR)1
#else
#define DLL_INITFUNC_SYSTEM "CreateSystemInterface"
#endif //XENON

#ifdef _LIB //if in static library
extern "C" 
{
	ISystem* CreateSystemInterface(const SSystemInitParams& initParams );
}
#endif //_LIB

// utility functions
template<typename fnCreate>	
fnCreate GetEntryProc(HMODULE hDll, const char* entryProcName)
{
	CRY_ASSERT(hDll);
	CRY_SAFE_RETURN(!hDll, NULL);

	CRY_ASSERT(entryProcName);
	CRY_SAFE_RETURN(!entryProcName, NULL);

	return (fnCreate)CryGetProcAddress(hDll, entryProcName);
}
//////////////////////


CGameFramework::CGameFramework() :
#if !defined(_LIB) && !defined(LINUX) && !defined(PS3)
	m_hSystemDll(NULL),
#endif	//!_LIB && !LINUX && !PS3
	m_pSystem(NULL),
	m_pLevelSystem(NULL),
	m_pActorSystem(NULL)
{
}

CGameFramework::~CGameFramework()
{
	CRY_ASSERT(gEnv);

	//exit ISystem interfaces
	gEnv->pEntitySystem->Reset();
	gEnv->pMovieSystem->SetUser(NULL);

	SAFE_DELETE(m_pActorSystem);
	SAFE_DELETE(m_pLevelSystem);

	SAFE_RELEASE(m_pSystem);

#if !defined(_LIB) && !defined(LINUX) && !defined(PS3)
	if(m_hSystemDll)
		CryFreeLibrary(m_hSystemDll);
#endif	//!_LIB && !LINUX && !PS3
}

IGameFramework* CGameFramework::CreateInstance()
{
	//creating on static memory to keep compatibility with CCryAction
	//needed for deletion, etc...
	static char spMGFBuffer[sizeof(CGameFramework)];
	return new ((void*)spMGFBuffer)CGameFramework();	
}

bool CGameFramework::Init(SSystemInitParams &startupParams)
{
	if(startupParams.pSystem)
	{
		m_pSystem = startupParams.pSystem;
		if (*startupParams.szUserPath)
			startupParams.pSystem->ChangeUserPath(startupParams.szUserPath);
	}
	else
	{
#if !defined(_LIB) && !defined(LINUX) && !defined(PS3)
		m_hSystemDll = CryLoadLibrary("CrySystem.dll");
		CRY_ASSERT(m_hSystemDll);
		CRY_SAFE_RETURN(!m_hSystemDll, false);

		PFNCREATESYSTEMINTERFACE CreateSystemInterface = 
			GetEntryProc<PFNCREATESYSTEMINTERFACE>(m_hSystemDll, DLL_INITFUNC_SYSTEM);
#endif	//!_LIB && !LINUX && !PS3

		CRY_ASSERT(!m_pSystem);
		CRY_SAFE_CREATE(m_pSystem, CreateSystemInterface(startupParams));
		CRY_ASSERT(m_pSystem);
		CRY_SAFE_RETURN(!m_pSystem, false);
		startupParams.pSystem = m_pSystem;
	}

	ModuleInitISystem(m_pSystem, "GameFramework");	//only in its own dll

	CRY_ASSERT(NULL == m_pLevelSystem);
	m_pLevelSystem = new CLevelSystem;

	CRY_ASSERT(NULL == m_pActorSystem);
	m_pActorSystem = new CActorSystem;

	return true;
}

bool CGameFramework::CompleteInit()
{
	CRY_ASSERT(gEnv);
	CRY_SAFE_RETURN(!gEnv, false);

	CRY_ASSERT(m_pSystem);
	CRY_SAFE_RETURN(!m_pSystem, false);

	//complete game init here
	//WHY HERE?
	CRY_ASSERT(gEnv->pGame);
	CRY_SAFE_RETURN(!gEnv->pGame, false);
	gEnv->pGame->CompleteInit();

	return true;
}

void CGameFramework::Shutdown()
{
	this->~CGameFramework();
}

bool CGameFramework::PreUpdate(const bool haveFocus, const unsigned int updateFlags)
{
	CRY_ASSERT(gEnv);
	CRY_SAFE_RETURN(!gEnv, false);

	CRY_ASSERT(gEnv->pSystem);
	CRY_SAFE_RETURN(!gEnv->pSystem, false);

	CRY_ASSERT(gEnv->pTimer);
	CRY_SAFE_RETURN(!gEnv->pTimer, false);

	CRY_ASSERT(gEnv->pFrameProfileSystem);
	CRY_SAFE_RETURN(!gEnv->pFrameProfileSystem, false);

	CRY_ASSERT(gEnv->pConsole);
	CRY_SAFE_RETURN(!gEnv->pConsole, false);

	ITextModeConsole* pTextModeConsole = gEnv->pSystem->GetITextModeConsole();
	//CRY_ASSERT(pTextModeConsole);
	//CRY_SAFE_RETURN(!pTextModeConsole, false);

	//get frame time
	float	frameTime	= gEnv->pTimer->GetFrameTime();
	bool	bRetRun = true;

	//execute next console command
	// 	if(!m_nextFrameCommand->empty())
	// 	{
	// 		gEnv->pConsole->ExecuteString(*m_nextFrameCommand);
	// 		m_nextFrameCommand->resize(0);
	// 	}

	//draw console	
	if(pTextModeConsole)
		pTextModeConsole->BeginDraw();

	//update frame profiling (when in game mode)
	if (!(updateFlags & ESYSUPDATE_EDITOR))
	{
		gEnv->pFrameProfileSystem->StartFrame();
	}

	//begin rendering
	gEnv->pSystem->RenderBegin();	

	//update system (in game mode)
	if (!(updateFlags & ESYSUPDATE_EDITOR))
	{		
		bRetRun = gEnv->pSystem->Update(updateFlags, 0);
	}

	//update other submodules, regardless of whether in editor or not
	//	CRY_SAFE_PTR_ACCESS(m_pMaterialEffects, Update(frameTime));	
	//	CRY_SAFE_PTR_ACCESS(m_pMusicLogic, Update());	
	//	CRY_SAFE_PTR_ACCESS(m_pMusicGraphState, Update());
	//etc

	return bRetRun;
}

void CGameFramework::PostUpdate(const bool haveFocus, const unsigned int updateFlags)
{
	CRY_ASSERT(gEnv);
	CRY_SAFE_RETURN(!gEnv, CRY_NO_RETURN_VALUE);

	CRY_ASSERT(gEnv->pSystem);
	CRY_SAFE_RETURN(!gEnv->pSystem, CRY_NO_RETURN_VALUE);

	ITextModeConsole* pTextModeConsole = gEnv->pSystem->GetITextModeConsole();
	//CRY_ASSERT(pTextModeConsole);

	//nothing to do in editor mode
	if( updateFlags & ESYSUPDATE_EDITOR_ONLY )
		return;

	//get timer before rendering
	CRY_ASSERT(gEnv->pTimer);	
	float delta = gEnv->pTimer->GetFrameTime();

	//game mode only updates
	if(!(updateFlags & ESYSUPDATE_EDITOR_AI_PHYSICS))
	{
		// syncronize all animations to ensure that their computations have finished	
		CRY_ASSERT(gEnv->pCharacterManager);
		gEnv->pCharacterManager->SyncAllAnimations();
	}	

	//finally render
	gEnv->pSystem->Render();

	//update submodule listeners
	//CALL_FRAMEWORK_LISTENERS(OnPostUpdate(delta));

	//end rendering
	gEnv->pSystem->RenderEnd();

	if (!(updateFlags & ESYSUPDATE_EDITOR))
	{
		CRY_ASSERT(gEnv->pFrameProfileSystem);
		gEnv->pFrameProfileSystem->EndFrame();
	}

	//finish console drawing
	if(pTextModeConsole)
		pTextModeConsole->EndDraw();
}

#endif  // not USE_EXTERNAL_FRAMEWORK_DLL