#include "StdAfx.h"
#include "PlayerManager.h"

#include <IActorSystem.h>
#include <IGameFramework.h>

#include "Game.h"
#include "TestConnection.h"


// CPlayer
CPlayer::CPlayer(const EntityId id) :
	m_entityId(id),
	m_hasMoveTarget(false),
	m_startPoint(0.0f, 0.0f, 0.0f),
	m_moveTarget(0.0f, 0.0f, 0.0f),
	m_hasTargetYaw(false),
	m_targetYawRad(0.0f)
{
}

CPlayer::~CPlayer()
{
}

IEntity* CPlayer::GetEntity()
{
	IActorSystem* pActorSystem = gEnv->pGame->GetIGameFramework()->GetIActorSystem();
	CRY_ASSERT(pActorSystem);
	CRY_SAFE_RETURN(!pActorSystem, NULL);
	IActor* pActor = pActorSystem->GetActor(m_entityId);
	CRY_ASSERT(pActor);
	CRY_SAFE_RETURN(!pActor, NULL);
	return pActor->GetEntity();
}

void CPlayer::Request_SetMoveTarget(const Vec3& currentPos, const Vec3& moveTarget)
{
	//CRY_ASSERT(g_pGame->GetLocalPlayer(), this);

	if (g_pGame->IsConnectedToServer())
	{
		CTestConnection* pConn = g_pGame->GetServerConnection();
		CRY_ASSERT(pConn);
		CRY_SAFE_RETURN(!pConn, CRY_NO_RETURN_VALUE);
		
		char buf[512];
    sprintf_s(buf, 512, "movement %f %f %f %f %f %f",
			currentPos.x, currentPos.y, currentPos.z, moveTarget.x, moveTarget.y, moveTarget.z);
		pConn->Execute(buf);
	}
	else 
	{
		SetMoveTarget(currentPos, moveTarget);
	}
}

void CPlayer::Request_SetYaw(const float yawRad)
{
	SetYaw(yawRad);
}

void CPlayer::SetMoveTarget(const Vec3& currentPos, const Vec3& moveTarget)
{
	m_hasMoveTarget = true;
	m_startPoint = currentPos;
	m_moveTarget = moveTarget;
}

void CPlayer::SetYaw(const float yawRad)
{
		m_hasTargetYaw = true;
		m_targetYawRad = yawRad;
}

bool CPlayer::Update()
{
	IEntity* pEntity = GetEntity();
	CRY_SAFE_RETURN(!pEntity, false);

	if (m_hasMoveTarget)
	{
		Vec3 old = pEntity->GetPos();
		Vec3 dir = m_moveTarget - pEntity->GetPos();
		dir.Normalize();
		dir *= 1.0f;

		pEntity->SetPos(m_moveTarget);
		m_hasMoveTarget = false;

		IPhysicalEntity* pPE = pEntity->GetPhysics();
		ray_hit hit;
		int objTypes = ent_all | ent_no_ondemand_activation;
		int flags = rwi_stop_at_pierceable | rwi_ignore_terrain_holes;
		int col = gEnv->pPhysicalWorld->RayWorldIntersection(pEntity->GetPos()+ Vec3(0.0f, 0.0f, 1.0f), dir, objTypes, flags, &hit, 1, &pPE, 1);
		if (col != 0 && hit.dist > 0)
		{
			pEntity->SetPos(old);
		}
	}

	if (m_hasTargetYaw)
	{
		Quat yaw;
		yaw.SetRotationZ(m_targetYawRad);
		pEntity->SetRotation(yaw);
		m_hasTargetYaw = false;
	}
	return true;
}


// CPlayerManager
CPlayerManager::CPlayerManager()
{
}

CPlayerManager::~CPlayerManager()
{
}

CPlayerPtr CPlayerManager::AddPlayer(const uint id, const char* name, const Vec3& pos)
{
	CPlayerPtr pPlayer = FindPlayer(id);
	if (pPlayer)
		return pPlayer;

	IActorSystem* pActorSystem = gEnv->pGame->GetIGameFramework()->GetIActorSystem();
	EntityId entityId = pActorSystem->CreateActor(name, pos, "MiniActor");
	if (0 == entityId)
		return CPlayerPtr();

	m_playerMap[id].reset(new CPlayer(entityId));
	return m_playerMap[id];
}

CPlayerPtr CPlayerManager::FindPlayer(const uint id)
{
	PlayerMap::iterator it = m_playerMap.find(id);
	if (it == m_playerMap.end())
		return CPlayerPtr();
	return it->second;
}

void CPlayerManager::RemovePlayer(const uint id)
{
	m_playerMap.erase(id);
}

bool CPlayerManager::Update()
{
	for (PlayerMap::iterator it = m_playerMap.begin(); it != m_playerMap.end(); ++it)
		it->second->Update();

	return true;
}