#include "stdafx.h"
#include "NetworkThread.h"
#include <Drei/NetworkFacade.h>
#include <Drei/MessageHeader.h>
#include <Drei/TimerUtil.h>
#include <Drei/SimpleConfig.h>
#include "EchoBackChecker.h"

//////////////////////////
// NetworkThread
//////////////////////////

NetworkThread::NetworkThread(DreiNetwork::SimpleConfig& config) : _configObj(config)
{
	_bufferChecker = new EchoBackChecker();
	_disconnectCycle = new DreiNetwork::CycleChecker();
	_debugMsgCycle = new DreiNetwork::CycleChecker(3000);
	_disconnectCycle->Set(_configObj.GetValue<uint32>("DisconnectTerm"));
	_sendTerm = _configObj.GetValue<uint32>("SendTerm");
}

NetworkThread::~NetworkThread()
{
	SAFE_DELETE(_bufferChecker);
	SAFE_DELETE(_disconnectCycle);
	SAFE_DELETE(_debugMsgCycle);
}

bool NetworkThread::Begin()
{
	return true;
}

void NetworkThread::End()
{
}

void NetworkThread::Update()
{
	_ProcessRecvQueue(_queueId);
}

void NetworkThread::_ProcessRecvQueue(unsigned char queue_id)
{
	using namespace DreiNetwork;
	ACE_Message_Block* commandBlock = NULL;
	if ( false == NetworkInstance->PopMessage(queue_id, &commandBlock, -1))
		return;

	MessagePostee postee;
	memcpy(&postee, commandBlock->rd_ptr(), sizeof(MessagePostee));
	commandBlock->rd_ptr(sizeof(MessagePostee));
	ACE_Message_Block* payloadBlock = commandBlock->cont();
	ACE_ASSERT(NULL != payloadBlock);

	switch(postee.command)
	{
	case eMH_Establish:
		_OnConnect(postee.stream_id, payloadBlock->rd_ptr());
		break;
	case eMH_Read:
		_OnRead(postee.stream_id, payloadBlock->rd_ptr(), payloadBlock->length());
		break;
	case eMH_Close:
		_OnClose(postee.stream_id);
		break;
	}

	commandBlock->release();
}

void NetworkThread::_OnConnect(unsigned int streamID, const char* ip)
{
	//ACE_DEBUG ((LM_DEBUG, "[%t] OnConnect(%d).\n", streamID));
	SendDummyData(streamID);
}

void NetworkThread::_OnRead(unsigned int streamID, char* buffer, size_t length)
{
	//ACE_DEBUG ((LM_DEBUG, "[%t] OnRead(%d, %d).\n", streamID, length));
	if (_DisconnectTest(streamID))
		return;

	_EchoTest(streamID, buffer, length);
	SendDummyData(streamID);
}

bool NetworkThread::_DisconnectTest( unsigned int streamID ) 
{
	if (false == _configObj.GetValue<bool>("DisconnectTest"))
		return false;

	if (_disconnectCycle->IsExpire())
	{
		NetworkInstance->CloseStream(streamID);
		return true;
	}

	return false;
}

void NetworkThread::_EchoTest( unsigned int streamID, char* buffer, size_t length )
{
	if (false == _configObj.GetValue<bool>("EchoTest"))
		return;
	int32 elaspedTick = 0;
	if (false == _bufferChecker->IsValidBuffer(streamID, buffer, (uint32)length, elaspedTick))
		printf("Verify fail. Elasped(%d) ElementCount(%d)\n", elaspedTick, _bufferChecker->GetElementCount());
	else
	{
		if (_debugMsgCycle->IsExpire())
			printf("Verify success. Elasped(%d) ElementCount(%d)\n", elaspedTick, _bufferChecker->GetElementCount());
	}
}

void NetworkThread::_OnClose(unsigned int streamID)
{
}

long NetworkThread::GetCurrentConnection()
{
	DreiNetwork::NetworkFacadeInfo info;
	NetworkInstance->GetInformation(info);
	return info.service_count;
}

void NetworkThread::SetReceiveQueueId( uint8 id )
{
	_queueId = id;
}

void NetworkThread::MakeRandomBuffer( char* buffer, uint32 len )
{
	for(uint32 i=0; i<len; ++i)
		buffer[i] = rand()%255;
}

void NetworkThread::SendDummyData( uint32 streamId )
{
	char buffer[1024] = {0,};
	uint32 sendSize = rand()%1000 + 4;
	MakeRandomBuffer(buffer, sendSize);

	if (_configObj.GetValue<bool>("EchoTest"))
		_bufferChecker->AddSendBuffer(streamId, buffer, sendSize);

	NetworkInstance->SendReserve(streamId, buffer, sendSize, _sendTerm);
}