// TestServer.cpp : Defines the entry point for the console application.
//

#include "stdafx.h"

#include <ace/ACE.h>
#include <Drei/NetworkFacade.h>
#ifdef WIN32
	#include <Drei/ProactorFactory.h>
#else
	#include <Drei/ReactorFactory.h>
#endif
#include <Drei/MessageHeader.h>
#include <Drei/SimpleConfig.h>

#include <Drei/MemoryObject.h>
#include <Drei/TimerUtil.h>

void OnAccept(uint32 streamID, const char* ip);
void OnRead(uint32 streamID, char* buffer, uint32 length);
void OnClose(uint32 streamID);

bool NetworkInit(const DreiNetwork::SimpleConfig& configReader, uint8 queueID)
{
	uint16 port = configReader.GetValue<uint16>("port");
	DreiNetwork::SystemConfig config;
	config.Worker_Thread_Count = configReader.GetValue<uint8>("workerthread");
	config.Receive_Buffer_Len = configReader.GetValue<uint32>("receivebufferlength");
	config.Interval_Send_Term = configReader.GetValue<uint32>("updateterm");;
	//config.Send_Mode = DreiNetwork::eSM_Interval;
	config.Send_Mode = DreiNetwork::eSM_Direct;
	config.ZombieConnectionTerm = 100;
	
	DREI_LOG(ConsoleLogger, ACE_TEXT("Port(%d)"), port);
	DREI_LOG(ConsoleLogger, ACE_TEXT("WorkerThread Count(%d)"), config.Worker_Thread_Count);
	DREI_LOG(ConsoleLogger, ACE_TEXT("Receive buffer length(%d)"), config.Receive_Buffer_Len);
	DREI_LOG(ConsoleLogger, ACE_TEXT("Update term(%d)"), config.Interval_Send_Term);

#ifdef WIN32
	if (false == NetworkInstance->Create<DreiNetwork::ProactorFactory>())
#else
	if (false == NetworkInstance->Create<DreiNetwork::ReactorFactory>())
#endif
		return false;
	if (false == NetworkInstance->AddAcceptor(NULL, port, queueID))
		return false;
	if (false == NetworkInstance->Open(&config))
		return false;

	DREI_LOG(ConsoleLogger, ACE_TEXT("Network start"), 0);
	return true;
}

void NetworkFini()
{
	NetworkInstance->Close();
	NetworkInstance->Destroy();
}

void NetworkUpdate(uint8 queueID)
{
	using namespace DreiNetwork;
	ACE_Message_Block* headBlock = NULL;
	ACE_Message_Block* commandBlock = NULL;

	bool receiveData = NetworkInstance->PopAllMessage(queueID, &headBlock, -1);
	if (false == receiveData)
		return;

	commandBlock = headBlock;
	while(NULL != commandBlock)
	{
		MessagePostee postee;
		memcpy(&postee, commandBlock->rd_ptr(), sizeof(MessagePostee));
		commandBlock->rd_ptr(sizeof(MessagePostee));
		ACE_Message_Block* payloadBlock = commandBlock->cont();

		switch(postee.command)
		{
		case eMH_Establish:
			OnAccept(postee.stream_id, payloadBlock->rd_ptr());
			break;
		case eMH_Read:
			OnRead(postee.stream_id, payloadBlock->rd_ptr(), (unsigned int)payloadBlock->length());
			break;
		case eMH_Close:
			OnClose(postee.stream_id);
			break;
		default:
			break;
		}
		commandBlock = payloadBlock->cont();
	}
	headBlock->release();
}

void OnAccept(uint32 streamID, const char* ip)
{
	DREI_LOG(ConsoleLogger, ACE_TEXT("OnAccept(%d)."), streamID);
}

void OnRead(uint32 streamID, char* buffer, uint32 length)
{
	DREI_LOG (ConsoleLogger, ACE_TEXT("OnRead(%d, %d)."), streamID, length);
	ACE_Message_Block* block = NetworkInstance->AllocateBlock(length);
	block->copy(buffer, length);
	NetworkInstance->SendRequest(streamID, block);
}

void OnClose(uint32 streamID)
{
	//int length = 126;
	//ACE_Message_Block* block = NetworkInstance->AllocateBlock(length);
	//NetworkInstance->SendRequest(streamID, block);
	DREI_LOG(ConsoleLogger, ACE_TEXT("OnClose(%d)."), streamID);
}

int ACE_TMAIN (int, ACE_TCHAR *[])
{
	ACE::init();
	ConsoleLogger->SetConsoleLog();

	const char* ConfigFilename = "TestServerConfig.txt";
	DreiNetwork::SimpleConfig configReader;
	if (false == configReader.ReadFile(ConfigFilename))
	{
		DREI_LOG(ConsoleLogger, ACE_TEXT("Read config fail(%s)"), ConfigFilename);
		return 0;
	}

	const uint8 QueueID = 103;
	if (false == NetworkInit(configReader, QueueID))
	{
		DREI_LOG(ConsoleLogger, ACE_TEXT("NetworkInit Fail."), 0);
		return 0;
	}

	const uint32 DumpTerm = 10000;
	uint32 lastTick = DreiNetwork::GetTick();
	while(true)
	{
		NetworkUpdate(QueueID);
		uint32 currentTick = DreiNetwork::GetTick();
		if ((currentTick-lastTick)>DumpTerm)
		{
			//DREI_DEBUG("Pool.\n", 0);
			//DreiNetwork::MemoryPoolInstance->Dump();
			//DREI_DEBUG("NetworkInstance.\n", 0);
			//NetworkInstance->Dump();
			lastTick = currentTick;
		}
	}

	NetworkFini();
	ACE::fini();
	return 0;
}

