// Testcase.cpp : Defines the entry point for the console application.
//

#include "stdafx.h"
#include <time.h>
#include <ace/ace.h>
#include <ace/Sock_Connector.h>
#include <ace/OS_NS_unistd.h>
#include <Drei/NetworkFacade.h>
#include <Drei/ProactorFactory.h>
#include <Drei/ReactorFactory.h>
#include <Drei/TimerUtil.h>
#include <Drei/MessageBlockUtil.h>
#include <Drei/SimpleConfig.h>
#include "EchoBackServer.h"

class CTestFunctionPrint
{
public:
	CTestFunctionPrint(const char* functionname) : m_functionname(functionname)
	{
		printf("Begin %s\n", m_functionname);
	}

	~CTestFunctionPrint()
	{
		printf("End.. %s\n", m_functionname);
	}

private:
	const char* m_functionname;
};

using namespace DreiNetwork;

class NetworkTestcase
{
protected:
	template<class EventModel>
	void DoEchoTest(ESendMode mode, uint32 port, uint8 queueID)
	{
		CTestFunctionPrint autoPrint(__FUNCTION__);
		// startup
		NetworkInstance->Create<EventModel>();
		NetworkInstance->SetSendMode(mode);
		ACE_TEST_ASSERT(NetworkInstance->AddAcceptor(_T(""), port, queueID));
		ACE_TEST_ASSERT(NetworkInstance->Open());

		// verify information setting
		NetworkFacadeInfo info;
		NetworkInstance->GetInformation(info);
		ACE_TEST_ASSERT(mode == info.send_mode);
		ACE_TEST_ASSERT(GetProperWorkerThreadCount<EventModel>() == info.worker_thread_count);
		//EXPECT_STREQ(GetEventModelName<EventModel>(), info.event_dispatch_model.c_str());

		// same port is not accepted
		ACE_TEST_ASSERT(false == NetworkInstance->AddAcceptor(NULL, port, queueID+1));
		// same Queue_ID is not accepted
		ACE_TEST_ASSERT(false == NetworkInstance->AddAcceptor(NULL, port+1, queueID));

		// get acceptor information
		ACE_Unbounded_Queue<AcceptorInfo>::ITERATOR acceptor_iter(info.acceptor_que);
		AcceptorInfo* acceptor_info = NULL;
		while(false == acceptor_iter.done())
		{
			acceptor_iter.next(acceptor_info);
			if (port == acceptor_info->port)
				break;
			acceptor_iter.advance();
		}

		// verify acceptor information
		ACE_TEST_ASSERT(port == acceptor_info->port);
		ACE_TEST_ASSERT(false == acceptor_info->suspend);
		ACE_TEST_ASSERT(queueID == acceptor_info->queue_id);
		//EXPECT_EQ(0, acceptor_info->ip);

		// echo server startup
		EchoBackServer echoServer(NetworkInstance);
		echoServer.RecvQueueID(queueID);
		ACE_TEST_ASSERT(echoServer.Open(1));

		// Echo client(send -> receive -> verify buffer)
		DoEchoClientTest(_T("127.0.0.1"), port);

		// cleanup
		echoServer.Close();
		NetworkInstance->Close();
		NetworkInstance->Destroy();
	}

	void DoEchoClientTest(const TCHAR* ip, uint32 port)
	{
		ACE_INET_Addr connect_addr(port, ip);
		ACE_SOCK_Connector connector;
		ACE_SOCK_Stream peer;

		// Connect Test
		ACE_TEST_ASSERT(-1 != connector.connect(peer, connect_addr));

		// echo test
		const int send_recv_test_count = 32;
		for (int j=0; j<send_recv_test_count; ++j)
		{
			const size_t buffer_size = 1024;
			char send_buffer[buffer_size] = {0,};
			char recv_buffer[buffer_size] = {0,};
			GenerateRandomBuffer((unsigned char*)send_buffer, buffer_size);
			ACE_TEST_ASSERT( buffer_size == peer.send_n(send_buffer, buffer_size));
			//ACE_Time_Value t(0,1000);
			ACE_TEST_ASSERT( buffer_size == peer.recv_n(recv_buffer, buffer_size));
			// check same buffer
			ACE_TEST_ASSERT(0 == memcmp(send_buffer, recv_buffer, buffer_size));
			//printf("DoEchoClientTest %d\n", j);
		}
		peer.close();
	}

	template<class EventModel>
	uint32 GetProperWorkerThreadCount()
	{
		return 0;
	}

	template<>
	uint32 GetProperWorkerThreadCount<ProactorFactory>()
	{
		return ACE_OS::num_processors_online()*2;
	}

	template<>
	uint32 GetProperWorkerThreadCount<ReactorFactory>()
	{
		return 1;
	}

	template<class EventModel>
	const TCHAR* GetEventModelName()
	{
		return _T("");
	}

	template<>
	const TCHAR* GetEventModelName<ProactorFactory>()
	{
		return _T("Proactor");
	}
	template<>
	const TCHAR* GetEventModelName<ReactorFactory>()
	{
		return _T("Reactor");
	}

	void ProactorTest()
	{
		CTestFunctionPrint autoPrint(__FUNCTION__);
		const int server_port = 35892;
		const int Recv_Queue_ID = 10;
		DoEchoTest<ProactorFactory>(eSM_Interval, server_port, Recv_Queue_ID);
		DoEchoTest<ProactorFactory>(eSM_Direct, server_port, Recv_Queue_ID+1);
	}

	void ReactorTest()
	{
		CTestFunctionPrint autoPrint(__FUNCTION__);
		const int server_port = 45932;
		const int Recv_Queue_ID = 20;
		DoEchoTest<ReactorFactory>(eSM_Interval, server_port, Recv_Queue_ID);
		DoEchoTest<ReactorFactory>(eSM_Direct, server_port, Recv_Queue_ID+1);
	}


private:
	void GenerateRandomBuffer(unsigned char* buffer, int len)
	{
		srand((unsigned int)time(NULL));
		for (int i=0; i<len; ++i)
			buffer[i] = rand()%255+1;
	}

public:
	void Run()
	{
		ProactorTest();
		ReactorTest();
	}
};


class UtilityTestcase
{
protected:
	void MakeRandomBuffer( char* buffer, uint32 len )
	{
		for(uint32 i=0; i<len; ++i)
			buffer[i] = rand();
	}

	void BlockTest()
	{
		CTestFunctionPrint autoPrint(__FUNCTION__);
		const uint32 BlockSize = 32;
		const uint32 BlockCount = 18;
		char buffer[BlockSize] = {0,};

		ACE_Message_Block* blockList = new ACE_Message_Block(BlockSize);
		MakeRandomBuffer(buffer, BlockSize);
		blockList->copy(buffer, BlockSize);	
		for(int i=0; i<BlockCount-1; ++i)
		{
			ACE_Message_Block* tail = FindTailBlock(blockList);
			ACE_TEST_ASSERT(NULL != tail);
			ACE_Message_Block* block = new ACE_Message_Block(BlockSize);
			MakeRandomBuffer(buffer, BlockSize);
			block->copy(buffer, BlockSize);	
			tail->cont(block);
		}

		ACE_TEST_ASSERT(blockList->total_length() == BlockSize*BlockCount);
		ACE_Message_Block* mergedBlock = new ACE_Message_Block(blockList->total_length());
		ACE_TEST_ASSERT(MakeMergedBlock(blockList, mergedBlock));
		ACE_TEST_ASSERT(blockList->total_length() == mergedBlock->length());

		ACE_Message_Block* block = blockList;
		char* mergedBlockPtr = mergedBlock->rd_ptr();
		do 
		{
			char* blockListPtr = block->rd_ptr();		
			for(uint32 i=0; i<block->length(); ++i)
			{
				ACE_TEST_ASSERT(blockListPtr[i] == mergedBlockPtr[i]);
			}
			mergedBlockPtr += block->length();
		} while (NULL != (block = block->cont()));

		ACE_Message_Block* fakeBlock = new ACE_Message_Block(blockList->total_length()-2);
		ACE_TEST_ASSERT(false == MakeMergedBlock(blockList, fakeBlock));

		fakeBlock->release();
		blockList->release();
		mergedBlock->release();
	}

	void SimpleConfigTest()
	{
		CTestFunctionPrint autoPrint(__FUNCTION__);
		std::string targetString = "serial=10;name=alice;level=41;";
		DreiNetwork::SimpleConfig configObj;
		configObj.Parse(targetString);
		uint32 level = configObj.GetValue<uint32>("level");
		ACE_TEST_ASSERT(level == 41);
		uint32 serial = configObj.GetValue<uint32>("serial");
		ACE_TEST_ASSERT(serial == 10);
		std::string name = configObj.GetValue<std::string>("name");
		ACE_TEST_ASSERT(name == "alice");
	}

	void SimpleConfigAdvancedTest()
	{
		CTestFunctionPrint autoPrint(__FUNCTION__);
		std::string targetString = "serial = 10;name=alice ;level  =41;";
		DreiNetwork::SimpleConfig configObj;
		configObj.Parse(targetString);
		uint32 level = configObj.GetValue<uint32>("level");
		ACE_TEST_ASSERT(level == 41);
		uint32 serial = configObj.GetValue<uint32>("serial");
		ACE_TEST_ASSERT(serial == 10);
		std::string name = configObj.GetValue<std::string>("name");
		ACE_TEST_ASSERT(name == "alice");
	}

public:
	void Run()
	{
		BlockTest();
		SimpleConfigTest();
		SimpleConfigAdvancedTest();
	}
};


int _tmain(int argc, _TCHAR* argv[])
{
	srand((unsigned int)time(NULL));
	ACE::init();

	NetworkTestcase networkTestcase;
	networkTestcase.Run();
	UtilityTestcase utilTestcase;
	utilTestcase.Run();

	printf("DreiNetwork test success.\n");
	ACE::fini();
	getchar();
	return 0;
}

