///////////////////////////////////////////////////////////////////////////
//
//  Crytek Engine Source File.
//  Copyright (C), Crytek Studios, 2002.
// -------------------------------------------------------------------------
//  File name:   brush.cpp
//  Version:     v1.00
//  Created:     8/7/2002 by Timur.
//  Compilers:   Visual Studio.NET
//  Description: 
// -------------------------------------------------------------------------
//  History: Based on Andrey's Indoor editor.
//	26/02/2010 Refactored by Jaesik.
//	09/04/2010 Implement a part of creating cone, cylinder, sphere are 
//						 added by Jaesik
//	12/04/2010 Implement merging by Jaesik.
//	21/04/2010 Implement CSG by Jaesik.
////////////////////////////////////////////////////////////////////////////

#include "StdAfx.h"
#include <IIndexedMesh.h>
#include "Brush.h"
#include "BrushFace.h"
#include "Objects\SubObjSelection.h"
#include "SolidBrushObject.h"
#include "DisplaySettings.h"
#include "Grid.h"
#include "BrushPrimitive.h"
#include "BrushSerialize.h"

const float SBrush::VER_EPS							= 0.01f;
const float SBrush::VER_TIGHTEPS				= 0.0001f;
const float SBrush::BASIS_COPLANARPLANE	= 0.99999f;

const char* SBrush::CSGOperationString[3] = { "Union", "Intersection", "Difference" };

class CUndoBrush : public IUndoObject
{
public:
	CUndoBrush( SBrush* pBrush,const char *undoDescription )
	{
		assert( pBrush != 0 );
		if (undoDescription)
			m_undoDescription = undoDescription;
		else
			m_undoDescription = "SolidBrushUndo";
		m_undo = *pBrush;
		m_pBrush = pBrush;
	}

protected:

	virtual int GetSize()
	{
		return m_undo.GetMemorySize() + m_redo.GetMemorySize() + sizeof(CString);
	}

	virtual const char* GetDescription()
	{ 
		return m_undoDescription;
	};

	virtual void Undo( bool bUndo )
	{
		if (bUndo)
			m_redo = *m_pBrush;
		*m_pBrush = m_undo;
		m_pBrush->UpdateMesh();
	}

	virtual void Redo()
	{
		*m_pBrush = m_redo;
		m_pBrush->UpdateMesh();
	}

private:

	CString m_undoDescription;
	_smart_ptr<SBrush> m_pBrush;
	SBrush m_undo;
	SBrush m_redo;
};


CEdGeometry* SBrush::Clone()
{
	SBrush* brush = new SBrush;	
	*brush = *this;
	brush->BuildBrush(false,false);
	return (CEdGeometry*)brush;
}

SBrush* SBrush::CreateBrush( CSolidBrushObject* obj )
{
	SBrush* brush = new SBrush;
	brush->SetOwner(obj);
	return brush;
}

bool SBrushSubSelection::AddPoint( Vec3 *pnt )
{
	if (std::find(points.begin(),points.end(),pnt) != points.end())
		return false;

	points.push_back(pnt);
	return true;
};


void SBrushSubSelection::Clear()
{
	points.clear();
}

SBrush::SBrush()
{
	m_nFlags = 0;
	m_pStatObj = 0;	
	m_Owner = NULL;

	m_bounds.Reset();

	m_selectionType = 0;
	m_isZeroTolerance = false;
}


SBrush::~SBrush()
{
	ClearFaces();
}


SBrush::SBrush( const SBrush& b )
{
	operator =( (const SBrush&)b );
}


// This operator is necessary for every single operation 
// because Sandbox is constantly doing an operation and undoing it whenever 
// you're editing an asset. And in order to do so, it creates and destroys 
// brushes all the time. 
// In order to keep information between creations and destructions, this  
// operator = is the key. Notice that this same operator is already called 
// as the copy constructor. 
SBrush& SBrush::operator = (const SBrush& b)
{
	ClearFaces();

	for( size_t i = 0; i < b.m_BrushVertices.size(); ++i )
	{
		m_BrushVertices.push_back( new SBrushVertex(*b.m_BrushVertices[i]) );
	}

	for( size_t i = 0; i < b.m_BrushFaces.size(); ++i )
	{
		m_BrushFaces.push_back( new SBrushFace(*b.m_BrushFaces[i]) );
	}

	for( size_t i = 0; i < b.m_BrushTriangles.size(); ++i )
	{
		m_BrushTriangles.push_back( new SBrushTriangle(*b.m_BrushTriangles[i]) );
	}

	m_bounds = b.GetBoundBox();
	m_nFlags = b.GetFlags();
	m_isZeroTolerance = b.GetZeroTolerance();

	return *this;
}


void SBrush::RecordUndo( const char *sUndoDescription )
{	
	if (CUndo::IsRecording())
	{
		CUndo::Record( new CUndoBrush(this,sUndoDescription) );
	}
}


void SBrush::Serialize( XmlNodeRef &xmlNode, bool bLoading, bool bUndo )
{
	CBrushSerialize Serializer;
	Serializer.Serialize( this, xmlNode, bLoading, bUndo );
}

bool SBrush::HitTest( HitContext &hit )
{
	if( !( (m_nFlags & BRF_SUB_OBJ_SEL) && GetEditModeFlag() ) )
	{	
		float dist = 0;
		SBrushFace* face = RayHit( hit.raySrc,hit.rayDir,&dist );
		if (face)
		{
			hit.dist = dist;
			return true;
		}
	}

	// OS - Object Space, WS - World Space
	const Matrix34 &worldTM = m_matrix;
	Matrix34 invWorldTM		= worldTM.GetInverted();
	Vec3 vOSCameraPos			= invWorldTM.TransformPoint( hit.view->GetViewTM().GetTranslation() );
	Vec3 vWSCameraVector	= worldTM.GetTranslation() - hit.view->GetViewTM().GetTranslation();
	Vec3 vOSCameraVector	= invWorldTM.TransformVector(vWSCameraVector).GetNormalized();

	bool bSelectValue = !(hit.nSubObjFlags & SO_HIT_SELECT_REMOVE);
	bool bSelChanged = false;

	IUndoObject* pUndoObject = NULL;
	if( CUndo::IsRecording() && (hit.nSubObjFlags&SO_HIT_SELECT) )
	{
		pUndoObject = new CUndoBrush( this, "Brush, HitTest"  );
	}

	if ( (hit.nSubObjFlags & SO_HIT_SELECT) && 
			!(hit.nSubObjFlags & SO_HIT_SELECT_ADD) && 
			!(hit.nSubObjFlags & SO_HIT_SELECT_REMOVE) )
	{
		bSelChanged = ClearSelection();
	}

	if (m_selectionType == SO_ELEM_VERTEX)
	{
		bSelChanged = SelectVertex( hit, vOSCameraPos, worldTM );
	}
	else if (m_selectionType == SO_ELEM_EDGE)
	{
		bSelChanged = SelectEdge( hit, vOSCameraVector, worldTM );
	}
	else if (m_selectionType == SO_ELEM_FACE || m_selectionType == SO_ELEM_POLYGON)
	{
		bSelChanged = SelectFace( hit, vOSCameraVector, worldTM );
	}

	if( pUndoObject && bSelChanged )
	{
		CUndo::Record( pUndoObject );
	}
	else
	{
		if( pUndoObject )
			pUndoObject->Release();
	}

	return bSelChanged;
}


bool SBrush::SelectVertex( HitContext& hit, const Vec3& osCameraPos, const Matrix34& worldTM )
{
	bool bSelChanged = false;
	bool bSelectValue = !(hit.nSubObjFlags & SO_HIT_SELECT_REMOVE);

	for( size_t i = 0; i < m_BrushFaces.size(); ++i)
	{
		SBrushFace *face = m_BrushFaces[i];

		for( size_t k = 0; k < face->pointindexlist.size(); ++k )
		{
			SBrushVertex* vert = m_BrushVertices[face->pointindexlist[k]];

			if (g_SubObjSelOptions.bIgnoreBackfacing && (vert->pos-osCameraPos).dot(face->plane.normal) > 0)
				continue;

			CPoint p = hit.view->WorldToView( worldTM.TransformPoint(vert->pos) );
			if (p.x >= hit.rect.left && p.x <= hit.rect.right && p.y >= hit.rect.top && p.y <= hit.rect.bottom)
			{
				if( !( hit.nSubObjFlags & SO_HIT_SELECT ) )
				{
					if ( !(hit.nSubObjFlags & SO_HIT_TEST_SELECTED) )
						return true;
					else if (vert->bSelected)
						return true;
				}
				else
				{
					if( vert->bSelected != bSelectValue )
					{
						bSelChanged = true;
						vert->bSelected = bSelectValue;
					}
				}
			}
		}
	}
	return bSelChanged;
}


bool SBrush::SelectEdge( HitContext& hit, const Vec3& osCameraDir, const Matrix34& worldTM )
{
	bool bSelChanged = false;
	bool bSelectValue = !(hit.nSubObjFlags & SO_HIT_SELECT_REMOVE);

	for( size_t k = 0; k < m_BrushFaces.size(); ++k )
	{
		SBrushFace * f = m_BrushFaces[k];
		int edgesize = GetNumberOfFaceEdges(f);

		if( g_SubObjSelOptions.bIgnoreBackfacing && osCameraDir.Dot(f->plane.normal) > 0 )
		{
			continue;
		}

		for( int i = 0; i < edgesize; ++i )	
		{
			int firstVtxIdx = 0;
			int secondVtxIdx = 0;

			f->MapEdgeIndexToPolyIndices( m_BrushVertices, m_BrushTriangles, i, firstVtxIdx, secondVtxIdx );

			const SBrushVertex* v0 = m_BrushVertices[f->pointindexlist[firstVtxIdx]];
			const SBrushVertex* v1 = m_BrushVertices[f->pointindexlist[secondVtxIdx]];

			Vec3 wv0 = worldTM.TransformPoint(v0->pos);
			Vec3 wv1 = worldTM.TransformPoint(v1->pos);

			if( !(hit.nSubObjFlags & SO_HIT_SELECT) )
			{
				if( !(hit.nSubObjFlags & SO_HIT_TEST_SELECTED) )
				{
					return hit.view->HitTestLine( wv0, wv1, hit.point2d, 5 );
				}
				else if( f->selectededge&(1<<i))
				{
					return hit.view->HitTestLine( wv0, wv1, hit.point2d, 5 );
				}
			} 
			else if( hit.nSubObjFlags & SO_HIT_POINT )
			{
				if( hit.view->HitTestLine( wv0, wv1, hit.point2d, 5 ) )
				{
					bool bSelected = (f->selectededge&(1<<i)) ? true : false;
					if( bSelected != bSelectValue )
					{
						bSelChanged = true;
						f->selectededge |= (1<<i);
						m_BrushVertices[f->pointindexlist[firstVtxIdx]]->bSelected	= true;
						m_BrushVertices[f->pointindexlist[secondVtxIdx]]->bSelected	= true;
					}
				}
			}
			else
			{
				Vec3 mid = wv0 + ( wv1 - wv0 ) * 0.5f;
				CPoint stViewPoint = hit.view->WorldToView(mid);

				if( stViewPoint.x >= hit.rect.left && 
						stViewPoint.x <= hit.rect.right && 
						stViewPoint.y >= hit.rect.top && 
						stViewPoint.y <= hit.rect.bottom )
				{
					bool bSelected = (f->selectededge&(1<<i)) ? true : false;
					if( bSelected != bSelectValue )
					{
						bSelChanged = true;
						f->selectededge |= (1<<i);
						m_BrushVertices[firstVtxIdx]->bSelected		= true;
						m_BrushVertices[secondVtxIdx]->bSelected	= true;
					}
				}
			}
		}
	}

	return bSelChanged;
}


bool SBrush::SelectFace( HitContext& hit, const Vec3& osCameraDir, const Matrix34& worldTM )
{
	bool bSelectValue = !(hit.nSubObjFlags & SO_HIT_SELECT_REMOVE);
	SBrushFace * pHitFace = RayHit( hit.raySrc, hit.rayDir, &hit.dist );

	if( !(hit.nSubObjFlags&SO_HIT_SELECT) )
	{
		if( pHitFace )
		{
			if (!(hit.nSubObjFlags & SO_HIT_TEST_SELECTED) )
				return true;
			else if (pHitFace->bSelected)
				return true;
		}
	}
	else if( hit.nSubObjFlags & SO_HIT_POINT )
	{
		if( pHitFace && pHitFace->bSelected != bSelectValue )
		{
			pHitFace->bSelected = bSelectValue;
			return true;
		}
	}
	else
	{
		for (int i = 0; i < m_BrushFaces.size(); ++i)
		{
			SBrushFace * face = m_BrushFaces[i];

			if (g_SubObjSelOptions.bIgnoreBackfacing && osCameraDir.Dot(face->plane.normal) > 0)
				continue; 

			face->CalcCenter(m_BrushVertices);
			CPoint p = hit.view->WorldToView( worldTM.TransformPoint(face->center) );
			if( p.x >= hit.rect.left && p.x <= hit.rect.right && p.y >= hit.rect.top && p.y <= hit.rect.bottom &&
				face->bSelected != bSelectValue )
			{
				face->bSelected = bSelectValue;
				return true;
			}
		}
	}

	return false;
}


void SBrush::ModifySelection( SSubObjSelectionModifyContext &modCtx, bool isUndo )
{
	if( !m_isZeroTolerance && ( m_selectionType == SO_ELEM_VERTEX || m_selectionType == SO_ELEM_EDGE || m_selectionType == SO_ELEM_FACE))
		m_isZeroTolerance = true;

	Matrix34 worldTM = m_matrix;
	Matrix34 invTM = worldTM.GetInverted();
	Matrix34 modRefFrame = invTM * modCtx.worldRefFrame;
	Matrix34 modRefFrameInverse = modCtx.worldRefFrame.GetInverted() * worldTM;

	if (modCtx.type == SO_MODIFY_MOVE)
	{
		Vec3 vOffset = modCtx.vValue;
		vOffset = modCtx.worldRefFrame.GetInverted().TransformVector(vOffset);
		Matrix34 tm = modRefFrame * Matrix34::CreateTranslationMat(vOffset) * modRefFrameInverse;

		if( m_selectionType == SO_ELEM_VERTEX )
		{
			for( size_t i = 0; i < m_BrushVertices.size(); ++i )
			{
				SBrushVertex* v = m_BrushVertices[i];
				if( v->bSelected == true )
				{
					v->pos = tm.TransformPoint( v->pos );
					v->pos = invTM * modCtx.view->SnapToGrid( worldTM * v->pos );
				}
			}

			ComputeFacePlanes();
			ComputeTexCoord();
			ComputeBoundBox();
			Invalidate();
		}
		else if( m_selectionType == SO_ELEM_EDGE )
		{
			std::set<short> vlist;

			for( size_t i = 0; i < m_BrushVertices.size(); ++i )
			{
				SBrushVertex* vi = m_BrushVertices[i];

				if( vi->bSelected == false )
					continue;
				
				for( size_t k = 0; k < m_BrushVertices.size(); ++k )
				{
					SBrushVertex* vk = m_BrushVertices[k];

					if( i == k )
						continue;

					if( vk->bSelected == true )
						continue;

					if( vi->pos.IsEquivalent(vk->pos,VER_EPS) )
						vlist.insert(k);
				}
			}

			for( size_t i = 0; i < m_BrushVertices.size(); ++i )
			{
				SBrushVertex* v = m_BrushVertices[i];
				if( v->bSelected == true )
				{
					v->pos = tm.TransformPoint( v->pos );
					v->pos = invTM * modCtx.view->SnapToGrid( worldTM * v->pos );
				}
			}

			std::set<short>::iterator ii = vlist.begin();
			for( ; ii != vlist.end(); ++ii )
			{
				SBrushVertex* v = m_BrushVertices[*ii];
				v->pos = tm.TransformPoint( v->pos );
				v->pos = invTM * modCtx.view->SnapToGrid( worldTM * v->pos );
			}

			ComputeFacePlanes();
			ComputeTexCoord();
			ComputeBoundBox();
			Invalidate();
		}
		else if( m_selectionType == SO_ELEM_FACE || m_selectionType == SO_ELEM_POLYGON )
		{
			for(size_t i = 0; i < m_BrushFaces.size(); ++i)
			{
				SBrushFace * face = m_BrushFaces[i];
				if( face->bSelected )
				{
					std::vector<Vec3*> ptlist;

					// I think this part should need to be optimized using KD-Tree or OCTree
					for( size_t j = 0; j < m_BrushVertices.size(); ++j )
					{
						SBrushVertex* v = m_BrushVertices[j];
						
						for( size_t k = 0; k < face->pointindexlist.size(); ++k )
						{
							if( std::find( face->pointindexlist.begin(), face->pointindexlist.end(), j ) != face->pointindexlist.end() )
								continue;
							if( m_BrushVertices[face->pointindexlist[k]]->pos.IsEquivalent( v->pos ) )
							{
								ptlist.push_back(&v->pos);
							}
						}
					}

					for( size_t j = 0; j < face->pointindexlist.size(); ++j )
					{
						m_BrushVertices[face->pointindexlist[j]]->pos = tm.TransformPoint( m_BrushVertices[face->pointindexlist[j]]->pos );
					}

					for( size_t j = 0; j < ptlist.size(); ++j )
					{
						*ptlist[j] = tm.TransformPoint( *ptlist[j] );
					}
				}
			}

			ComputeFacePlanes();
			ComputeTexCoord();
			ComputeBoundBox();
			Invalidate();
		}
	}

	SetModified();
}



Vec3 SBrush::CalcNormal( const Vec3& v0, const Vec3& v1, const Vec3& v2 ) const
{
	Vec3 n = (v2-v1)^(v0-v1);
	n.Normalize();
	return n;
}


bool SBrush::BuildBrush(	bool bComputeAdjacentTriangle, 
													bool bComputeCoplanarTriangle, 
													bool bComputeTexCoord, 
													bool bComputeBoundBox )
{
	Invalidate();

	if( bComputeAdjacentTriangle )
	{
		ComputeAdjacentTriangle();
		if( bComputeCoplanarTriangle )
			ComputeCoplanarTriangle();
	}

	if( bComputeTexCoord )
		ComputeTexCoord();

	if( bComputeBoundBox )
		ComputeBoundBox();

	return true;
}



void SBrush::RecalcTexCoords()
{
	Invalidate();

	for( size_t i = 0; i < m_BrushFaces.size(); ++i)
	{
		SBrushFace * f = m_BrushFaces[i];
		int numPts = f->pointindexlist.size();
		for( int j = 0; j < numPts; ++j )
		{
			f->CalcTexCoords( *m_BrushVertices[f->pointindexlist[j]] );
		}
	}
}


void SBrush::ComputeTexCoord()
{
	std::vector<SBrushFace*>::iterator ibf = m_BrushFaces.begin();
	for( ; ibf != m_BrushFaces.end(); ++ibf)
	{
		SBrushFace *f = *ibf;
		for( size_t i=0; i < f->pointindexlist.size(); ++i )
			f->CalcTexCoords( *m_BrushVertices[f->pointindexlist[i]] );
	}
}


bool SBrush::ComputeBoundBox()
{
	AABB box;
	box.Reset();

	std::vector<SBrushVertex*>::iterator ibv = m_BrushVertices.begin();
	for( ; ibv != m_BrushVertices.end(); ++ibv )
	{
		SBrushVertex *v = *ibv;
		box.Add( v->pos );
	}

	if(	box.min.x > box.max.x ||
		box.min.y > box.max.y ||
		box.min.z > box.max.z)
	{
		m_bounds.min = m_bounds.max = Vec3(0,0,0);
		return false;
	}

	m_bounds = box;

	if( m_Owner )
		m_Owner->UpdateBoundBox();

	return true;
}


void SBrush::ComputeAdjacentTriangle()
{
	const int NumberOfTriangleEdge = 3;
	const int NumberOfEdgeVertex = 2;

	size_t trisize = m_BrushTriangles.size();

	for( size_t i = 0; i < trisize; ++i )
	{
		SBrushTriangle* basetri = m_BrushTriangles[i];

		Vec3 baseedge[NumberOfTriangleEdge][NumberOfEdgeVertex] = 
								{	{ m_BrushVertices[basetri->vertexindices[0]]->pos, m_BrushVertices[basetri->vertexindices[1]]->pos },
									{ m_BrushVertices[basetri->vertexindices[1]]->pos, m_BrushVertices[basetri->vertexindices[2]]->pos },
									{ m_BrushVertices[basetri->vertexindices[2]]->pos, m_BrushVertices[basetri->vertexindices[0]]->pos } };

		for( int a = 0; a < NumberOfTriangleEdge; ++a )
		{
			for( int k = 0; k < trisize; ++k )
			{
				if( i == k )
					continue;

				SBrushTriangle* othertri = m_BrushTriangles[k];

				Vec3 otheredge[NumberOfTriangleEdge][NumberOfEdgeVertex] = 
										{{ m_BrushVertices[othertri->vertexindices[1]]->pos, m_BrushVertices[othertri->vertexindices[0]]->pos },
										 { m_BrushVertices[othertri->vertexindices[2]]->pos, m_BrushVertices[othertri->vertexindices[1]]->pos },
										 { m_BrushVertices[othertri->vertexindices[0]]->pos, m_BrushVertices[othertri->vertexindices[2]]->pos } };

				bool bFindEdge = false;

				for( int b = 0; b < NumberOfTriangleEdge; ++b )
				{
					if( baseedge[a][0].IsEquivalent(otheredge[b][0],VER_EPS) && baseedge[a][1].IsEquivalent(otheredge[b][1],VER_EPS) )
					{
						basetri->adjacentfidx[a] = k;
						bFindEdge = true;
						break;
					}
				}

				if( bFindEdge == true )
					break;
			}
		}
	}
}


void SBrush::ComputeCoplanarTriangle()
{
	size_t trisize = m_BrushTriangles.size();

	for( int i = 0; i < trisize; ++i )
	{
		m_BrushTriangles[i]->faceidx = -1;
	}

	std::vector<SBrushFace*>::iterator ibf = m_BrushFaces.begin();
	for( ; ibf != m_BrushFaces.end(); ++ibf )
	{
		delete *ibf;
	}
	m_BrushFaces.clear();

	for( int i = 0; i < trisize; ++i )
	{
		if( m_BrushTriangles[i]->faceidx != -1 )
			continue;

		SBrushTriangle* tri = m_BrushTriangles[i];
		tri->faceidx = (short)m_BrushFaces.size();

		SBrushFace* face = new SBrushFace;

		face->pointindexlist.push_back(tri->vertexindices[0]);
		face->pointindexlist.push_back(tri->vertexindices[1]);
		face->pointindexlist.push_back(tri->vertexindices[2]);
		face->triangleidxlist.push_back(i);

		ComputeAdjacentCoplanar(i,face,tri->faceidx);
		
		m_BrushFaces.push_back(face);
	}

	ComputeFacePlanes();
}


void SBrush::ComputeAdjacentCoplanar( int current, SBrushFace* face, short faceidx )
{	
	SBrushTriangle* currentt = m_BrushTriangles[current];

	for( int i = 0; i < 3; ++i )
	{
		short adjacent = currentt->adjacentfidx[i];

		if( adjacent == -1 )
			continue;

		SBrushTriangle* adjacentt = m_BrushTriangles[adjacent];

		if( adjacentt->faceidx != -1 )
			continue;

		float difference = currentt->normal.Dot(adjacentt->normal);
		if( difference >= BASIS_COPLANARPLANE )
		{
			face->triangleidxlist.push_back(adjacent);

			for( int k = 0; k < 3; ++k )
			{
				if( adjacentt->vertexindices[k] != currentt->vertexindices[i] && 
						adjacentt->vertexindices[k] != currentt->vertexindices[(i+1)%3] )
				{
					std::vector<short>::iterator ip = std::find(	face->pointindexlist.begin(), 
																												face->pointindexlist.end(), 
																												currentt->vertexindices[i] );
					face->pointindexlist.insert( ip+1, adjacentt->vertexindices[k] );
				}
			}
			adjacentt->faceidx = faceidx;
			ComputeAdjacentCoplanar(adjacent,face,faceidx);
		}
	}
}


void SBrush::Move( Vec3& delta )
{
	size_t vsize = m_BrushVertices.size();
	SBrush backup = *this;

	for( int i = 0; i < vsize; ++i )
	{
		SBrushVertex* v = m_BrushVertices[i];
		v->pos += delta;
	}

	if( IsValid() == false )
	{
		*this = backup;
	}
	else
	{
		Invalidate();
	}
}


void SBrush::Transform( const Matrix34 &tm, bool bUndo )
{
	if( bUndo )
		RecordUndo( "Brush Transform");

	size_t vsize = m_BrushVertices.size();	
	for( int i = 0; i < vsize; ++i )
	{
		SBrushVertex* v = m_BrushVertices[i];
		v->pos = tm.TransformPoint(v->pos);
	}

	ComputeFacePlanes();
	ComputeBoundBox();
	ComputeTexCoord();

	Invalidate();
}


void SBrush::SnapToGrid( bool bOnlySelected )
{
	if( bOnlySelected && m_selectionType != SO_ELEM_VERTEX )
		return;

	SBrush backup = *this;	

	size_t vsize = m_BrushVertices.size();

	for( int i = 0; i < vsize; ++i )
	{
		SBrushVertex* bv = m_BrushVertices[i];

		if( bOnlySelected && !bv->bSelected )
			continue;

		bv->pos = gSettings.pGrid->Snap( bv->pos );
	}

	if( IsValid() == false )
	{
		*this = backup;
	}
	else
	{
		Invalidate();
	}
}


void SBrush::SelectSide( Vec3 Origin, Vec3 Dir,bool shear,SBrushSubSelection &subSelection )
{
}


void SBrush::GenerateIndexMesh(IIndexedMesh *pMesh)
{	
	size_t numVerts = m_BrushVertices.size();
	size_t numTriangles = m_BrushTriangles.size();

	pMesh->SetVertexCount(numVerts);
	pMesh->SetFacesCount(numTriangles);
	pMesh->SetTexCoordsCount(numVerts);

	IIndexedMesh::SMeshDesc md;
	pMesh->GetMesh(md);

	int nMaxMatId = 0;

	for( int i=0; i<numVerts; ++i )
	{
		SBrushVertex* v = m_BrushVertices[i];
		md.m_pVerts[i] = v->pos;
		md.m_pTexCoord[i].s = v->st[0];
		md.m_pTexCoord[i].t = v->st[1];
	}

	for( int i=0; i<numTriangles; ++i )
	{
		SBrushTriangle* bt = m_BrushTriangles[i];
		SMeshFace* mf = &md.m_pFaces[i];

		mf->v[0] = mf->t[0] = bt->vertexindices[0];
		mf->v[1] = mf->t[1] = bt->vertexindices[1];
		mf->v[2] = mf->t[2] = bt->vertexindices[2];

		md.m_pNorms[bt->vertexindices[0]] = bt->normal;
		md.m_pNorms[bt->vertexindices[1]] = bt->normal;
		md.m_pNorms[bt->vertexindices[2]] = bt->normal;

		short matID = m_BrushFaces[bt->faceidx]->matID;
		mf->nSubset = matID;
		if( matID > nMaxMatId )
			nMaxMatId = matID;
		mf->dwFlags = 0;
	}

	pMesh->SetSubSetCount(nMaxMatId+1);
	for( int i=0; i<nMaxMatId+1; ++i)
	{
		SMeshSubset &subset = pMesh->GetSubSet(i);
		subset.nMatID = i;
	}
	
	pMesh->CalcBBox();
}


void SBrush::OptimizeMesh( IIndexedMesh* pMesh )
{
	pMesh->Optimize();

	pMesh->RestoreFacesFromIndices();
}


void SBrush::FitTexture(int nX, int nY)
{
	size_t fsize = m_BrushFaces.size();
	for( int i = 0; i < fsize; ++i)
	{
		m_BrushFaces[i]->FitTexture(m_BrushVertices, nX, nY);
	}
}


bool SBrush::ClearSelection()
{
	bool bSelChanged = false;

	std::vector<SBrushFace*>::iterator iface = m_BrushFaces.begin();
	for(;iface!=m_BrushFaces.end();++iface)
	{
		SBrushFace* face = *iface;
		if( face->bSelected == true )
		{
			face->bSelected = false;
			bSelChanged = true;
		}

		if( face->selectededge )
		{
			face->selectededge = 0;
			bSelChanged = true;
		}
	}

	std::vector<SBrushVertex*>::iterator ivertex = m_BrushVertices.begin();
	for(;ivertex!=m_BrushVertices.end();++ivertex)
	{
		SBrushVertex* vertex = *ivertex;
		if( vertex->bSelected == true )
		{
			vertex->bSelected = false;
			bSelChanged = true;
		}
	}

	return bSelChanged;
}


void SBrush::Selection_SnapToGrid()
{
	RecordUndo( "Brush SnapToGrid");
	SnapToGrid(true);
}


void SBrush::Selection_SetMatId( int nMatId )
{
	if( !( m_selectionType == SO_ELEM_FACE || m_selectionType == SO_ELEM_POLYGON ) )
		return;

	RecordUndo("Brush Set Material ID");

	for( int i = 0; i < m_BrushFaces.size(); ++i)
	{
		SBrushFace  * face = m_BrushFaces[i];
		if (face->bSelected)
		{
			face->matID = nMatId;
		}
	}
}


void SBrush::Selection_SelectMatId( int nMatId )
{
	if (m_selectionType == SO_ELEM_VERTEX)
	{
		RecordUndo( "Brush, Select Material ID by vertex");
		ClearSelection();

		for (int i = 0; i < m_BrushFaces.size(); ++i)
		{
			SBrushFace * face = m_BrushFaces[i];
			if (face->matID == nMatId)
			{
				for( size_t j = 0; j < face->pointindexlist.size(); ++j )
				{
					m_BrushVertices[face->pointindexlist[j]]->bSelected = true;
				}
			}
		}
	}
	else if (m_selectionType == SO_HIT_ELEM_EDGE)
	{
		RecordUndo( "Brush, Select Material ID by Edge");
		ClearSelection();

		int nNumberOfFaces(m_BrushFaces.size());
		int nCurrentFace(0);
		int nCurrentEdge(0);

		for( nCurrentFace = 0; nCurrentFace < nNumberOfFaces; ++nCurrentFace )
		{
			SBrushFace * face = m_BrushFaces[nCurrentFace];
			if (face->matID == nMatId )
			{
				for (nCurrentEdge=0;nCurrentEdge < GetNumberOfFaceEdges(face);++nCurrentEdge)
				{
					face->selectededge |= (1<<nCurrentEdge);
				}
			}
		}
	}
	else if(	m_selectionType == SO_ELEM_FACE || 
						m_selectionType == SO_ELEM_POLYGON)
	{	
		RecordUndo( "Brush, Select Material ID by Face");
		ClearSelection();

		for (int i = 0; i < m_BrushFaces.size(); ++i)
		{
			SBrushFace * face = m_BrushFaces[i];
			if (face->matID == nMatId )
			{
				face->bSelected = true;
			}
		}
	}
}

void SBrush::ClearFaces()
{
	std::vector<SBrushVertex*>::iterator iv = m_BrushVertices.begin();
	for( ; iv != m_BrushVertices.end(); ++iv )
	{
		delete *iv;
	}
	m_BrushVertices.clear();

	std::vector<SBrushFace*>::iterator iface = m_BrushFaces.begin();
	for( ; iface != m_BrushFaces.end(); ++iface )
	{
		delete *iface;
	}
	m_BrushFaces.clear();

	std::vector<SBrushTriangle*>::iterator itri = m_BrushTriangles.begin();
	for( ; itri != m_BrushTriangles.end(); ++itri )
	{
		delete *itri;
	}
	m_BrushTriangles.clear();
}


bool SBrush::Clone( _smart_ptr<SBrush>& brush )
{
	brush = new SBrush;
	*brush = *this;
	return true;
}


void SBrush::ComputeFacePlanes()
{
	std::vector<SBrushFace*>::iterator ii = m_BrushFaces.begin();

	for( ; ii != m_BrushFaces.end(); ++ii )
	{
		SBrushFace* f = *ii;
		f->MakePlane(m_BrushVertices,0);
	}
}


SBrushFace* SBrush::RayHit( Vec3 rayOrigin, Vec3 rayDir, float *dist ) const
{
	for( size_t i = 0; i < m_BrushFaces.size(); ++i )
	{
		SBrushFace* brushface = m_BrushFaces[i];

		if( brushface->plane.normal.Dot(rayDir) > 0 )
			continue;

		for( size_t k = 0; k < brushface->triangleidxlist.size(); ++k )
		{
			SBrushTriangle* brushTri = m_BrushTriangles[brushface->triangleidxlist[k]];

			Vec3 v0 = m_BrushVertices[brushTri->vertexindices[0]]->pos;
			Vec3 v1 = m_BrushVertices[brushTri->vertexindices[1]]->pos;
			Vec3 v2 = m_BrushVertices[brushTri->vertexindices[2]]->pos;

			Vec3 e1 = v1-v0;
			Vec3 e2 = v2-v0;

			Vec3 dd = rayDir;
			Vec3 p = dd.cross( e2 );

			float det = e1.dot( p );

			if( det > -VER_EPS && det < VER_EPS )
				continue;

			float	invDet = 1.0f / det;
			Vec3	s = rayOrigin - v0;
			float	u = s.dot( p ) * invDet;

			if( u < 0.f || u > 1.f )
				continue;

			Vec3	q = s.cross( e1 );
			float	v = dd.dot( q ) * invDet;

 			if( v < 0.f || (u + v) > 1.f )
 				continue;

			*dist = e2.dot( q ) * invDet;

			return brushface;
		}
	}

	return NULL;
}


bool SBrush::IsValid() const
{
	size_t tsize = m_BrushTriangles.size();
	for( int i = 0; i < tsize; ++i )
	{
		SBrushTriangle* t = m_BrushTriangles[i];

		const Vec3& v0 = m_BrushVertices[t->vertexindices[0]]->pos;
		const Vec3& v1 = m_BrushVertices[t->vertexindices[1]]->pos;
		const Vec3& v2 = m_BrushVertices[t->vertexindices[2]]->pos;

		if( ((v0-v1)^(v2-v1)).GetLength() < VEC_EPSILON )
			return false;
	}
	return true;
}


int SBrush::GetMemorySize()
{
	int size = 0;

	for( size_t i = 0; i < m_BrushTriangles.size(); ++i )
	{
		size += m_BrushTriangles[i]->GetMemorySize();
	}

	for( size_t i = 0; i < m_BrushVertices.size(); ++i )
	{
		size += m_BrushVertices[i]->GetMemorySize();
	}

	for( size_t i = 0; i < m_BrushFaces.size(); ++i )
	{
		size += m_BrushFaces[i]->GetMemorySize();
	}

	return size;
}


bool SBrush::UpdateMesh()
{
	Invalidate();

	if( IsFaceListEmpty() )
		return false;

	if (!m_pStatObj)
	{
		m_pStatObj = GetIEditor()->Get3DEngine()->CreateStatObj();
		m_pStatObj->AddRef();

		string filename = PathUtil::Make(PathUtil::GetGameFolder(),"Shaders/EngineAssets/TextureMsg/DefaultSolids");

		m_pStatObj->SetMaterial( gEnv->p3DEngine->GetMaterialManager()->LoadMaterial(filename) );
	}

	IIndexedMesh *pMesh = m_pStatObj->GetIndexedMesh();
	GenerateIndexMesh(pMesh);
	OptimizeMesh(pMesh);

	m_pStatObj->Invalidate(true, m_isZeroTolerance ? 0.0f : 0.05f);

	m_nFlags |= BRF_MESH_VALID;
	return m_nFlags & BRF_MESH_VALID;
}


IStatObj* SBrush::GetIStatObj()
{
	if (!m_pStatObj || !(m_nFlags&BRF_MESH_VALID))
	{
		UpdateMesh();
	}
	return m_pStatObj;
}


IIndexedMesh* SBrush::GetIndexedMesh()
{
	IStatObj *pObj = GetIStatObj();
	if (pObj)
		return pObj->GetIndexedMesh();
	return 0;
}


void SBrush::SetModified( bool bModified )
{
	if (bModified)
		m_nFlags |= BRF_MODIFIED;
	else
		m_nFlags &= ~BRF_MODIFIED;
}


bool SBrush::StartSubObjSelection( const Matrix34 &nodeWorldTM,int elemType,int nFlags )
{
	GetEditModeFlag()=true;
	m_nFlags |= BRF_SUB_OBJ_SEL;
	m_selectionType = elemType;
	return true;
}


void SBrush::EndSubObjSelection()
{
	GetEditModeFlag()=false;

	m_nFlags &= ~BRF_SUB_OBJ_SEL;

	if (m_pStatObj)
	{
		int nStatObjFlags = m_pStatObj->GetFlags();
		nStatObjFlags &= ~STATIC_OBJECT_HIDDEN;
		m_pStatObj->SetFlags( nStatObjFlags );
	}
}

AABB SBrush::GetWorldBound() const
{
	AABB aabb;
	aabb.Reset();

	for( size_t i = 0; i < m_BrushVertices.size(); ++i )
	{
		Vec3 worldpts = m_matrix * m_BrushVertices[i]->pos;
		aabb.Add(worldpts);
	}

	return aabb;
}


size_t SBrush::GetNumberOfFaceEdges( SBrushFace* face ) const
{
	return face->triangleidxlist.size() * 3;
}


size_t SBrush::GetNumberOfFaces() const															
{	
	return m_BrushFaces.size();
}


size_t SBrush::GetNumberOfTriangles() const
{
	return m_BrushTriangles.size();
}


const Vec3& SBrush::GetFaceNormal( int idx ) const
{	
	return m_BrushFaces[idx]->plane.normal;
}

float SBrush::GetFaceDistance( int idx ) const
{	
	return m_BrushFaces[idx]->plane.distance;
}

size_t SBrush::GetNumberOfFacePoints( int idx )	const
{	
	return m_BrushFaces[idx]->pointindexlist.size();
}


const Vec3& SBrush::GetFacePointPos( int idx, int vidx ) const
{	
	return m_BrushVertices[m_BrushFaces[idx]->pointindexlist[vidx]]->pos;
}


bool SBrush::IsValidFace( int idx ) const
{	
	return	idx >= 0 && idx < m_BrushFaces.size() && 
					!m_BrushFaces[idx]->pointindexlist.empty() && 
					!m_BrushFaces[idx]->triangleidxlist.empty();	
}


bool SBrush::IsPointSelected( int idx, int vidx ) const
{	
	return m_BrushVertices[m_BrushFaces[idx]->pointindexlist[vidx]]->bSelected;
}


bool SBrush::IsFaceSelected( int idx ) const
{	
	return m_BrushFaces[idx]->bSelected;
}


void SBrush::SetFaceSelected( int idx, bool bSelected ) const
{	
	m_BrushFaces[idx]->bSelected = bSelected;
}


size_t SBrush::GetNumberOfFaceEdges( int idx ) const
{	
	return GetNumberOfFaceEdges(m_BrushFaces[idx]);
}


int SBrush::GetFaceSelectedEdge( int idx ) const
{	
	return m_BrushFaces[idx]->selectededge;
}


const Vec3& SBrush::GetFaceCenter( int idx ) const
{	
	return m_BrushFaces[idx]->GetCenter();
}


void SBrush::CalculateFaceCenter( int idx ) const
{	
	m_BrushFaces[idx]->CalcCenter(m_BrushVertices);
}

void SBrush::CalculateFaceTextureBasis( int idx, Vec3& tu, Vec3& tv) const
{
	SBrushFace::CalcTextureBasis(	m_BrushFaces[idx]->plane,
		m_BrushFaces[idx]->texinfo,
		tu,tv);
}


void SBrush::CalculateFaceBound( int idx, Vec3& mins, Vec3& maxs ) const
{	
	m_BrushFaces[idx]->CalcBound(m_BrushVertices, mins, maxs);
}


const SBrush::STexInfo& SBrush::GetFaceTexInfo( int idx ) const
{
	return m_BrushFaces[idx]->texinfo;
}

void SBrush::SetFaceTexInfo( int idx, const SBrush::STexInfo& ti )	const
{
	m_BrushFaces[idx]->texinfo = ti;
}

int	SBrush::GetFaceMaterialID( int idx ) const
{	
	return m_BrushFaces[idx]->matID;
}

void SBrush::SetFaceMaterialID( int idx, int matID )	const
{	
	m_BrushFaces[idx]->matID = matID;
}

void SBrush::FitFaceTexture( int idx, float fTileU, float fTileV ) const
{
	m_BrushFaces[idx]->FitTexture(m_BrushVertices,fTileU,fTileV);
}

void SBrush::MapEdgeIndexToPolyIndices( int idx, const int& nEdgeIndex, int &rnFirstPolyVertex, int& rnSecondPolyVertex ) const
{
	m_BrushFaces[idx]->MapEdgeIndexToPolyIndices( m_BrushVertices, m_BrushTriangles, nEdgeIndex, rnFirstPolyVertex, rnSecondPolyVertex );
}

size_t SBrush::GetNumberOfVertices() const
{
	return m_BrushVertices.size();
}


const Vec3& SBrush::GetVertexPos( int idx ) const
{
	assert( idx >= 0 && idx < (int)m_BrushVertices.size() );
	return m_BrushVertices[idx]->pos;
}

void SBrush::SetVertexPos( int idx, const Vec3& pos )
{
	assert( idx >= 0 && idx < (int)m_BrushVertices.size() );
	m_BrushVertices[idx]->pos = pos;
}

short SBrush::GetFaceVertexIndex( int idx, int pidx ) const
{
	return m_BrushFaces[idx]->pointindexlist[pidx];
}


const SBrushPlane& SBrush::GetFacePlane( int idx ) const
{
	return m_BrushFaces[idx]->plane;
}


const AABB& SBrush::GetFaceBBox( int idx ) const
{
	return m_BrushFaces[idx]->boundbox;
}


void SBrush::SetVertexList( const std::vector<Vec3>& poslist )
{
	for( size_t i = 0; i < m_BrushVertices.size(); ++i )
	{
		delete m_BrushVertices[i];
	}
	m_BrushVertices.clear();	

	size_t vlistsize = poslist.size();

	for( size_t i = 0; i < vlistsize; ++i )
	{
		m_BrushVertices.push_back(new SBrushVertex(poslist[i]));
	}
}


void SBrush::AddFace( const std::vector<short>& pointindexlist, 
											const STexInfo* texinfo, 
											const short* matID )
{
	SBrushFace* f = new SBrushFace;
	f->pointindexlist = pointindexlist;	
	
	if( texinfo )
		f->texinfo = *texinfo;

	if( matID )
		f->matID = *matID;

	m_BrushFaces.push_back(f);

	if( UpdateFromFace( static_cast<short>(m_BrushFaces.size())-1 ) == false )
	{
		size_t index = m_BrushFaces.size()-1;
		SBrushFace* deletedface = m_BrushFaces[index];
		m_BrushFaces.erase( m_BrushFaces.begin() + index );
		delete deletedface;
	}
}


bool SBrush::UpdateFromFace( short fidx )
{	
	SBrushFace * f = m_BrushFaces[fidx];

	size_t listsize = f->pointindexlist.size();
	std::vector<Vec3> poslist;
	for( size_t i = 0; i < listsize; ++i )
		poslist.push_back(m_BrushVertices[f->pointindexlist[i]]->pos);	

	int entryIdx = FindEntryPointInPolygonForFace(poslist);
	if( entryIdx == -1 )
		entryIdx = 0;

	f->MakePlane( m_BrushVertices, entryIdx );
	f->CalcBound( m_BrushVertices );
	f->CalcCenter( m_BrushVertices );

	for( size_t i = 0; i < f->pointindexlist.size()-2; ++i )
	{
		SBrushTriangle tempTriangle;
		if( tempTriangle.SetVertexIndices(	m_BrushVertices,
																				f->pointindexlist[entryIdx],
																				f->pointindexlist[(entryIdx+i+1)%listsize],
																				f->pointindexlist[(entryIdx+i+2)%listsize], fidx ) == true )
		{
			SBrushTriangle* triangle = new SBrushTriangle(tempTriangle);
			m_BrushTriangles.push_back(triangle);
			f->triangleidxlist.push_back((short)m_BrushTriangles.size()-1);
		}
	}

	return true;
}


short SBrush::FindEntryPointInPolygonForFace( const std::vector<Vec3>& poslist )
{
	const float SameDirectionStandard = 0.9999999f;

	size_t listsize = poslist.size();
	int entryIdx = -1;	

	for( size_t i = 0; i < listsize; ++i )
	{
		bool bImpossible = false;
		
		for( size_t j = 0; j < listsize-2; ++j )
		{
			Vec3 v0 = poslist[i];
			Vec3 v1 = poslist[(i+j+1)%listsize];
			Vec3 v2 = poslist[(i+j+2)%listsize];

			Vec3 v01 = (v1-v0).GetNormalized();
			Vec3 v02 = (v2-v0).GetNormalized();
			float dotV01V02 = v01.Dot(v02);

			if( dotV01V02 >= SameDirectionStandard )
			{	
				bImpossible = true;
				break;
			}
		}

		if( bImpossible == false )
		{
			entryIdx = i;
			break;
		}
	}

	return entryIdx;
}


bool SBrush::FindEntryPointInPolygonForPlane( const std::vector<Vec3>& poslist, short& index0, short& index1, short& index2 )
{
	const float SameDirectionStandard = 0.9999999f;

	size_t listsize = poslist.size();
	int entryIdx = -1;	

	for( size_t i = 0; i < listsize; ++i )
	{
		bool bImpossible = false;

		for( size_t j = 0; j < listsize-2; ++j )
		{
			Vec3 v0 = poslist[i];
			Vec3 v1 = poslist[(i+j+1)%listsize];
			Vec3 v2 = poslist[(i+j+2)%listsize];

			Vec3 v01 = (v1-v0).GetNormalized();
			Vec3 v02 = (v2-v0).GetNormalized();
			float dotV01V02 = v01.Dot(v02);

			if( dotV01V02 < SameDirectionStandard )
			{
				index0 = i;
				index1 = (i+j+1)%listsize;
				index2 = (i+j+2)%listsize;
				return true;
			}
		}
	}

	return false;
}


void SBrush::BackupBrush( SBackupBrushInfo& BackupInfo )
{
	for( size_t i = 0; i < m_BrushVertices.size(); ++i )
		BackupInfo.m_Vertices.push_back( new SBrushVertex( *m_BrushVertices[i] ) );

	for( size_t i = 0; i < m_BrushTriangles.size(); ++i )
		BackupInfo.m_Triangles.push_back( new SBrushTriangle( *m_BrushTriangles[i] ) );

	for( size_t i = 0; i < m_BrushFaces.size(); ++i )
		BackupInfo.m_Faces.push_back( new SBrushFace( *m_BrushFaces[i] ) );
}


void SBrush::RestoreBrush(	SBackupBrushInfo& BackupInfo )
{
	for( size_t i = 0; i < m_BrushVertices.size(); ++i )
	{
		if( m_BrushVertices[i] )
			delete m_BrushVertices[i];
	}

	for( size_t i = 0; i < m_BrushTriangles.size(); ++i )
	{
		if( m_BrushTriangles[i] )
			delete m_BrushTriangles[i];
	}

	for( size_t i = 0; i < m_BrushFaces.size(); ++i )
	{
		if( m_BrushFaces[i] )
			delete m_BrushFaces[i];
	}

	m_BrushVertices		= BackupInfo.m_Vertices;
	m_BrushTriangles	= BackupInfo.m_Triangles;
	m_BrushFaces			= BackupInfo.m_Faces;

	BackupInfo.m_Vertices.clear();
	BackupInfo.m_Triangles.clear();
	BackupInfo.m_Faces.clear();
}


SBrush::SBackupBrushInfo::~SBackupBrushInfo()
{
	for( size_t i = 0; i < m_Vertices.size(); ++i )
		delete m_Vertices[i];

	for( size_t i = 0; i < m_Triangles.size(); ++i )
		delete m_Triangles[i];

	for( size_t i = 0;i  < m_Faces.size(); ++i )
		delete m_Faces[i];
}


bool SBrush::IsConvexHull() const
{
	for( size_t i = 0; i < m_BrushFaces.size(); ++i )
	{
		SBrushFace* baseface = m_BrushFaces[i];
		SBrushTriangle* triangle = m_BrushTriangles[baseface->triangleidxlist[0]];

		Vec3 v[3] = {	m_BrushVertices[triangle->vertexindices[0]]->pos,
									m_BrushVertices[triangle->vertexindices[1]]->pos,
									m_BrushVertices[triangle->vertexindices[2]]->pos };

		SBrushPlane p_base(v[0],v[1],v[2]);

		for( size_t j = 0; j < m_BrushFaces.size(); ++j )
		{
			if( i == j )
				continue;

			SBrushFace* face = m_BrushFaces[j];

			for( size_t k = 0; k < face->pointindexlist.size(); ++k )
			{
				SBrushVertex* vertex = m_BrushVertices[face->pointindexlist[k]];
				if( p_base.Distance(vertex->pos) > VER_EPS )
					return false;
			}
		}
	}
	return true;
}


void SBrush::PivotToCenter( Vec3* newPivot )
{
	Vec3 RelativePivot = m_bounds.GetCenter();

	for( size_t i = 0; i < m_BrushVertices.size(); ++i )
		m_BrushVertices[i]->pos -= RelativePivot;

	m_bounds.min -= RelativePivot;
	m_bounds.max -= RelativePivot;

	if( newPivot )
		*newPivot = GetMatrix().GetTranslation() + RelativePivot;
}


void SBrush::SetMatrix( const Matrix34 &tm, bool Transfer2Owner )
{
	m_matrix = tm;
	if( m_Owner && Transfer2Owner )	
		m_Owner->SetWorldTM(tm);
}


void SBrush::Create( const Vec3 &mins,const Vec3& maxs,int numSides, SHAPETYPE shapetype )
{
	assert( shapetype >= SHAPE_BOX && shapetype < SHAPE_MAX );
	assert( numSides >= 3 );

	ClearFaces();
	CBrushPrimitive BrushPrimitive;

	if( shapetype == SHAPE_BOX )
	{
		BrushPrimitive.CreateBox( this, mins, maxs );
	}
	else if( shapetype == SHAPE_CONE )
	{
		BrushPrimitive.CreateCone( this, mins, maxs, numSides );
	}
	else if( shapetype == SHAPE_SPHERE )
	{
		BrushPrimitive.CreateSphere( this, mins, maxs, numSides );
	}
	else if( shapetype == SHAPE_CYLINDER )
	{
		BrushPrimitive.CreateCylinder( this, mins, maxs, numSides );
	}	

	BuildBrush(true,true);
}


void SBrush::CreateSimpleMesh(const std::vector<Vec3>& vtxs, const int height)
{
	ClearFaces();
	CBrushPrimitive BrushPrimitive;
	BrushPrimitive.CreateSimpleMesh( this, vtxs, height );
	BuildBrush(true,true);	
}


SBrushTriangle* SBrush::AddTriangle()
{
	SBrushTriangle* triangle = new SBrushTriangle;
	m_BrushTriangles.push_back(triangle);
	return triangle;
}


SBrushTriangle* SBrush::AddTriangle(short vertexindex0, short vertexindex1, short vertexindex2, short faceindex)
{
	SBrushTriangle triangle;
	if( triangle.SetVertexIndices( m_BrushVertices, vertexindex0, vertexindex1, vertexindex2, faceindex ) == true )
	{
		SBrushTriangle* triobj = new SBrushTriangle(triangle);
		m_BrushTriangles.push_back(triobj);
		return triobj;
	}
	return NULL;
}


SBrushTriangle* SBrush::AddTriangle(const Vec3& v0, const Vec3& v1, const Vec3& v2, short i0, short i1, short i2, short fidx)
{
	SBrushTriangle triangle;
	if( triangle.SetVertexIndices( v0, v1, v2, i0, i1, i2, fidx ) == true )
	{
		SBrushTriangle* triobj = new SBrushTriangle(triangle);
		m_BrushTriangles.push_back(triobj);
		return triobj;
	}
	return NULL;
}


SBrushVertex* SBrush::AddVertex()
{
	SBrushVertex* vertex = new SBrushVertex;
	m_BrushVertices.push_back(vertex);
	return vertex;
}


SBrushVertex* SBrush::AddVertex( const Vec3& pos )
{
	SBrushVertex* vertex = new SBrushVertex(pos);
	m_BrushVertices.push_back(vertex);
	return vertex;
}
