////////////////////////////////////////////////////////////////////////////
//
//  Crytek Engine Source File.
//  Copyright (C), Crytek Studios, 2002.
// -------------------------------------------------------------------------
//  File name:   BrushCSGCompiler.h
//  Version:     v1.00
//  Created:     12/4/2010 by Jaesik.
//  Compilers:   Visual Studio.NET
//  Description: 
// -------------------------------------------------------------------------
//  History: 
////////////////////////////////////////////////////////////////////////////

#ifndef __BRUSHCSGCOMPILER_H__
#define __BRUSHCSGCOMPILER_H__

#if _MSC_VER > 1000
#pragma once
#endif

#include "BrushPlane.h"
#include "Brush.h"

class CBrushCSGCompiler
{

public:

	bool Compile( SBrush::ECSGOperationEnum operation, 
								const SBrush* brush0, 
								const SBrush* brush1, 
								_smart_ptr<SBrush>& outBrush,
								Vec3& newPivot ) const;


private:

	class CCSGBrush;

	struct SBSP_CSGNode
	{
	public:

		enum EBSPNodeEnum
		{
			eBNE_Partition	= 0x00000001,
			eBNE_LeafFace		= 0x00000002,
			eBNE_LeafSolid	= 0x00000004
		};

	public:

		SBSP_CSGNode( CCSGBrush* brush )
			:			m_flag(0),
						m_faceindex(-1),
						m_frontnode(NULL),
						m_backnode(NULL),
						m_brush(brush)
		{
		}

		~SBSP_CSGNode()
		{
			SAFE_DELETE(m_frontnode);
			SAFE_DELETE(m_backnode);
		}

		inline bool IsLeaf() const
		{
			return (m_flag&eBNE_LeafFace) || (m_flag&eBNE_LeafSolid);
		}

		inline bool IsSolid() const
		{
			return m_flag&eBNE_LeafSolid;
		}

		static SBSP_CSGNode* CreateNode( CCSGBrush* brush )
		{	
			return new SBSP_CSGNode(brush);
		}

		inline void SetFlag( unsigned char flag )		{	m_flag = flag;	}

	private:

		friend class CBrushCSGCompiler::CCSGBrush;

		unsigned char m_flag;
		short					m_faceindex;
		SBrushPlane		m_plane;
		CCSGBrush*		m_brush;

		SBSP_CSGNode* m_frontnode;
		SBSP_CSGNode* m_backnode;
	};

	struct SCSGVertex : public CRefCountBase
	{
		SCSGVertex( const SCSGVertex& rhs )
			: m_position(rhs.m_position)
		{
		}

		SCSGVertex( const Vec3& v )
			: m_position(v)
		{
		}

		Vec3 m_position;
	};

	typedef _smart_ptr<SCSGVertex>	CSGVertexSmartPointer;

	struct SCSGFace : public CRefCountBase
	{	
		SCSGFace()
		{
			m_boundbox.Reset();
		}

		SCSGFace( const SCSGFace& rhs )
			: m_vertexindexlist(rhs.m_vertexindexlist),
				m_boundbox(rhs.m_boundbox),
				m_plane(rhs.m_plane),
				m_TexInfo(rhs.m_TexInfo),
				m_matID(rhs.m_matID)
		{
		}

		inline void ComputeBoundBox( const std::vector<SCSGVertex*>& vertexlist )
		{
			m_boundbox.Reset();
			for( size_t i(0); i < m_vertexindexlist.size(); ++i )
				m_boundbox.Add(vertexlist[m_vertexindexlist[i]]->m_position);
		}

		inline void AddVertexIndex( short index )
		{
			m_vertexindexlist.push_back(index);
		}

		inline void InsertVertexIndex( short location, short index )
		{
			m_vertexindexlist.insert( m_vertexindexlist.begin() + location, index );
		}

		std::vector<short> m_vertexindexlist;
		AABB m_boundbox;
		SBrushPlane m_plane;
		SBrush::STexInfo m_TexInfo;
		short	m_matID;
	};
	
	typedef _smart_ptr<SCSGFace>		CSGFaceSmartPointer;

	class CCSGBrush : public CRefCountBase
	{
	public:

		CCSGBrush()
			: m_BSPRootNode(NULL)
		{
		}

		~CCSGBrush()
		{
			SAFE_DELETE(m_BSPRootNode);
		}

		bool GenerateBSPTreeFromBrush();

		void RecursivelyClip( const SBSP_CSGNode* node,
													const SBrush::ECSGOperationEnum& csgoperation,
													std::vector<short>& facepool,
													bool bFlip,
													_smart_ptr<CCSGBrush> finalBrush );

		void AddFaceGroup(	const CCSGBrush* brush, 
												const std::vector<short>& facepool,
												bool bInverseOrder = false );	

		void EliminateTJunction();
		void FlipPlanes();

		SCSGFace*		AddFace();
		void				DeleteFace( size_t faceindex );
		SCSGVertex*	AddVertex( const SCSGVertex& vertex );

		inline size_t GetNumberOfFaces() const										{		return m_FaceList.size();								}
		inline size_t GetNumberOfVertices() const									{		return m_VertexList.size();							}
		inline const SBSP_CSGNode* GetBSPRootNode() const					{		return m_BSPRootNode;										}
		inline const Vec3& GetVertexPosition( int index ) const		{		return m_VertexList[index]->m_position;	}
		inline const std::vector<short>& GetFaceVertexIndexList( int index )	const	{	return m_FaceList[index]->m_vertexindexlist;	}
		inline const SBrushPlane& GetFacePlane( int index )	const						{	return m_FaceList[index]->m_plane;		}
		inline const SBrush::STexInfo& GetFaceTexInfo( int index ) const		{	return m_FaceList[index]->m_TexInfo;	}
		inline short GetFaceMatID( int index ) const												{	return m_FaceList[index]->m_matID;		}
		inline void SetVertexPosition( int index, const Vec3& position )		{	m_VertexList[index]->m_position = position;	}	

		bool PullOntoPlane( const Vec3& position, const SBrushPlane& plane, Vec3* outPosition ) const;


	private:

		struct SFindFaceOutputParameters
		{
			SFindFaceOutputParameters( 	std::vector<short>& _frontFaces,
																	std::vector<short>& _backFaces,
																	std::vector<short>& _onplaneFaces )
																	: frontFaces(_frontFaces),
																		backFaces(_backFaces),
																		onplaneFaces(_onplaneFaces),
																		facetouchlist(NULL)
			{
			}

			SFindFaceOutputParameters( 	std::vector<short>& _frontFaces,
																	std::vector<short>& _backFaces,
																	std::vector<short>& _onplaneFaces,
																	std::vector<bool>& _facetouchlist )
																	: frontFaces(_frontFaces),
																		backFaces(_backFaces),
																		onplaneFaces(_onplaneFaces),
																		facetouchlist(&_facetouchlist)
			{
			}

			std::vector<short>& frontFaces;
			std::vector<short>& backFaces;
			std::vector<short>& onplaneFaces;
			std::vector<bool>* facetouchlist;
		};

		struct SSplitFaceOutputParameters
		{
			SSplitFaceOutputParameters( CSGFaceSmartPointer& _frontFace, 
																	CSGFaceSmartPointer& _backFace )
																	: frontFace(_frontFace),
																		backFace(_backFace),
																		newvertexindexlist(NULL)
			{
			}

			SSplitFaceOutputParameters( CSGFaceSmartPointer& _frontFace, 
																	CSGFaceSmartPointer& _backFace,
																	std::vector<short>& _newvertexindexlist )
																	: frontFace(_frontFace),
																		backFace(_backFace),
																		newvertexindexlist(&_newvertexindexlist)
			{
			}

			CSGFaceSmartPointer&		frontFace;
			CSGFaceSmartPointer&		backFace;
			std::vector<short>* newvertexindexlist;
		};

		struct SFindPlaneOutputParameters
		{
			SFindPlaneOutputParameters( SBrushPlane& plane, 
																	short& faceindex )
																	: bestplane(plane),
																		bestfaceindex(faceindex)	
			{
			}

			SFindPlaneOutputParameters( SBrushPlane& plane, 
																	short& faceindex, 
																	std::vector<bool>& touches )
																	: bestplane(plane),
																		bestfaceindex(faceindex),
																		facetouches(&touches)
			{
			}

			SBrushPlane& bestplane;
			short& bestfaceindex;
			std::vector<bool>* facetouches;
		};
		

	private:

		void CountFrontBackVertex(	const SBrushPlane& plane,
																std::vector<char>& vertexsign,
																short* numberOfFrontVertices = NULL,
																short* numberOfBackVertices = NULL,
																short* numberOfZeroVertices = NULL ) const;

		void SplitFace( const SBrushPlane& plane, 
										const SCSGFace& face, 
										const std::vector<char>& vertexsign,
										CSGFaceSmartPointer& frontFace,
										CSGFaceSmartPointer& backFace,
										CSGFaceSmartPointer& onPlaneFace );

		bool RecursivelyGenerateBSPTree(	SBSP_CSGNode* parent, 
																			std::vector<short>& facepool,
																			std::vector<bool>& facetouches );

		void FindFrontAndBackFaces(	const SBrushPlane& plane,
																const std::vector<short>& facepool,
																SFindFaceOutputParameters& outputParameters );

		void SplitFaceTo2Parts(	const SBrushPlane& plane,
														const SCSGFace& face,
														const std::vector<char>& vertexsign,
														SSplitFaceOutputParameters& outputPararmeters );

		bool FindBestPlane(	const std::vector<short>& facepool,
												SFindPlaneOutputParameters& outputParameters ) const;


	private:

		std::vector<CSGVertexSmartPointer>	m_VertexList;
		std::vector<CSGFaceSmartPointer>		m_FaceList;
		SBSP_CSGNode*	m_BSPRootNode;
	};

	typedef _smart_ptr<CCSGBrush>	CSGBrushSmartPointer;


private:

	bool	CreateCSGBrushFromSBrush(	const SBrush*	fromBrush, 
																	const Vec3& pivot,
																	CSGBrushSmartPointer& resultBrush ) const;

	bool	CreateSBrushFromCSGBrush(	const CSGBrushSmartPointer&	fromBrush, 
																	_smart_ptr<SBrush>& resultBrush ) const;

	bool	ModifyPivot(	const Vec3& origin, 
											CSGBrushSmartPointer brush, 
											Vec3& newOrigin ) const;


private:

	static const float VER_EPS;
	static const float VER_TIGHTEPS;

};

#endif //__BRUSHCSGCOMPILER_H__