#include "StdAfx.h"
#include "PropertiesPanel.h"
#include "AICoverSurface.h"
#include "AI/AIManager.h"
#include "AI/CoverSurfaceManager.h"


IMPLEMENT_DYNCREATE(CAICoverSurface, CBaseObject);


namespace
{
	int s_propertiesID = 0;
	CPropertiesPanel* s_properties = 0;

	struct PropertiesDeleter
	{
		~PropertiesDeleter()
		{
			delete s_properties;
		}
	} __deleter;
};


CAICoverSurface::CAICoverSurface()
: m_sampler(0)
, m_surfaceID(0)
, m_helperScale(1.0f)
{
	GetIEditor()->GetAI()->GetCoverSurfaceManager()->AddSurfaceObject(this);
}

CAICoverSurface::~CAICoverSurface()
{
	if (m_sampler)
	{
		m_sampler->Release();
		m_sampler = 0;
	}

	GetIEditor()->GetAI()->GetCoverSurfaceManager()->RemoveSurfaceObject(this);
}

bool CAICoverSurface::Init(IEditor* editor, CBaseObject *prev, const CString &file)
{
	SetColor(RGB(70, 130, 180));

	if (CBaseObject::Init(editor, prev, file))
	{
		if (prev)
		{
			CAICoverSurface* original = (CAICoverSurface*)prev;

			// Clone Properties.
			if (original->m_propertyVars.get())
				ClonePropertyVars(original->m_propertyVars.get());

			UpdatePropertyVars();
		}
		else
			CreatePropertyVars();

		return true;
	}

	return false;
}

void CAICoverSurface::DeleteThis()
{
	delete this;
}

void CAICoverSurface::Display(DisplayContext& disp)
{
	if (IsFrozen())
		disp.SetFreezeColor();
	else
		disp.SetColor(GetColor());

	Matrix34 scale(Matrix34::CreateScale(Vec3(gSettings.gizmo.helpersScale * GetHelperScale())));

	disp.RenderObject(STATOBJECT_ANCHOR, GetWorldTM() * scale);

	if (m_sampler)
	{
		switch(m_sampler->Update(0.033f))
		{
		case ICoverSampler::Finished:
			CommitSurface();
			ReleaseSampler();
			break;
		case ICoverSampler::Error:
			ClearSurface();
			ReleaseSampler();
			break;
		default:
			break;
		}
	}

	if (m_surfaceID && IsSelected() && (GetIEditor()->GetSelection()->GetCount() < 15))
		gEnv->pAISystem->GetCoverSystem()->DrawSurface(m_surfaceID);

	DrawDefault(disp);
}

int CAICoverSurface::MouseCreateCallback(CViewport* view, EMouseEvent event, CPoint& point, int flags)
{
	if ((event == eMouseMove) || (event == eMouseLDown))
	{
		if (GetIEditor()->GetAxisConstrains() != AXIS_TERRAIN)
			SetPos(view->MapViewToCP(point));
		else
		{
			bool terrain;
			Vec3 pos = view->ViewToWorld(point, &terrain);
			if (terrain)
				pos.z = GetIEditor()->GetTerrainElevation(pos.x, pos.y) + 0.25f;

			SetPos(view->SnapToGrid(pos));
		}

		if (event == eMouseLDown)
			return MOUSECREATE_OK;

		return MOUSECREATE_CONTINUE;
	}

	return CBaseObject::MouseCreateCallback(view, event, point, flags);
}

bool CAICoverSurface::HitTest(HitContext& hitContext)
{
	Vec3 origin = GetWorldPos();
	float radius = GetHelperSize() * GetHelperScale();

	Vec3 w = origin - hitContext.raySrc;
	w = hitContext.rayDir.Cross(w);

	float d = w.GetLengthSquared();
	if (d < (radius * radius) + hitContext.distanceTolerance)
	{
		Vec3 i0;
		if (Intersect::Ray_SphereFirst(Ray(hitContext.raySrc, hitContext.rayDir), Sphere(origin, radius), i0))
		{
			hitContext.dist = hitContext.raySrc.GetDistance(i0);

			return true;
		}

		hitContext.dist = hitContext.raySrc.GetDistance(origin);
		
		return true;
	}

	return false;
}

void CAICoverSurface::GetLocalBounds(AABB &aabb)
{
	float extent = GetHelperSize() * GetHelperScale();
	
	std::swap(aabb, AABB(Vec3(-extent), Vec3(extent)));
}

void CAICoverSurface::SetHelperScale(float scale)
{
	m_helperScale = scale;
}

float CAICoverSurface::GetHelperScale()
{
	return m_helperScale;
}

float CAICoverSurface::GetHelperSize() const
{
	return 0.5f * gSettings.gizmo.helpersScale;
}

const CoverSurfaceID& CAICoverSurface::GetSurfaceID() const
{
	return m_surfaceID;
}

void CAICoverSurface::SetSurfaceID(const CoverSurfaceID& coverSurfaceID)
{
	m_surfaceID = coverSurfaceID;
}

void CAICoverSurface::Generate()
{
	CreateSampler();
	StartSampling();

	while (m_sampler->Update(0.5f) == ICoverSampler::InProgress);

	if ((m_sampler->GetState() == ICoverSampler::Error) || !m_sampler->GetSampleCount())
	{
		Vec3 pos = GetWorldPos();

		CString error;
		error.Format("AI Cover Surfaces '%s' at (%f, %f, %f) is empty!", (const char*)GetName(), pos.x, pos.y, pos.z);

		GetIEditor()->GetErrorReport()->ReportError(CErrorRecord(this, CErrorRecord::ESEVERITY_WARNING, error));
	}
	else
		CommitSurface();

	ReleaseSampler();
}

void CAICoverSurface::CreateSampler()
{
	if (m_sampler)
		m_sampler->Release();

	m_sampler = gEnv->pAISystem->GetCoverSystem()->CreateCoverSampler();
}

void CAICoverSurface::ReleaseSampler()
{
	if (m_sampler)
	{
		m_sampler->Release();
		m_sampler = 0;
	}
}

void CAICoverSurface::StartSampling()
{
	ICoverSampler::Params params(GetParamsFromPropertyVars());

	Matrix34 worldTM = GetWorldTM();
	params.position = worldTM.GetTranslation();
	params.direction = worldTM.GetColumn1();
	
	m_sampler->StartSampling(params);
}

void CAICoverSurface::CommitSurface()
{
	uint32 sampleCount = m_sampler->GetSampleCount();
	if (sampleCount)
	{
		ICoverSystem::SurfaceInfo surfaceInfo;
		surfaceInfo.sampleCount = sampleCount;
		surfaceInfo.samples = m_sampler->GetSamples();

		if (!m_surfaceID)
			m_surfaceID = gEnv->pAISystem->GetCoverSystem()->AddSurface(surfaceInfo);
		else
			gEnv->pAISystem->GetCoverSystem()->UpdateSurface(m_surfaceID, surfaceInfo);
	}
	else
		ClearSurface();
}

void CAICoverSurface::ClearSurface()
{
	if (m_surfaceID)
	{
		gEnv->pAISystem->GetCoverSystem()->RemoveSurface(m_surfaceID);
		m_surfaceID = 0;
	}
}

void CAICoverSurface::SetPropertyVarsFromParams(const ICoverSampler::Params& params)
{
	m_limitLeftVar = params.limitLeft;
	m_limitRightVar = params.limitRight;

	m_limitDepthVar = params.limitDepth;
	m_limitHeightVar = params.limitHeight;

	m_widthIntervalVar = params.widthSamplerInterval;
	m_heightIntervalVar = params.heightSamplerInterval;

	m_widthGapToleranceVar = params.widthGapTolerance;
	m_heightGapToleranceVar = params.heightGapTolerance;

	m_refineEdgesVar = params.refineEdges;
}

ICoverSampler::Params CAICoverSurface::GetParamsFromPropertyVars()
{
	ICoverSampler::Params params;

	params.limitDepth = m_limitDepthVar;
	params.limitHeight = m_limitHeightVar;

	params.limitLeft = m_limitLeftVar;
	params.limitRight = m_limitRightVar;

	params.widthSamplerInterval = m_widthIntervalVar;
	params.heightSamplerInterval = m_heightIntervalVar;

	params.widthGapTolerance = m_widthGapToleranceVar;
	params.heightGapTolerance = m_heightGapToleranceVar;

	params.refineEdges = m_refineEdgesVar;

	return params;
}

void CAICoverSurface::Serialize(CObjectArchive& archive)
{
	CBaseObject::Serialize(archive);

	SerializeValue(archive, "SurfaceID", m_surfaceID);
	SerializeVarEnum(archive, "Sampler", m_samplerVar);
	SerializeVar(archive, "LimitDepth", m_limitDepthVar);
	SerializeVar(archive, "LimitHeight", m_limitHeightVar);
	SerializeVar(archive, "LimitLeft", m_limitLeftVar);
	SerializeVar(archive, "LimitRight", m_limitRightVar);
	SerializeVar(archive, "SampleWidth", m_widthIntervalVar);
	SerializeVar(archive, "SampleHeight", m_heightIntervalVar);
	SerializeVar(archive, "WidthGapTolerance", m_widthGapToleranceVar);
	SerializeVar(archive, "HeightGapTolerance", m_heightGapToleranceVar);
	SerializeVar(archive, "RefineEdges", m_refineEdgesVar);
}

XmlNodeRef CAICoverSurface::Export(const CString &levelPath, XmlNodeRef &xmlNode)
{
	return 0;
}

void CAICoverSurface::InvalidateTM(int whyFlags)
{
	CBaseObject::InvalidateTM(whyFlags);

	if (!m_sampler)
		CreateSampler();

	StartSampling();
}

void CAICoverSurface::BeginEditParams(IEditor* editor, int flags)
{
	CBaseObject::BeginEditParams(editor, flags);

	if (!s_properties)
		s_properties = new CPropertiesPanel(AfxGetMainWnd());
	else
		s_properties->DeleteVars();
	s_properties->AddVars(m_propertyVars.get());

	if (!s_propertiesID)
		s_propertiesID = AddUIPage(CString(GetTypeName()) + " Properties", s_properties);
}

void CAICoverSurface::EndEditParams(IEditor* editor)
{
	if (s_propertiesID)
	{
		RemoveUIPage(s_propertiesID);
		s_propertiesID = 0;
		s_properties = 0;
	}

	CBaseObject::EndEditParams(editor);
}

void CAICoverSurface::BeginEditMultiSelParams(bool allSameType)
{
	CBaseObject::BeginEditMultiSelParams(allSameType);

	if (!allSameType)
		return;

	if (!s_properties)
		s_properties = new CPropertiesPanel(AfxGetMainWnd());
	else
		s_properties->DeleteVars();

	s_propertiesID = AddUIPage(CString(GetTypeName()) + " Properties", s_properties);

	if (m_propertyVars.get())
	{
		CSelectionGroup* selectionGroup = GetIEditor()->GetSelection();
		for (int i = 0; i < selectionGroup->GetCount(); ++i)
		{
			CAICoverSurface* surfaceObject = (CAICoverSurface*)selectionGroup->GetObject(i);
			if (CVarBlockPtr surfaceObjectVars = surfaceObject->m_propertyVars.get())
				s_properties->AddVars(surfaceObjectVars);
		}

		if (!s_propertiesID)
			s_propertiesID = AddUIPage(CString(GetTypeName()) + " Properties", s_properties);
	}
}

void CAICoverSurface::EndEditMultiSelParams()
{
	if (s_propertiesID)
	{
		RemoveUIPage(s_propertiesID);
		s_propertiesID = 0;
		s_properties = 0;
	}

	CBaseObject::EndEditMultiSelParams();
}

void CAICoverSurface::UpdatePropertyVars()
{
/*	if (s_properties)
	{
		s_properties->DeleteVars();
		s_properties->AddVars(m_propertyVars.get());
	}
	*/
}

void CAICoverSurface::OnPropertyVarChange(IVariable* var)
{
	ReleaseSampler();
	CreateSampler();

	StartSampling();
}

void CAICoverSurface::CreatePropertyVars()
{
	m_propertyVars.reset(new CVarBlock());

	m_propertyVars->AddVariable(m_samplerVar, "Sampler");
	m_propertyVars->AddVariable(m_limitLeftVar, "Limit Left");
	m_propertyVars->AddVariable(m_limitRightVar, "Limit Right");
	m_propertyVars->AddVariable(m_limitDepthVar, "Limit Depth");
	m_propertyVars->AddVariable(m_limitHeightVar, "Limit Height");

	m_propertyVars->AddVariable(m_widthIntervalVar, "Sample Width");
	m_propertyVars->AddVariable(m_heightIntervalVar, "Sample Height");

	m_propertyVars->AddVariable(m_widthGapToleranceVar, "Width Gap Tolerance");
	m_propertyVars->AddVariable(m_heightGapToleranceVar, "Height Gap Tolerance");
	m_propertyVars->AddVariable(m_refineEdgesVar, "Refine Edges");


	m_samplerVar->AddEnumItem(CString("Default"), CString("Default"));
	m_samplerVar->Set(CString("Default"));

	m_limitLeftVar->SetLimits(0.0f, 100.0f, 0.05f);
	m_limitRightVar->SetLimits(0.0f, 100.0f, 0.05f);
	m_limitDepthVar->SetLimits(0.0f, 10.0f, 0.05f);
	m_limitHeightVar->SetLimits(0.0f, 100.0f, 0.05f);
	m_widthIntervalVar->SetLimits(0.01f, 0.5f, 0.01f);
	m_heightIntervalVar->SetLimits(0.01f, 0.5f, 0.01f);
	m_widthGapToleranceVar->SetLimits(0.0f, 1.0f, 0.01f);
	m_heightGapToleranceVar->SetLimits(0.0f, 1.0f, 0.01f);

	SetPropertyVarsFromParams(ICoverSampler::Params()); // defaults

	m_propertyVars->AddOnSetCallback(functor(*this, &CAICoverSurface::OnPropertyVarChange));
}

void CAICoverSurface::ClonePropertyVars(CVarBlockPtr originalPropertyVars)
{
	m_propertyVars.reset(originalPropertyVars->Clone(true));
	m_propertyVars->AddOnSetCallback(functor(*this, &CAICoverSurface::OnPropertyVarChange));
}