////////////////////////////////////////////////////////////////////////////
//
//  Crytek Engine Source File.
//  Copyright (C), Crytek Studios, 2002.
// -------------------------------------------------------------------------
//  File name:   objectlayermanager.h
//  Version:     v1.00
//  Created:     9/5/2003 by Timur.
//  Compilers:   Visual Studio.NET
//  Description: Manages objects layers.
// -------------------------------------------------------------------------
//  History:
//
////////////////////////////////////////////////////////////////////////////

#ifndef __objectlayermanager_h__
#define __objectlayermanager_h__
#pragma once

#include "ObjectLayer.h"

class CObjectManager;
class CObjectArchive;
class CLayerNodeAnimator;

/** Manager of objects layers.
		Instance of these hosted by CObjectManager class.
*/
class CObjectLayerManager
{
public:
	enum EUpdateType {
		ON_LAYER_ADD,
		ON_LAYER_REMOVE,
		ON_LAYER_MODIFY,
		ON_LAYER_SELECT,
		ON_LAYER_UPDATEALL,
		ON_LAYER_CHANGESIZE
	};
	//! Update callback, first integer is one of EUpdateType enums.
	typedef Functor2<int,CObjectLayer*> LayersUpdateCallback;

	CObjectLayerManager( CObjectManager *pObjectManager );

	//! Creates a new layer, and associate it with layer manager.
	CObjectLayer* CreateLayer();
	//! Add already created layer to manager.
	void AddLayer( CObjectLayer *pLayer );
	//! Delete layer from layer manager.
	void DeleteLayer( CObjectLayer *pLayer );
	//! Delete all layers from layer manager.
	void ClearLayers();

	//! Find layer by layer GUID.
	CObjectLayer* FindLayer( REFGUID guid ) const;

	//! Search for layers by name.
	bool FindLayersByName( const CString &layerName,std::vector<CObjectLayer*>* layers = NULL ) const;

	//! Search for layer by full path name.
	CObjectLayer* FindLayerByFullPathName( const CString &layerFullPathName, const char *delimiter = CObjectLayer::kDefaultPathDelimiter ) const;

	//! Search for layer by its parent layer and name.
	CObjectLayer* FindLayer( const CObjectLayer* pParent, const CString& layerName ) const;

	//! 
	CObjectLayer* FindLayerByName_Workaround(const CString& layerName);

	//! Get a list of all managed layers.
	void GetLayers( std::vector<CObjectLayer*> &layers ) const;

	//! Set this layer is current.
	void SetCurrentLayer( CObjectLayer* pCurrLayer );
	CObjectLayer* GetCurrentLayer() const { return m_pCurrentLayer; };

	const CString& GetLayersPath() const { return m_layersPath; }
	void SetLayersPath(const CString& layersPath) { m_layersPath = layersPath; }

	//! Associate On Layers update listener.
	//! This callback will be called everytime layers are added/removed or modified.
	void AddUpdateListener( const LayersUpdateCallback &cb );
	//! Remove update listeners.
	void RemoveUpdateListener( const LayersUpdateCallback &cb );

	//! Called when some layers gets modified.
	void NotifyLayerChange( CObjectLayer *pLayer );

	//! Called for changing size of window with layers control.
	void ChangeLayerWindowSize( );

	//! Get pointer to object manager who owns this layer manager.
	CObjectManager* GetObjectManager() const { return m_pObjectManager; }

	//! Export layer to objects archive.
	void ExportLayer( CObjectLayer *pLayer,CObjectArchive &ar,bool bExportExternalChilds );
	//! Import layer from objects archive.
	CObjectLayer* ImportLayer( CObjectArchive &ar,bool bResolveObject );

	// Serialize layer manager (called from object manager).
	void	Serialize( CObjectArchive &ar,bool bIgnoreExternalLayers );

	//! Resolve links between layers.
	void ResolveLayerLinks();

	void CreateMainLayer();

	void RemoveLayerSwitch(CObjectLayer* pLayer);
	bool AddLayerSwitch(CObjectLayer* pLayer);
	bool InitLayerSwitches(bool isOnlyClear = false);
	void ExportLayerSwitches(XmlNodeRef &node);
	void SetupLayerSwitches(bool isOnlyClear = false,  bool isOnlyRenderNodes = false );

	void SetLayerNodeAnimators(IAnimSequence* pSequence, bool isOnlyClear = false);
	void SetLayerNodeAnimators(IMovieSystem* pMovieSystem, bool isOnlyClear = false);
	void SaveUserSpecificData(XmlNodeRef& node);
	void LoadUserSpecificData(XmlNodeRef& node);

	void SetGameMode( bool inGame );
	
private:
	void SaveExternalLayer( CObjectLayer *pLayer,CObjectArchive &archive );
	void LoadExternalLayer( const CString &layerName,const CString &layerFullName,CObjectArchive &archive );
	void NotifyListeners( EUpdateType type,CObjectLayer *pLayer );

	CLayerNodeAnimator* GetOrCreateLayerNodeAnimator();
	void LoadLayersRecursively(CObjectArchive &ar, XmlNodeRef &layerNode, const bool bIgnoreExternalLayers);
	void SaveLayersRecursively(CObjectArchive &ar, XmlNodeRef& layerNode, CObjectLayer* pLayer, const bool bIgnoreExternalLayers);

	//////////////////////////////////////////////////////////////////////////
	//! Map of layer GUID to layer pointer.
	typedef std::map<GUID,TSmartPtr<CObjectLayer>,guid_less_predicate> LayersMap;		
	LayersMap m_layersMap;

	//! Pointer to currently active layer.
	TSmartPtr<CObjectLayer>	m_pCurrentLayer;
	//! Main layer, root for all other layers.
	CObjectLayer * m_pMainLayer;
	//////////////////////////////////////////////////////////////////////////

	CFunctorsList<LayersUpdateCallback> m_listeners;
	CObjectManager *m_pObjectManager;
	//! Layers path relative to level folder.
	CString m_layersPath;

	bool m_bOverwriteDuplicates;

	// support Layer Switches in Flow Graph
	std::list<CObjectLayer*> m_layerSwitches;

	// Callback used in the editor to animate the layer node.
	_smart_ptr<IAnimNodeAnimator> m_layerNodeAnimator;

};

#endif // __objectlayermanager_h__
