////////////////////////////////////////////////////////////////////////////
//
//  Crytek Engine Source File.
//  Copyright (C), Crytek Studios, 2002.
// -------------------------------------------------------------------------
//  File name:   vegetationmap.h
//  Version:     v1.00
//  Created:     31/7/2002 by Timur.
//  Compilers:   Visual Studio.NET
//  Description: 
// -------------------------------------------------------------------------
//  History:
//
////////////////////////////////////////////////////////////////////////////

#ifndef __vegetationmap_h__
#define __vegetationmap_h__

#if _MSC_VER > 1000
#pragma once
#endif

#include "VegetationObject.h"

/** This is description of single static vegetation object instance.
*/
struct CVegetationInstance
{
	CVegetationInstance *next;	//! Next object instance
	CVegetationInstance *prev;	//! Prev object instance.

	Vec3	pos;									//!< Instance position.
	float scale;								//!< Instance scale factor.
	CVegetationObject *object;	//!< Object of this instance.
	IRenderNode       *pRenderNode; // Render node of this object.
  IDecalRenderNode  *pRenderNodeGroundDecal; // Decal under vegetation.
	uint8 brightness;						//!< Brightness of this object.
	uint8 angle;								//!< Instance rotation.

	int m_refCount;								//!< Number of references to this vegetation instance.

	bool							m_boIsSelected;

	//! Add new refrence to this object.
	void AddRef() {	m_refCount++; };

	//! Release refrence to this object.
	//! when reference count reaches zero, object is deleted.
	void Release()
	{
		if ((--m_refCount) <= 0)
			delete this;
	}
private:
	// Private destructor, to not be able to delete it explicitly.
	~CVegetationInstance() { };
};

//////////////////////////////////////////////////////////////////////////
/** CVegetationMap stores static objects distributed over terrain.
		It keeps a list of all allocated geometry objects and all places instances.
*/
class CVegetationMap : public IEditorNotifyListener
{
public:
	CVegetationMap();
	~CVegetationMap();

	//////////////////////////////////////////////////////////////////////////
	// Map
	//////////////////////////////////////////////////////////////////////////
	//! Allocate sectors map.
	//! @param terrainSize Size of terrain in meters.
	void Allocate( CHeightmap *heightmap );

	//! Get number of sectors at side.
	int GetNumSectors() const;

	//! Get total Size of vegetation map.
	int GetSize() const;

	//! Convert world coordinate to sector coordinate.
	int WorldToSector( float worldCoord ) const;

	//! Place all objects in vegetation map to 3d engine terrain.
	void PlaceObjectsOnTerrain();

	//! Get total number of vegetation instances.
	int GetNumInstances() const { return m_numInstances; };

	//////////////////////////////////////////////////////////////////////////
	// Vegetation Objects
	//////////////////////////////////////////////////////////////////////////
	//! Get number of use vegetation objects.
	int GetObjectCount() const { return m_objects.size(); }
	//! Get vegetation object.
	CVegetationObject*	GetObject( int i ) { return m_objects[i]; }

	//! Create new object.
	//! @param prev Source object to clone from.
	CVegetationObject* CreateObject( CVegetationObject *prev=0 );

	// Generates a vegetation object id, when available.
	// If no vegetation object id is available, returns -1.
	int GenerateVegetationObjectId();

	// Inserts an object, assigning a new ID to it.
	// Should be used only with objects which ID is not registered.
	// One such case, is while Undo.
	bool InsertObject(CVegetationObject *obj);

	//! Remove object.
	void RemoveObject( CVegetationObject *obj );
	//! Replace one vegetation object with another.
	void ReplaceObject( CVegetationObject *pOldObject,CVegetationObject *pNewObject );
	//! Remove all objects.
	void ClearObjects();

	//! Hide all instances of this object.
	void HideObject( CVegetationObject *object,bool bHide );
	void HideAllObjects( bool bHide );

	// !Merge 2 object to first one
	void CVegetationMap::MergeObjects( CVegetationObject *object, CVegetationObject *objectMerged );

	//! Export static object and all its instances.
	//! @return Number of serialized instances.
	int ExportObject( CVegetationObject *object,XmlNodeRef &node,CRect *saveRect=NULL );
	void ImportObject( XmlNodeRef &node,const Vec3& offset=Vec3(0,0,0) );

	//! Export part of vegetation map.
	void ExportBlock( const CRect &rect,CXmlArchive &ar );
	//! Import part of vegetation map.
	void ImportBlock( CXmlArchive &ar,CPoint placeOffset=CPoint(0,0) );

	//! Unload all rendering geometry from objects.
	void UnloadObjectsGeometry();

  //! Rather ugly way to ensure the radius is synchronised between editor objects and
  //! IStatObjects. Must change after E3 - hassle Danny if you see it after this!
  void SetAIRadius(IStatObj *pObj, float radius);

	//////////////////////////////////////////////////////////////////////////
	// Object Painting.
	//////////////////////////////////////////////////////////////////////////

	//! Place single object at specified location.
	CVegetationInstance* PlaceObjectInstance( const Vec3 &worldPos,CVegetationObject* brush );
	void DeleteObjInstance( CVegetationInstance *obj );
	void RemoveDuplVegetation( int x1=-1, int y1=-1, int x2=-1, int y2=-1);
	//! Find object instances closest to the point within given radius.
	CVegetationInstance* GetNearestInstance( const Vec3 &worldPos,float radius );
	void GetObjectInstances( float x1,float y1,float x2,float y2,std::vector<CVegetationInstance*> &instances );
	void GetAllInstances( std::vector<CVegetationInstance*> &instances );
	//! Move instance to new position.
	void MoveInstance( CVegetationInstance* obj,const Vec3 &newPos );
	//! Remove object from 3D engine and place it back again.
	void RepositionObject( CVegetationObject *object );

  //! Update vegetation position z coordinate
  void OnHeightMapChanged( );

	//! Remove objects within specified area from 3D engine and then place it back again on top of terrain.
	void RepositionArea( const AABB &box, const Vec3 &offset =  Vec3(0, 0, 0), int nRot=0);

	//! Scale all instances of this objects.
	void ScaleObjectInstances( CVegetationObject *object,float fScale );
	//! Randomally rotate all instances of this objects.
	void RandomRotateInstances( CVegetationObject *object );
	//! Clear rotation for all instances of this objects.
	void ClearRotateInstances( CVegetationObject *object );

	//! Remove any object at specified location.
	void RemoveObjectInstance( const Vec3 &worldPos );

	//! Paint objects on rectangle using given brush.
	void PaintBrush( CRect &rc,bool bCircle,CVegetationObject* brush, Vec3 * pPos = 0 );

	//! Clear objects in rectangle using given brush.
	//! @param brush Object to remove, if NULL all object will be cleared.
	void ClearBrush( CRect &rc,bool bCircle,CVegetationObject* brush );

	//! Clear all object within mask.
	void ClearMask( const CString &maskFile );

	//! Sets this brighness to all objects within specified rectangle.
	//! x,y,w,h are specified in world units (meters).
	//! \param brightness 0..255 brightness without ground texture, with hill shadows but without object shadows
	//! \param brightness_shadowmap 0..255 brightness without ground texture, with hill shadows and with object shadows
	void PaintBrightness( float x,float y,float w,float h,uint8 brightness,uint8 brightness_shadowmap );

	//////////////////////////////////////////////////////////////////////////
	//! Serialize vegetation map to archive.
	void Serialize( CXmlArchive &xmlAr );

	//! Serialize vegetation instances
	void SerializeInstances( CXmlArchive &xmlAr,CRect *rect=NULL );

	//! Generate shadows from static objects and place them in shadow map bitarray.
	void GenerateShadowMap( CByteImage &shadowmap,float shadowAmmount,const Vec3 &sunVector );

	//! Draw sectors to texture.
	void DrawToTexture( uint32 *texture,int textureWidth,int textureHeight,int srcX,int srcY );

	//! Record undo info for vegetation instance.
	void RecordUndo( CVegetationInstance *obj );

	//////////////////////////////////////////////////////////////////////////
	// Validate Vegetation map about errors.
	//////////////////////////////////////////////////////////////////////////
	void Validate( CErrorReport &report );

	// Calculates texture memory usage for the vegatation objects.
	int GetTexureMemoryUsage( bool bOnlySelectedObjects );
	int GetSpritesMemoryUsage( bool bOnlySelectedObjects );

	void GetMemoryUsage( ICrySizer *pSizer );

	// Set engine params for all objects
	void SetEngineObjectsParams();

	// Calculate Height for vegetation instance if it placed on the brush.
	float CalcHeightOnBrushes(const Vec3 & p, const Vec3 & posUpper );

	//////////////////////////////////////////////////////////////////////////
	// IEditorNotifyListener
	//////////////////////////////////////////////////////////////////////////
	virtual void OnEditorNotifyEvent( EEditorNotifyEvent event );
	//////////////////////////////////////////////////////////////////////////

private:
	struct SectorInfo
	{
		CVegetationInstance *first; // First vegetation object instance in this sector.
	};

	//! Get sector by world coordinates.
	SectorInfo*	GetVegSector( const Vec3 &worldPos );

	//! Get sector by 2d map coordinates.
	SectorInfo*	GetVegSector( int x,int y );

	//! Remove all object in vegetation map from 3d engine terrain.
	void RemoveObjectsFromTerrain();

	//! Create new object instance in map.
	CVegetationInstance* CreateObjInstance( CVegetationObject *object,const Vec3 &pos );
	void DeleteObjInstance( CVegetationInstance *obj,SectorInfo *sector );
	//! Only to be used by undo/redo.
	void AddObjInstance( CVegetationInstance *obj );

	void SectorLink( CVegetationInstance *object,SectorInfo *sector );
	void SectorUnlink( CVegetationInstance *object,SectorInfo *sector );

	//! Return true if there is no specified objects within radius from given position.
	bool CanPlace( CVegetationObject *object,const Vec3 &pos,float radius );

	//! For the purpose optimization, the data used frequently in the loop is retrieved once.
	void PrepareRoadObjectData( std::vector<CBaseObject*>& vRoads,std::vector<std::vector<Vec3> >& vvRoadCurvePoints);

	//! Return true if there is any 'Road' object which prevents planting vegetation.
	bool IsVegetationMask( const Vec3& pos,const float radius,const std::vector<CBaseObject*>& vRoads, const std::vector<std::vector<Vec3> >& vvRoadCurvePoints);

	//! Returns true if theres no any objects within radius from given position.
	bool IsPlaceEmpty( const Vec3 &pos,float radius,CVegetationInstance *ignore );

	void Clear();
	void ClearSectors();

	void LoadOldStuff( CXmlArchive &xmlAr );

private:
	friend class CUndoVegInstanceCreate;
	void RegisterInstance( CVegetationInstance *obj );

	//! 2D Array of sectors that store vegetation objects.
	SectorInfo*	m_sectors;
	//! Size of single sector in meters.
	int m_sectorSize;
	//! Number of sectors in each dimension in map (Resolution of sectors map).
	int m_numSectors;
	//! Size of all map in meters.
	int m_mapSize;

	//! Minimal distance between two objects.
	//! Objects cannot be placed closer together that this distance.
	float m_minimalDistance;

	//! world to sector scaling ratio.
	float m_worldToSector;

	typedef std::vector<TSmartPtr<CVegetationObject> > Objects;
	Objects m_objects;

	//! Taken group ids.
	std::set<int> m_usedIds;

	//////////////////////////////////////////////////////////////////////////
	I3DEngine *m_I3DEngine;
	CHeightmap* m_heightmap;

	int m_numInstances;
  int m_nSID;
};

//////////////////////////////////////////////////////////////////////////
inline CVegetationMap::SectorInfo*	CVegetationMap::GetVegSector( int x,int y )
{
	assert( x >= 0 && x < m_numSectors && y >= 0 && y < m_numSectors );
	return &m_sectors[y*m_numSectors + x];
}

//////////////////////////////////////////////////////////////////////////
inline CVegetationMap::SectorInfo*	CVegetationMap::GetVegSector( const Vec3 &worldPos )
{
	int x = ftoi(worldPos.x*m_worldToSector);
	int y = ftoi(worldPos.y*m_worldToSector);
	if (x >= 0 && x < m_numSectors && y >= 0 && y < m_numSectors )
		return &m_sectors[y*m_numSectors + x];
	else
		return 0;


}

#endif // __vegetationmap_h__
