#include "StdAfx.h"
#include "WeaponEditorSettingsDialog.h"
#include <IItem.h>
#include "WeaponEditorDialog.h"
#include <ISourceControl.h>
#include "ItemSystem.h"

enum EWeaponSlot
{
	WEAPON_SLOT_FP = 0,
	WEAPON_SLOT_TP = 1,
	WEAPON_SLOT_AUX = 3
};


#define  IDC_TASKPANEL 1

IMPLEMENT_DYNAMIC(CWeaponEditorSettingsDialog, CToolbarDialog)

//////////////////////////////////////////////////////////////////////////
BEGIN_MESSAGE_MAP(CWeaponEditorSettingsDialog, CToolbarDialog)
	ON_WM_SIZE()
	ON_BN_CLICKED(IDC_WEAPON_FPVIEW, OnClicked_FPCamera)
	ON_BN_CLICKED(IDC_WEAPON_ZOOM, OnClicked_Zoom)
	ON_BN_CLICKED(IDC_WEAPON_RELOAD, OnClicked_Reload)
	ON_BN_CLICKED(IDC_WEAPON_BROWSE_FP, OnClicked_BrowseFP)
	ON_BN_CLICKED(IDC_WEAPON_BROWSE_TP, OnClicked_BrowseTP)
	ON_BN_CLICKED(IDC_WEAPON_BROWSE_AUX, OnClicked_BrowseAUX)
	ON_BN_CLICKED(IDC_WEAPON_CLEAR_FP, OnClicked_ClearFP)
	ON_BN_CLICKED(IDC_WEAPON_CLEAR_TP, OnClicked_ClearTP)
	ON_BN_CLICKED(IDC_WEAPON_CLEAR_AUX, OnClicked_ClearAUX)
	ON_EN_CHANGE(IDC_WEAPON_OFFSET_X, OnOffsetChange)
	ON_EN_CHANGE(IDC_WEAPON_OFFSET_Y, OnOffsetChange)
	ON_EN_CHANGE(IDC_WEAPON_OFFSET_Z, OnOffsetChange)
	ON_EN_CHANGE(IDC_WEAPON_OFFSET_YW, OnOffsetChange)
	ON_EN_CHANGE(IDC_WEAPON_OFFSET_PT, OnOffsetChange)
	ON_EN_CHANGE(IDC_WEAPON_OFFSET_RL, OnOffsetChange)
	ON_EN_CHANGE(IDC_WEAPON_OFFSET_FOV, OnOffsetChange)
	ON_BN_CLICKED(IDC_WEAPON_ATTACH_ADD, OnAddAttachment)
	ON_BN_CLICKED(IDC_WEAPON_ATTACH_REM, OnRemoveAttachment)
	ON_LBN_SELCHANGE(IDC_WEAPON_HELPERS_LIST, OnHelperSelectionChanged)
	ON_LBN_SELCHANGE(IDC_WEAPON_BONE_HELPERS_LIST, OnBoneHelperSelected)
	ON_BN_CLICKED(IDC_SETHELPER, SetHelperPos)
	ON_BN_CLICKED(IDC_SETBONE_HELPER, SetBoneHelperBone)
	ON_BN_CLICKED(IDC_WEAPON_FP, &CWeaponEditorSettingsDialog::OnBnClickedWeaponFp)
	ON_BN_CLICKED(IDC_WEAPON_TP, &CWeaponEditorSettingsDialog::OnBnClickedWeaponTp)
END_MESSAGE_MAP()

static int g_offsetId[] = {IDC_WEAPON_OFFSET_X, IDC_WEAPON_OFFSET_Y, IDC_WEAPON_OFFSET_Z, IDC_WEAPON_OFFSET_YW, IDC_WEAPON_OFFSET_PT, IDC_WEAPON_OFFSET_RL};

//////////////////////////////////////////////////////////////////////////
// CWeaponEditorSettingsDialog
//////////////////////////////////////////////////////////////////////////
CWeaponEditorSettingsDialog::CWeaponEditorSettingsDialog() 
: CToolbarDialog(IDD,NULL)
, m_pItem(NULL)
, m_editor(NULL)
, m_hasChanges(false)
, m_pAttachedItem(NULL)
, m_attachedId(0)
, m_attachedSocketId(0)
, m_attachedSocketTypeId(0)
{
}

//////////////////////////////////////////////////////////////////////////
CWeaponEditorSettingsDialog::~CWeaponEditorSettingsDialog()
{
	m_attachmentsList.Detach();
	m_helpersList.Detach();
	m_subObjects.Detach();
	m_boneHelpersList.Detach();
	m_boneHelpersBone.Detach();
	CleanUpAttachedEntity();
}

//////////////////////////////////////////////////////////////////////////
void CWeaponEditorSettingsDialog::OnSize(UINT nType, int cx, int cy)
{
	__super::OnSize(nType, cx, cy);

	RecalcLayout();
}

//////////////////////////////////////////////////////////////////////////
void CWeaponEditorSettingsDialog::DoDataExchange(CDataExchange* pDX)
{
	__super::DoDataExchange(pDX);
}


//////////////////////////////////////////////////////////////////////////
BOOL CWeaponEditorSettingsDialog::OnInitDialog()
{
	CRect clRect(0, 0, 267, 800);
	SetWindowPos(NULL, 0, 0, clRect.Width(), clRect.Height(), SWP_NOACTIVATE|SWP_NOOWNERZORDER|SWP_NOMOVE|SWP_NOZORDER);

	clRect.top += 170;
	clRect.bottom = 800;	

	// Tasks panel
	m_wndTaskPanel.Create(WS_CHILD|WS_BORDER|WS_VISIBLE|WS_CLIPSIBLINGS|WS_CLIPCHILDREN, clRect, this, IDC_TASKPANEL);

	m_wndTaskPanel.SetBehaviour(xtpTaskPanelBehaviourExplorer);
	m_wndTaskPanel.SetTheme(xtpTaskPanelThemeNativeWinXP);
	m_wndTaskPanel.SetSelectItemOnFocus(TRUE);
	m_wndTaskPanel.AllowDrag(FALSE);

	{
		HWND hWnd;
		GetDlgItem(IDC_WEAPON_FP_GEOM, &hWnd);
		m_geometryFP.Attach(hWnd);
		GetDlgItem(IDC_WEAPON_TP_GEOM, &hWnd);
		m_geometryTP.Attach(hWnd);
		GetDlgItem(IDC_WEAPON_AUX_GEOM, &hWnd);
		m_geometryAUX.Attach(hWnd);
	}

	{
		// Offset group
		m_OffsetGroup = m_wndTaskPanel.AddGroup(1);
		m_OffsetGroup->SetCaption("Offset \\ Rotation \\ FOV");
		

		// Create sub-dialog
		m_dlgOffset.Create(IDD_WEAPON_OFFSET, &m_wndTaskPanel);
		CXTPTaskPanelGroupItem* pItem = m_OffsetGroup->AddControlItem(m_dlgOffset);
		
		pItem->GetMargins().SetRect(0, 0, 0, 0);		
		m_dlgOffset.SetItem(pItem);

		m_pos_x.Create(&m_dlgOffset, IDC_WEAPON_OFFSET_X);
		m_pos_x.SetRange(-10, 10);
		m_pos_x.SetValue(0);
		m_pos_x.SetStep(0.01);
		m_pos_x.SetInternalPrecision(2);
		m_pos_x.EnableWindow(TRUE);

		m_pos_y.Create(&m_dlgOffset, IDC_WEAPON_OFFSET_Y);
		m_pos_y.SetRange(-10, 10);
		m_pos_y.SetValue(0);
		m_pos_y.SetStep(0.01);
		m_pos_y.SetInternalPrecision(2);
		m_pos_y.EnableWindow(TRUE);

		m_pos_z.Create(&m_dlgOffset, IDC_WEAPON_OFFSET_Z);
		m_pos_z.SetRange(-10, 10);
		m_pos_z.SetValue(0);
		m_pos_z.SetStep(0.01);
		m_pos_z.SetInternalPrecision(2);
		m_pos_z.EnableWindow(TRUE);

		m_rot_yw.Create(&m_dlgOffset, IDC_WEAPON_OFFSET_YW);
		m_rot_yw.SetRange(-10, 10);
		m_rot_yw.SetValue(0);
		m_rot_yw.SetStep(0.01);
		m_rot_yw.SetInternalPrecision(2);
		m_rot_yw.EnableWindow(TRUE);

		m_rot_pt.Create(&m_dlgOffset, IDC_WEAPON_OFFSET_PT);
		m_rot_pt.SetRange(-10, 10);
		m_rot_pt.SetValue(0);
		m_rot_pt.SetStep(0.01);
		m_rot_pt.SetInternalPrecision(2);
		m_rot_pt.EnableWindow(TRUE);

		m_rot_rl.Create(&m_dlgOffset, IDC_WEAPON_OFFSET_RL);
		m_rot_rl.SetRange(-10, 10);
		m_rot_rl.SetValue(0);
		m_rot_rl.SetStep(0.01);
		m_rot_rl.SetInternalPrecision(2);
		m_rot_rl.EnableWindow(TRUE);

		m_fov.Create(&m_dlgOffset, IDC_WEAPON_OFFSET_FOV);
		m_fov.SetRange(10, 100);
		m_fov.SetValue(60);
		m_fov.SetStep(1.0);
		m_fov.SetInternalPrecision(0);
		m_fov.EnableWindow(TRUE);
	}

	{
		// Attachments group
		CXTPTaskPanelGroup* pAttachmentsGroup = m_wndTaskPanel.AddGroup(1);
		pAttachmentsGroup->SetCaption("Attachments");

		// Create sub-dialog
		m_dlgAttachmens.Create(IDD_WEAPON_ATTACHMENTS, &m_wndTaskPanel);
		CXTPTaskPanelGroupItem* pItem = pAttachmentsGroup->AddControlItem(m_dlgAttachmens);
		pItem->GetMargins().SetRect(0, 0, 0, 0);
		m_dlgAttachmens.SetItem(pItem);

		HWND hWnd;
		m_dlgAttachmens.GetDlgItem(IDC_WEAPON_ATTACH_LIST, &hWnd);
		m_attachmentsList.Attach(hWnd);
	}

	{
		// Helpers group
		m_HelpersGroup = m_wndTaskPanel.AddGroup(2);
		m_HelpersGroup->SetCaption("Helpers");

		// Create sub-dialog
		m_dlgHelpers.Create(IDD_WEAPON_HELPERS, &m_wndTaskPanel);
		CXTPTaskPanelGroupItem* pItem = m_HelpersGroup->AddControlItem(m_dlgHelpers);
		pItem->GetMargins().SetRect(0, 0, 0, 0);
		m_dlgHelpers.SetItem(pItem);

		HWND hWnd;
		m_dlgHelpers.GetDlgItem(IDC_WEAPON_HELPERS_LIST, &hWnd);
		m_helpersList.Attach(hWnd);

		m_dlgHelpers.GetDlgItem(IDC_WEAPON_SUBOBJ, &hWnd);
		m_subObjects.Attach(hWnd);

		m_dlgHelpers.GetDlgItem(IDC_WEAPON_BONE_HELPERS_LIST, &hWnd);
		m_boneHelpersList.Attach(hWnd);

		m_dlgHelpers.GetDlgItem(IDC_WEAPON_HELPER_BONE, &hWnd);
		m_boneHelpersBone.Attach(hWnd);

		for (int i=0; i<6; ++i)
		{
			m_helperPos[i].Create(&m_dlgHelpers, g_offsetId[i]);
			m_helperPos[i].SetRange(-180, 180);
			m_helperPos[i].SetValue(0);
			m_helperPos[i].SetStep(0.001);
			m_helperPos[i].SetInternalPrecision(4);
			m_helperPos[i].SetFloatFormat("%.4f");
			m_helperPos[i].EnableWindow(TRUE);
		}
	}

	m_wndTaskPanel.Reposition(FALSE);

	RecalcLayout();

	CheckRadioButton(IDC_WEAPON_FP, IDC_WEAPON_TP, IDC_WEAPON_FP);	

	return TRUE;
}

//////////////////////////////////////////////////////////////////////////
void CWeaponEditorSettingsDialog::RecalcLayout()
{
	//__super::RecalcLayout();
	if (m_wndTaskPanel.GetSafeHwnd() == 0)
		return;

	CRect rc;
	GetClientRect(rc);
	rc.top += 170;
	m_wndTaskPanel.SetWindowPos(NULL, rc.left, rc.top, rc.Width(), rc.Height(), SWP_NOZORDER | SWP_NOMOVE);
}

//////////////////////////////////////////////////////////////////////////
void CWeaponEditorSettingsDialog::SetEditorDialog(CWeaponEditorDialog* editor)
{
	m_editor = editor;
}

//////////////////////////////////////////////////////////////////////////
void CWeaponEditorSettingsDialog::OnClicked_FPCamera()
{
	CCustomButton* pButton=(CCustomButton*)GetDlgItem(IDC_WEAPON_FPVIEW);
	int check = pButton->GetCheck();
	if (m_editor)
		m_editor->GetPreviewDialog().GetModelViewport()->SetFPCamera(check);
}

//////////////////////////////////////////////////////////////////////////
void CWeaponEditorSettingsDialog::OnClicked_Zoom()
{
	CCustomButton* pButton=(CCustomButton*)GetDlgItem(IDC_WEAPON_ZOOM);
	int check = pButton->GetCheck();
	if (m_pItem && m_pItem->GetIWeapon())
	{
		if (check)
			m_pItem->GetIWeapon()->StartZoom(0);
		else
			m_pItem->GetIWeapon()->StopZoom(0);
	}	
	if (m_editor)
		m_editor->GetPreviewDialog().GetModelViewport()->SetZoom(check);
}

void CWeaponEditorSettingsDialog::OnClicked_Reload()
{
	if(m_pItem)
	{
		m_pItem->Select(false);
		m_pItem->Select(true);
		UpdateHelpers();
		CheckViewMode();
	}
}

void CWeaponEditorSettingsDialog::OnClicked_BrowseFP()
{
	BrowseForSlot(WEAPON_SLOT_FP);
}

void CWeaponEditorSettingsDialog::OnClicked_BrowseTP()
{
	BrowseForSlot(WEAPON_SLOT_TP);
}

void CWeaponEditorSettingsDialog::OnClicked_BrowseAUX()
{
	BrowseForSlot(WEAPON_SLOT_AUX);
}

void CWeaponEditorSettingsDialog::OnClicked_ClearFP()
{
	ClearSlot(WEAPON_SLOT_FP);
}

void CWeaponEditorSettingsDialog::OnClicked_ClearTP()
{
	ClearSlot(WEAPON_SLOT_TP);
}

void CWeaponEditorSettingsDialog::OnClicked_ClearAUX()
{
	ClearSlot(WEAPON_SLOT_AUX);
}

void CWeaponEditorSettingsDialog::ClearSlot(int slot)
{
	if(!m_pItem)
		return;

	IItemEdit* edit = m_pItem->GetIItemEdit();
	if(*edit->GetSlotGeometry(slot))
	{
		OnChange();
		edit->SetSlotGeometry(slot,"");
		FillGeometry();
		OnClicked_Reload();
	}
}

void CWeaponEditorSettingsDialog::BrowseForSlot(int slot)
{
	if(!m_pItem)
		return;

	IItemEdit* edit = m_pItem->GetIItemEdit();

	CString filename = Path::GetRelativePath(CString(edit->GetSlotGeometry(slot)));
	if ( CFileUtil::SelectSingleFile(EFILE_TYPE_GEOMETRY, filename, "Geometry files (*.chr, *.cgf)|*.chr;*.cgf") )
	{
		OnChange();
		edit->SetSlotGeometry(slot, filename.GetBuffer());
		FillGeometry();
		OnClicked_Reload();
	}
}

void CWeaponEditorSettingsDialog::OnOffsetChange()
{
	if (m_pItem)
		OnChange();

	if (gEnv->pConsole->GetCVar("i_offset_right"))
		gEnv->pConsole->GetCVar("i_offset_right")->Set((float)m_pos_x.GetValue());
	if (gEnv->pConsole->GetCVar("i_offset_front"))
		gEnv->pConsole->GetCVar("i_offset_front")->Set((float)m_pos_y.GetValue());
	if (gEnv->pConsole->GetCVar("i_offset_up"))
		gEnv->pConsole->GetCVar("i_offset_up")->Set((float)m_pos_z.GetValue());

	if (gEnv->pConsole->GetCVar("i_rotation_yaw"))
		gEnv->pConsole->GetCVar("i_rotation_yaw")->Set((float)m_rot_yw.GetValue());
	if (gEnv->pConsole->GetCVar("i_rotation_pitch"))
		gEnv->pConsole->GetCVar("i_rotation_pitch")->Set((float)m_rot_pt.GetValue());
	if (gEnv->pConsole->GetCVar("i_rotation_roll"))
		gEnv->pConsole->GetCVar("i_rotation_roll")->Set((float)m_rot_rl.GetValue());

	if (gEnv->pConsole->GetCVar("i_offset_nearFOV"))
		gEnv->pConsole->GetCVar("i_offset_nearFOV")->Set((float)m_fov.GetValue());

	if (!m_helpersList.GetSafeHwnd() || m_helpersList.GetCurSel() < 0)
		return;
	
	Matrix34 lM;
	if (!GetSelectedHelperPos(lM, true))
		return;
	Matrix34 wM;
	if (!GetSelectedHelperPos(wM))
		return;
	Matrix34 pM = wM * lM.GetInverted();

	Matrix34 m(IDENTITY);
	Ang3 rot(m_helperPos[3].GetValue(), m_helperPos[4].GetValue(), m_helperPos[5].GetValue());
	rot = DEG2RAD(rot);
	m.SetRotationXYZ(rot, Vec3(m_helperPos[0].GetValue(), m_helperPos[1].GetValue(), m_helperPos[2].GetValue()));
	m = pM * m;
	SetSelectedHelperPos(m);
}

void CWeaponEditorSettingsDialog::OnLoad()
{
	if (gEnv->pConsole->GetCVar("i_offset_right"))
		m_pos_x.SetValue(gEnv->pConsole->GetCVar("i_offset_right")->GetFVal());
	if (gEnv->pConsole->GetCVar("i_offset_front"))
		m_pos_y.SetValue(gEnv->pConsole->GetCVar("i_offset_front")->GetFVal());
	if (gEnv->pConsole->GetCVar("i_offset_up"))
		m_pos_z.SetValue(gEnv->pConsole->GetCVar("i_offset_up")->GetFVal());

	if (gEnv->pConsole->GetCVar("i_rotation_yaw"))
		m_rot_yw.SetValue(gEnv->pConsole->GetCVar("i_rotation_yaw")->GetFVal());
	if (gEnv->pConsole->GetCVar("i_rotation_pitch"))
		m_rot_pt.SetValue(gEnv->pConsole->GetCVar("i_rotation_pitch")->GetFVal());
	if (gEnv->pConsole->GetCVar("i_rotation_roll"))
		m_rot_rl.SetValue(gEnv->pConsole->GetCVar("i_rotation_roll")->GetFVal());

	if (gEnv->pConsole->GetCVar("i_offset_nearFOV"))
		m_fov.SetValue(gEnv->pConsole->GetCVar("i_offset_nearFOV")->GetFVal());

	m_pItem = m_editor->GetWeapon();
	CleanUpAttachedEntity();

	m_attachmentsList.ResetContent();
	UpdateHelpers();

	OnClicked_FPCamera();
	OnClicked_Zoom();
	FillGeometry();
	m_hasChanges = false;
}

void CWeaponEditorSettingsDialog::UpdateHelpers()
{
	m_helpersList.ResetContent();
	m_editHelpers.resize(0);
	if (!m_pItem)
		return;
	int numSockets = m_pItem->GetIItemEdit()->GetSocketsCount();
	for (int i=0; i<numSockets; ++i)
	{
		if (m_pItem->GetIItemEdit()->GetSocketParentSlot(i))
			continue;
		int numTypes = m_pItem->GetIItemEdit()->GetSocketTypesCount(i);
		for (int j=0; j<numTypes; ++j)
		{
			string name = "s-";
			const char* socket = m_pItem->GetIItemEdit()->GetSocketName(i);
			if (socket && socket[0] != '\0')
				name += socket;
			
			name += "-";
			const char* bone = m_pItem->GetIItemEdit()->GetSocketBone(i);
			if (bone && bone[0] != '\0')
				name += bone;

			name += "-";
			name += m_pItem->GetIItemEdit()->GetSocketType(i, j);
			if (m_pItem->GetIItemEdit()->IsHelperUsed(i, j))
				name += "*";
			int id = m_helpersList.AddString(name);
			m_helpersList.SetItemData(id, m_editHelpers.size());
			m_editHelpers.push_back(SEditHelper(i,j));
		}
	}

	int numPins = m_pItem->GetIItemEdit()->GetPinHelpersCount();
	for (int i=0; i<numPins; ++i)
	{
		string name = "p-";
		name += m_pItem->GetIItemEdit()->GetPinHelperName(i);
		if (m_pAttachedItem && i == m_attachedId)
			name += "*";
		m_helpersList.AddString(name);

		int id = m_helpersList.AddString(name);
		m_helpersList.SetItemData(id, m_editHelpers.size());
		m_editHelpers.push_back(SEditHelper(i));
	}

	for (int i=0; i<6; ++i)
		m_helperPos[i].SetValue(0.0f);

	FillHelpersBox();
	FillBoneHelpersList();
}

bool CWeaponEditorSettingsDialog::GetSelectedHelperPos(Matrix34& m, bool local) const
{
	m.SetIdentity();

	if (!m_pItem || m_helpersList.GetCurSel() < 0)
		return false;

	int id = m_helpersList.GetItemData(m_helpersList.GetCurSel());
	const SEditHelper& eh = m_editHelpers[id];

	if(eh.pin)
	{
		QuatT pos = m_pItem->GetIItemEdit()->GetPinHelperPos(eh.idx);
		m.SetRotation33(Matrix33(pos.q));
		m.SetTranslation(pos.t);
		m = m_pItem->GetEntity()->GetWorldTM() * m;
	}
	else
	{
		QuatT pos = m_pItem->GetIItemEdit()->GetHelperPos(eh.idx, eh.sub_idx, local);
		m.SetRotation33(Matrix33(pos.q));
		m.SetTranslation(pos.t);
		m = m_pItem->GetEntity()->GetWorldTM() * m;
	}
	return true;
}

bool CWeaponEditorSettingsDialog::SetSelectedHelperPos(const Matrix34& m)
{
	if (!m_pItem || m_helpersList.GetCurSel() < 0)
		return false;

	OnChange();

	int id = m_helpersList.GetItemData(m_helpersList.GetCurSel());
	const SEditHelper& eh = m_editHelpers[id];

	if(eh.pin)
	{
		m_pItem->GetIItemEdit()->SetPinHelperPos(eh.idx, QuatT(m));
	}
	else
	{
		m_pItem->GetIItemEdit()->SetHelperPos(eh.idx, eh.sub_idx, QuatT(m));
	}
	return true;
}

bool CWeaponEditorSettingsDialog::GetHelperPos(const char* name, Matrix34& m)
{
	if (!m_pItem)
		return false;

	int numPins = m_pItem->GetIItemEdit()->GetPinHelpersCount();
	for (int i=0; i<numPins; ++i)
	{
		if (!stricmp(m_pItem->GetIItemEdit()->GetPinHelperName(i), name))
		{
			m = Matrix34(m_pItem->GetIItemEdit()->GetPinHelperPos(i));
			return true;
		}		
	}

	return false;
}

void CWeaponEditorSettingsDialog::MoveSelectedHelper(const Vec3& v)
{
	Matrix34 m;
	if (GetSelectedHelperPos(m))
	{
		m.AddTranslation(v);
		SetSelectedHelperPos(m);
		UpdateHelperPos();
	}
}

void CWeaponEditorSettingsDialog::RotateSelectedHelper(const Quat& q)
{
	Matrix34 m;
	if (GetSelectedHelperPos(m))
	{
		m.SetRotationXYZ(Ang3(q * Quat(m)), m.GetTranslation());
		SetSelectedHelperPos(m);
		UpdateHelperPos();
	}
}

void CWeaponEditorSettingsDialog::OnHelperSelectionChanged()
{
	UpdateHelperPos();
	FillHelpersBox();
}

void CWeaponEditorSettingsDialog::UpdateHelperPos()
{
	Matrix34 m(IDENTITY);
	if (!GetSelectedHelperPos(m, true))
		return;

	m_helperPos[0].SetValue(m.GetTranslation().x);
	m_helperPos[1].SetValue(m.GetTranslation().y);
	m_helperPos[2].SetValue(m.GetTranslation().z);

	Ang3 rot(m);
	rot = RAD2DEG(rot);
	m_helperPos[3].SetValue(rot.x);
	m_helperPos[4].SetValue(rot.y);
	m_helperPos[5].SetValue(rot.z);
}

void CWeaponEditorSettingsDialog::SetHelperPos()
{
	if (!m_pItem)
		return;

	CString objName;
	if (m_subObjects.GetCurSel() >= 0)
		m_subObjects.GetLBText(m_subObjects.GetCurSel(), objName);

	if (objName == "")
		return;

	QuatT pos(IDENTITY);

	if (IStatObj* pStatObj = m_pItem->GetEntity()->GetStatObj(0))
	{
		int num = pStatObj->GetSubObjectCount();
		for (int i=0; i<num; ++i)
		{
			if (IStatObj::SSubObject* pObj = pStatObj->GetSubObject(i))
			{
				if (!stricmp(pObj->name, objName))
				{
					pos = QuatT(pObj->localTM);
					break;
				}
			}
		}
	}

	if (ICharacterInstance* pChar = m_pItem->GetEntity()->GetCharacter(0))
	{
		ISkeletonPose* pSkeleton = pChar->GetISkeletonPose();
		//pSkeleton->SetDefaultPose();
		int32 id = pSkeleton->GetJointIDByName(objName);
		pos = pSkeleton->GetAbsJointByID(id);
		Matrix34 m = m_pItem->GetEntity()->GetSlotLocalTM(0, true);
		pos = QuatT(m * Matrix34(pos));
	}

	if (SetSelectedHelperPos(Matrix34(pos)))
		UpdateHelperPos();
}

void CWeaponEditorSettingsDialog::OnBoneHelperSelected()
{
	FillBoneHelpersBox();
}

void CWeaponEditorSettingsDialog::SetBoneHelperBone()
{
	OnChange();

	int id = m_boneHelpersList.GetCurSel();
	if(id != -1)
	{
		int idx = m_boneHelpersList.GetItemData(id);
		CString name;
		m_boneHelpersList.GetText(id, name);
		
		
		IItemEdit* item = m_pItem->GetIItemEdit();
		IItemEdit::SItemHelper hlpr;

		if(item && item->GetBoneHelper(idx, hlpr) && name == hlpr.name)
		{
			int sel = m_boneHelpersBone.GetCurSel();
			if(sel != -1)
			{
				CString bonename;
				m_boneHelpersBone.GetLBText(sel, bonename);
				hlpr.bone = bonename.GetBuffer();
				item->SetBoneHelper(idx, hlpr);
			}
		}
	}
}

const char* CWeaponEditorSettingsDialog::GetFilePath() const
{
	const char* name = CWeaponEditorSettingsDialog::m_editor->GetWeaponName();
	if (!name)
		return NULL;
	static string path;
	if (!m_pItem->GetIItemEdit()->GetPinHelpersCount())
		path.Format("%s\\Items\\Weapons\\%s.xml", PathUtil::GetGameFolder(), name);
	else
		path.Format("%s\\Items\\Accessories\\%s.xml", PathUtil::GetGameFolder(), name);
	return path;
}

void CWeaponEditorSettingsDialog::OnSave()
{
	if (!m_pItem)
		return;
	if (!m_hasChanges)
		return;
	const char* path = GetFilePath();
	if (!path)
		return;	

	if (GetIEditor()->GetSourceControl())
	{
		uint32 fileAttribs = GetIEditor()->GetSourceControl()->GetFileAttributes(path);

		if ((fileAttribs & SCC_FILE_ATTRIBUTE_READONLY) && (fileAttribs & SCC_FILE_ATTRIBUTE_MANAGED))
		{
			AfxMessageBox("Error. File is ReadOnly.", MB_OK|MB_ICONQUESTION);
			return;
		}
	}		

	XmlNodeRef node = gEnv->pSystem->LoadXmlFile(path);
	if (!node)
		return;

	if (XmlNodeRef geom = node->findChild("geometry"))
	{
		IItemEdit* edit = m_pItem->GetIItemEdit();
		XmlNodeRef fp = geom->findChild("firstperson");
		if(!fp)
			fp = geom->newChild("firstperson");
		if (fp)
		{
			fp->setAttr("offset", string().Format("%.5f,%.5f,%.5f", m_pos_x.GetValue(), m_pos_y.GetValue(), m_pos_z.GetValue()));
			fp->setAttr("angles", string().Format("%.3f,%.3f,%.3f", m_rot_yw.GetValue(), m_rot_pt.GetValue(), m_rot_rl.GetValue()));
			fp->setAttr("fov", m_fov.GetValue());
			fp->setAttr("name", edit->GetSlotGeometry(WEAPON_SLOT_FP));
		}
		XmlNodeRef tp = geom->findChild("thirdperson");
		if(!tp && *edit->GetSlotGeometry(WEAPON_SLOT_TP))
			tp = geom->newChild("thirdperson");
		if(tp)
		{
			tp->setAttr("name", edit->GetSlotGeometry(WEAPON_SLOT_TP));
		}
		
		XmlNodeRef aux = geom->findChild("aux");
		if(!aux && *edit->GetSlotGeometry(WEAPON_SLOT_AUX))
			aux = geom->newChild("aux");
		if(aux)
		{
			aux->setAttr("name", edit->GetSlotGeometry(WEAPON_SLOT_AUX));
		}
	}
	if (XmlNodeRef sockets = node->findChild("sockets"))
	{
		std::vector< std::pair<string, int> > socketFlags;
		socketFlags.reserve(sockets->getChildCount());
		for (int i=0; i<sockets->getChildCount(); ++i)
		{
			CString name = "";
			sockets->getChild(i)->getAttr("name", name);
			int canBeEmpty = 1;
			sockets->getChild(i)->getAttr("can_be_empty", canBeEmpty);
			socketFlags.push_back(std::pair<string, int>(name.GetBuffer(), canBeEmpty));
		}
		sockets->removeAllChilds();

		int numSockets = m_pItem->GetIItemEdit()->GetSocketsCount();
		for (int i=0; i<numSockets; ++i)
		{
			if (m_pItem->GetIItemEdit()->GetSocketParentSlot(i))
				break;

			XmlNodeRef socket = sockets->newChild("socket");
			socket->setAttr("name", socketFlags[i].first);
			socket->setAttr("can_be_empty", socketFlags[i].second);
			const char* bone = m_pItem->GetIItemEdit()->GetSocketBone(i);
			if (bone && bone[0] != '\0')
				socket->setAttr("attachment", bone);
			
			int numTypes = m_pItem->GetIItemEdit()->GetSocketTypesCount(i);
			for (int j=0; j<numTypes; ++j)
			{
				XmlNodeRef support = socket->newChild("support");
				support->setAttr("type", m_pItem->GetIItemEdit()->GetSocketType(i, j));
				QuatT pos = m_pItem->GetIItemEdit()->GetHelperPos(i, j, true);
				Ang3 ang(pos.q);
				ang = RAD2DEG(ang);
				support->setAttr("offset", string().Format("%.5f,%.5f,%.5f", pos.t.x, pos.t.y, pos.t.z));
				support->setAttr("angles", string().Format("%.3f,%.3f,%.3f", ang.x, ang.y, ang.z));
			}
		}
	}	
	if (XmlNodeRef types = node->findChild("types"))
	{
		types->removeAllChilds();

		int numPins = m_pItem->GetIItemEdit()->GetPinHelpersCount();
		for (int i=0; i<numPins; ++i)
		{
			XmlNodeRef pin = types->newChild("type");
			pin->setAttr("name", m_pItem->GetIItemEdit()->GetPinHelperName(i));
			QuatT pos = m_pItem->GetIItemEdit()->GetPinHelperPos(i);
			Ang3 ang(pos.q);
			ang = RAD2DEG(ang);
			pin->setAttr("offset", string().Format("%.5f,%.5f,%.5f", pos.t.x, pos.t.y, pos.t.z));
			pin->setAttr("angles", string().Format("%.3f,%.3f,%.3f", ang.x, ang.y, ang.z));
		}
	}
	if (XmlNodeRef helpers = node->findChild("boneAttachments"))
	{
		IItemEdit::SItemHelper hlpr;
		IItemEdit* item = m_pItem->GetIItemEdit();
		for(int i=0;i<helpers->getChildCount(); ++i)
		{
			XmlNodeRef c = helpers->getChild(i);
			const char *name = c->getAttr("name");
			const char *target = c->getAttr("target");
			int slot = -1;
			if(!strcmp(target, "firstperson"))
			{
				slot = WEAPON_SLOT_FP;
			}
			else if(!strcmp(target, "thirdperson"))
			{
				slot = WEAPON_SLOT_TP;
			}
			if(slot == -1)
				continue;
			for(int j=0; j<item->GetBoneHelperCount(); ++j)
			{
				if(!item->GetBoneHelper(j, hlpr))
					break;
				if(hlpr.slot == slot && !strcmp(hlpr.name, name))
				{
					c->setAttr("bone", hlpr.bone);
				}
			}
		}
	}
	node->saveToFile(path);
	m_hasChanges = false;
}

void CWeaponEditorSettingsDialog::OnAddAttachment()
{
	if (!m_pItem)
		return;

	CString filepath;
	CString initDir(PathUtil::GetGameFolder()+"\\Items\\Accessories");
	if (!CFileUtil::SelectSingleFile( EFILE_TYPE_ANY, filepath, "Weapon Attachment Files (*.xml)|*.xml", initDir ))
		return;

	CString name = "";
	CString className = "";
	if (XmlNodeRef node = gEnv->pSystem->LoadXmlFile(filepath))
	{
		name = node->getAttr("name");
		className = node->getAttr("class");
	}

	if (name == "" || m_pItem->GetIItemBox()->HasAccessoryParams(name))
		return;

	if(className != "K01_Item" )
	{
		AfxMessageBox("Old weapon system files are not supported.");
		return;
	}

	m_pItem->GetIItemBox()->AttachAccessory(name, true, false, false);
	if (m_pItem->GetIItemBox()->HasAccessoryParams(name))
	{
		m_attachmentsList.AddString(name);
	}
	else
	{
		SEntitySpawnParams params;
		params.pClass = gEnv->pEntitySystem->GetClassRegistry()->FindClass(name);
		params.sName = name;
		params.nFlags = ENTITY_FLAG_NO_PROXIMITY | ENTITY_FLAG_CASTSHADOW;

		if (!params.pClass)
			return;

		EntityId accId=0;
		if (IEntity *pEntity = gEnv->pEntitySystem->SpawnEntity(params))
			accId=pEntity->GetId();

		IGameFramework* pGameFramework = gEnv->pGame->GetIGameFramework();
		IItemSystem* pItemSystem = pGameFramework->GetIItemSystem();
		if (accId)
		{
			CleanUpAttachedEntity();
			m_pAttachedItem = pItemSystem->GetItem(accId);
		}

		bool isSockedMatched = false;

		int numPins = m_pItem->GetIItemEdit()->GetPinHelpersCount();
		for (int i=0; i<numPins; ++i)
		{
			const char* pinName = m_pItem->GetIItemEdit()->GetPinHelperName(i);
			
			int numSockets = m_pAttachedItem->GetIItemEdit()->GetSocketsCount();
			for (int iSocket=0; iSocket<numSockets; ++iSocket)
			{
				if (m_pAttachedItem->GetIItemEdit()->GetSocketParentSlot(iSocket))
					continue;
				int numTypes = m_pAttachedItem->GetIItemEdit()->GetSocketTypesCount(iSocket);
				for (int j=0; j<numTypes; ++j)
				{
					const char* socketType = m_pAttachedItem->GetIItemEdit()->GetSocketType(iSocket, j);
					if (!stricmp(socketType, pinName))
					{
						m_attachedSocketId = iSocket;
						m_attachedSocketTypeId = j;
						m_attachedId = i;
						m_pAttachedItem->Select(true);
						m_pAttachedItem->GetEntity()->SetWorldTM(Matrix34(IDENTITY));
						if (ICharacterInstance* pChar = m_pAttachedItem->GetEntity()->GetCharacter(0))
						{
							ISkeletonPose* pSkeleton = pChar->GetISkeletonPose();
							//pSkeleton->SetDefaultPose();
						}							
						isSockedMatched = true;
						break;
					}
				}
			}

			if (isSockedMatched)
			{
				m_attachmentsList.AddString(name);
				break;
			}
		}
		if (!isSockedMatched)
			CleanUpAttachedEntity();
	}

	UpdateAttachments();

	int32 idx = m_attachmentsList.FindString(-1, name);
	m_attachmentsList.SetCurSel( idx );

	UpdateHelpers();
}

const Matrix34& CWeaponEditorSettingsDialog::GetAttachedItemTM() const
{
	static Matrix34 res;
	res.SetIdentity();

	if (m_pItem && m_pAttachedItem)
	{
		res = Matrix34(m_pItem->GetIItemEdit()->GetPinHelperPos(m_attachedId));
		res = res * Matrix34(m_pAttachedItem->GetIItemEdit()->GetHelperPos(m_attachedSocketId, m_attachedSocketTypeId)).GetInverted();
	}

	return res;
}

void CWeaponEditorSettingsDialog::UpdateAttachments()
{
	for (int i=0; i<m_attachmentsList.GetCount();)
	{
		CString att;
		m_attachmentsList.GetText(i, att);
		if (!m_pItem->GetIItemBox()->HasAccessoryParams(att))
			if (m_pAttachedItem && att == m_pAttachedItem->GetEntity()->GetClass()->GetName())
			{
				++i;
				continue;
			}
			else
				m_attachmentsList.DeleteString(i);
		else
			++i;
	}
}

void CWeaponEditorSettingsDialog::OnRemoveAttachment()
{
	if (!m_pItem)
		return;

	int sel = m_attachmentsList.GetCurSel();
	if (sel < 0)
		return;

	CString name;
	m_attachmentsList.GetText(sel, name);
	m_pItem->GetIItemBox()->AttachAccessory(name, false, false, false);
	UpdateAttachments();
	UpdateHelpers();
}

void CWeaponEditorSettingsDialog::OnChange()
{
	if (!m_hasChanges)
	{
		if (GetIEditor()->GetSourceControl())
		{
			const char* path = GetFilePath();
			if (!path)
				return;
			uint32 fileAttribs = GetIEditor()->GetSourceControl()->GetFileAttributes(path);

			if (fileAttribs & SCC_FILE_ATTRIBUTE_MANAGED)
			{
				if ((fileAttribs & SCC_FILE_ATTRIBUTE_INPAK))
				{
					if (IDYES == AfxMessageBox("Get latest version from source control?", MB_YESNO|MB_ICONQUESTION ))
					{
						GetIEditor()->GetSourceControl()->GetLatestVersion(path);
						m_editor->LoadFile(path);
					}
				}

				if (!(fileAttribs & SCC_FILE_ATTRIBUTE_CHECKEDOUT))
				{
					if (IDYES == AfxMessageBox("File is under source control. Proceed with Check Out?", MB_YESNO|MB_ICONQUESTION ))
					{
						if (!GetIEditor()->GetSourceControl()->GetLatestVersion(path, GETLATEST_ONLY_CHECK))
							if (IDYES == AfxMessageBox("Get latest version from source control?", MB_YESNO|MB_ICONQUESTION ))
							{
								GetIEditor()->GetSourceControl()->GetLatestVersion(path);
								m_editor->LoadFile(path);
							}
						GetIEditor()->GetSourceControl()->CheckOut(path);						
					}
				}
			}
			
		}		
	}
	m_hasChanges = true;
}

void CWeaponEditorSettingsDialog::FillHelpersBox()
{
	m_subObjects.ResetContent();
	
	if (IStatObj* pStatObj = m_pItem->GetEntity()->GetStatObj(WEAPON_SLOT_FP))
	{
		int num = pStatObj->GetSubObjectCount();
		for (int i=0; i<num; ++i)
			if (IStatObj::SSubObject* pObj = pStatObj->GetSubObject(i))
				m_subObjects.AddString(pObj->name);
	}


	int i,j;
	const char* bone = ""; 
	if(GetSelectedHelper(i,j))
	{
		IItemEdit* item = m_pItem->GetIItemEdit();
		const char* attachment = item->GetSocketBone(i);
		int i=0;
		IItemEdit::SItemHelper hlpr;
		while(item->GetBoneHelper(i++, hlpr))
		{
			if(hlpr.slot == WEAPON_SLOT_FP && !strcmp(hlpr.name, attachment))
			{
				bone = hlpr.bone;
				break;
			}
		}
	}

	assert(bone);

	if (ICharacterInstance* pChar = m_pItem->GetEntity()->GetCharacter(WEAPON_SLOT_FP))
	{
		int i = 0;
		ISkeletonPose* pSkeleton = pChar->GetISkeletonPose();
		int32 num = pSkeleton->GetJointCount();
		int32 pId = pSkeleton->GetJointIDByName(bone);
		for (int32 i=0; i<num; ++i)
			if (pSkeleton->GetParentIDByID(i) == pId)
				m_subObjects.AddString(pSkeleton->GetJointNameByID(i));
	}
}

void CWeaponEditorSettingsDialog::FillBoneHelpersBox()
{
	m_boneHelpersBone.ResetContent();
	int idx = m_boneHelpersList.GetCurSel();
	if(idx != -1)
	{
		//m_boneHelpersBone.Enable(true);
		if (ICharacterInstance* pChar = m_pItem->GetEntity()->GetCharacter(WEAPON_SLOT_FP))
		{
			ISkeletonPose* pSkeleton = pChar->GetISkeletonPose();
			int num = pSkeleton->GetJointCount();
			for (int32 i=0; i<num; ++i)
			{
				m_boneHelpersBone.AddString(pSkeleton->GetJointNameByID(i));
			}
		}
		int id = m_boneHelpersList.GetItemData(idx);
		IItemEdit::SItemHelper hlp;
		if(m_pItem->GetIItemEdit()->GetBoneHelper(id,hlp))
		{
			m_boneHelpersBone.SelectString(0, hlp.bone);
		}
	}
}

void CWeaponEditorSettingsDialog::FillBoneHelpersList()
{
	m_boneHelpersList.ResetContent();
	int i=0;
	IItemEdit::SItemHelper hlpr;
	IItemEdit *item = m_pItem->GetIItemEdit();
	while(item->GetBoneHelper(i, hlpr))
	{
		int id = m_boneHelpersList.AddString(hlpr.name);
		m_boneHelpersList.SetItemData(id, i);
		++i;
	}
	FillBoneHelpersBox();
}

static void SetTextScrollRight(CEdit& eb, const char* text)
{
	eb.SetWindowText(text);
	int caretPos = strlen(text);
	eb.SetSel(caretPos, caretPos, FALSE);
}

void CWeaponEditorSettingsDialog::FillGeometry()
{
	if(!m_pItem)
		return;

	IItemEdit* edit = m_pItem->GetIItemEdit();
	SetTextScrollRight(m_geometryFP, edit->GetSlotGeometry(WEAPON_SLOT_FP));
	SetTextScrollRight(m_geometryTP, edit->GetSlotGeometry(WEAPON_SLOT_TP));
	SetTextScrollRight(m_geometryAUX, edit->GetSlotGeometry(WEAPON_SLOT_AUX));
}

bool CWeaponEditorSettingsDialog::GetSelectedHelper(int& idx, int& subidx)const
{
	if (!m_pItem || m_helpersList.GetCurSel() < 0)
		return false;

	int id = m_helpersList.GetItemData(m_helpersList.GetCurSel());
	const SEditHelper& eh = m_editHelpers[id];

	if(eh.pin)
		return false;

	idx = eh.idx;
	subidx = eh.sub_idx;
	return true;
}

bool CWeaponEditorSettingsDialog::GetSelectedPin(int &idx)const
{
	if (!m_pItem || m_helpersList.GetCurSel() < 0)
		return false;

	int id = m_helpersList.GetItemData(m_helpersList.GetCurSel());
	const SEditHelper& eh = m_editHelpers[id];

	if(!eh.pin)
		return false;

	idx = eh.idx;
	return true;
}

void CWeaponEditorSettingsDialog::CleanUpAttachedEntity()
{
	if(m_pAttachedItem)
	{
		m_pAttachedItem->Select(false);
		gEnv->pEntitySystem->RemoveEntity(m_pAttachedItem->GetEntityId(),true);
		m_pAttachedItem = 0;
	}
}

void CWeaponEditorSettingsDialog::CheckViewMode()
{
	bool isFP = IDC_WEAPON_FP == GetCheckedRadioButton(IDC_WEAPON_FP, IDC_WEAPON_TP);
	
	m_OffsetGroup->SetVisible(isFP);
	m_HelpersGroup->SetVisible(isFP);

	if (m_pItem)
	{
		if (isFP)
		{
			m_pItem->GetIItemEdit()->SetViewMode(eIVM_FirstPerson);
		}
		else
		{
			m_pItem->GetIItemEdit()->SetViewMode(eIVM_ThirdPerson);
		}
	}	
}

void CWeaponEditorSettingsDialog::OnBnClickedWeaponFp()
{	
	CheckViewMode();  
}

void CWeaponEditorSettingsDialog::OnBnClickedWeaponTp()
{
	CheckViewMode(); 	
}
