////////////////////////////////////////////////////////////////////////////
//
//  Crytek Engine Source File.
//  Copyright (C), Crytek Studios, 2002.
// -------------------------------------------------------------------------
//  File name:   fileutil.cpp
//  Version:     v1.00
//  Created:     13/9/2002 by Timur.
//  Compilers:   Visual Studio.NET
//  Description: 
// -------------------------------------------------------------------------
//  History:
//
////////////////////////////////////////////////////////////////////////////

#include "StdAfx.h"
#include "FileUtil.h"
#include "LogFile.h"
#include "Util/PathUtil.h"

#include <CryFile.h>
#include <io.h>
#include <time.h>

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
// Get directory contents.
//////////////////////////////////////////////////////////////////////////
inline bool ScanDirectoryFiles( const CString &root,const CString &path,const CString &fileSpec,std::vector<CFileUtil::FileDesc> &files )
{
	bool anyFound = false;
	CString dir = Path::AddBackslash(root + path);

	CString findFilter = Path::Make(dir,fileSpec);
	ICryPak *pIPak = gEnv->pSystem->GetIPak();

	// Add all directories.
	_finddata_t fd;
	intptr_t fhandle;

	fhandle = pIPak->FindFirst( findFilter,&fd );
	if (fhandle != -1)
	{
		do {
			// Skip back folders.
			if (fd.name[0] == '.')
				continue;
			if (fd.attrib & _A_SUBDIR) // skip sub directories.
				continue;

			anyFound = true;

			CFileUtil::FileDesc file;
			file.filename = path + fd.name;
			file.attrib = fd.attrib;
			file.size = fd.size;
			file.time_access = fd.time_access;
			file.time_create = fd.time_create;
			file.time_write = fd.time_write;

			files.push_back( file );
		} while (pIPak->FindNext( fhandle,&fd ) == 0);
		pIPak->FindClose(fhandle);
	}

	/*
	CFileFind finder;
	BOOL bWorking = finder.FindFile( Path::Make(dir,fileSpec) );
	while (bWorking)
	{
	bWorking = finder.FindNextFile();

	if (finder.IsDots())
	continue;

	if (!finder.IsDirectory())
	{
	anyFound = true;

	CFileUtil::FileDesc fd;
	fd.filename = dir + finder.GetFileName();
	fd.nFileSize = finder.GetLength();

	finder.GetCreationTime( &fd.ftCreationTime );
	finder.GetLastAccessTime( &fd.ftLastAccessTime );
	finder.GetLastWriteTime( &fd.ftLastWriteTime );

	fd.dwFileAttributes = 0;
	if (finder.IsArchived())
	fd.dwFileAttributes |= FILE_ATTRIBUTE_ARCHIVE;
	if (finder.IsCompressed())
	fd.dwFileAttributes |= FILE_ATTRIBUTE_COMPRESSED;
	if (finder.IsNormal())
	fd.dwFileAttributes = FILE_ATTRIBUTE_NORMAL;
	if (finder.IsHidden())
	fd.dwFileAttributes = FILE_ATTRIBUTE_HIDDEN;
	if (finder.IsReadOnly())
	fd.dwFileAttributes = FILE_ATTRIBUTE_READONLY;
	if (finder.IsSystem())
	fd.dwFileAttributes = FILE_ATTRIBUTE_SYSTEM;
	if (finder.IsTemporary())
	fd.dwFileAttributes = FILE_ATTRIBUTE_TEMPORARY;

	files.push_back(fd);
	}
	}
	*/

	return anyFound;
}

//////////////////////////////////////////////////////////////////////////
// Get directory contents.
//////////////////////////////////////////////////////////////////////////
inline bool ScanDirectoryRecursive( const CString &root,const CString &path,const CString &fileSpec,std::vector<CFileUtil::FileDesc> &files, bool recursive )
{
	bool anyFound = false;
	CString dir = Path::AddBackslash(root + path);

	if (ScanDirectoryFiles( root,Path::AddBackslash(path),fileSpec,files ))
		anyFound = true;

	if (recursive)
	{
		/*
		CFileFind finder;
		BOOL bWorking = finder.FindFile( Path::Make(dir,"*.*") );
		while (bWorking)
		{
		bWorking = finder.FindNextFile();

		if (finder.IsDots())
		continue;

		if (finder.IsDirectory())
		{
		// Scan directory.
		if (ScanDirectoryRecursive( root,Path::AddBackslash(path+finder.GetFileName()),fileSpec,files,recursive ))
		anyFound = true;
		}
		}
		*/

		ICryPak *pIPak = gEnv->pSystem->GetIPak();

		// Add all directories.
		_finddata_t fd;
		intptr_t fhandle;

		fhandle = pIPak->FindFirst( Path::Make(dir,"*.*"),&fd );
		if (fhandle != -1)
		{
			do {
				// Skip back folders.
				if (fd.name[0] == '.')
					continue;
				if (!(fd.attrib & _A_SUBDIR)) // skip not directories.
					continue;

				// Scan directory.
				if (ScanDirectoryRecursive( root,Path::AddBackslash(path + fd.name),fileSpec,files,recursive ))
					anyFound = true;

			} while (pIPak->FindNext( fhandle,&fd ) == 0);
			pIPak->FindClose(fhandle);
		}
	}

	return anyFound;
}

//////////////////////////////////////////////////////////////////////////
bool CFileUtil::ScanDirectory( const CString &path,const CString &file,std::vector<FileDesc> &files, bool recursive )
{
	CString fileSpec = Path::GetFile(file);
	CString localPath = Path::GetPath(file);
	return ScanDirectoryRecursive(Path::AddBackslash(path),localPath,fileSpec,files,recursive );
}

/*
bool CFileUtil::ScanDirectory( const CString &startDirectory,const CString &searchPath,const CString &fileSpec,FileArray &files, bool recursive=true )
{
return ScanDirectoryRecursive(startDirectory,SearchPath,file,files,recursive );
}
*/
//////////////////////////////////////////////////////////////////////////
// Create new directory, check if directory already exist.
static bool CheckAndCreateDirectory( const char *path )
{
	WIN32_FIND_DATA FindFileData;

	HANDLE hFind = FindFirstFile( path,&FindFileData );
	if (hFind == INVALID_HANDLE_VALUE) {
		return ::CreateDirectory( path,NULL ) == TRUE;
	} else {
		DWORD attr = FindFileData.dwFileAttributes;
		FindClose(hFind);
		if (attr & FILE_ATTRIBUTE_DIRECTORY) {
			return true;
		}
	}
	return false;
}

//////////////////////////////////////////////////////////////////////////
void CFileUtil::CreateDirectory( const char *directory )
{
	CString path = directory;
	CString dir;
	bool res = CheckAndCreateDirectory( path );
	if (!res)
	{
		int iStart = 0;
		CString token = TokenizeString(path,"\\/",iStart );
		dir = token;
		while (token != "")
		{
			CheckAndCreateDirectory( dir );
			token = TokenizeString(path,"\\/",iStart );
			dir += CString("\\") + token;
		}
	}
}

//////////////////////////////////////////////////////////////////////////
void CFileUtil::BackupFile( const char *filename )
{
	// Make a backup of previous file.
	bool makeBackup = true;

	CString bakFilename = Path::ReplaceExtension( filename,"bak" );

	{
		// Check if backup needed.
		CFile bak;
		if (bak.Open( filename,CFile::modeRead ))
		{
			if (bak.GetLength() <= 0)
				makeBackup = false;
		}
		else
			makeBackup = false;
	}
	// Backup the backup..
	if (makeBackup)
	{
		MoveFileEx( bakFilename,Path::ReplaceExtension( bakFilename,"bak2" ),MOVEFILE_REPLACE_EXISTING|MOVEFILE_WRITE_THROUGH );
		SetFileAttributes( filename,FILE_ATTRIBUTE_NORMAL );
		MoveFileEx( filename,bakFilename,MOVEFILE_REPLACE_EXISTING|MOVEFILE_WRITE_THROUGH );
	}
}

//////////////////////////////////////////////////////////////////////////
void CFileUtil::BackupFileDated( const char *filename, bool bUseBackupSubDirectory /*=false*/ )
{
	bool makeBackup = true;
	{
		// Check if backup needed.
		CFile bak;
		if (bak.Open( filename,CFile::modeRead ))
		{
			if (bak.GetLength() <= 0)
				makeBackup = false;
		}
		else
			makeBackup = false;
	}

	if ( makeBackup )
	{
		// Generate new filename
		time_t ltime;
		time( &ltime );
		tm *today = localtime( &ltime );

		char sTemp[128];
		strftime( sTemp, sizeof(sTemp), ".%Y%m%d.%H%M%S.", today);		
		CString bakFilename = Path::RemoveExtension(filename) + sTemp + Path::GetExt(filename);

		if ( bUseBackupSubDirectory )
		{
			CString sBackupPath = Path::ToUnixPath( Path::GetPath( filename ) ) + CString( "/backups" );
			CFileUtil::CreateDirectory( sBackupPath );
			bakFilename = sBackupPath + CString("/") + Path::GetFile( bakFilename );
		}

		// Do the backup
		SetFileAttributes( filename,FILE_ATTRIBUTE_NORMAL );
		MoveFileEx( filename,bakFilename,MOVEFILE_REPLACE_EXISTING|MOVEFILE_WRITE_THROUGH );	
	}
}
//////////////////////////////////////////////////////////////////////////
bool CFileUtil::Deltree(const char *szFolder,bool bRecurse)
{
	bool							boRemoveResult(false);

	__finddata64_t		fd;
	string						filespec = szFolder;

	filespec += "*.*";

	intptr_t hfil = 0;
	if ((hfil = _findfirst64(filespec.c_str(), &fd)) == -1)
	{
		return false;
	}

	do
	{
		if (fd.attrib & _A_SUBDIR)
		{
			CString name = fd.name;

			if ((name != ".") && (name != ".."))
			{
				if (bRecurse)
				{
					name = szFolder;
					name += fd.name;
					name += "/";

					Deltree(name.GetBuffer(), bRecurse);
				}
			}
		}
		else
		{
			CString name = szFolder;

			name += fd.name;

			CFileUtil::DeleteFile(name);
		}

	} while(!_findnext64(hfil, &fd));

	_findclose(hfil);

	boRemoveResult=CFileUtil::RemoveDirectory(szFolder);

	return boRemoveResult;
}
//////////////////////////////////////////////////////////////////////////
bool   CFileUtil::Exists(const CString& strPath,bool boDirectory)
{
	ICryPak *pIPak = gEnv->pSystem->GetIPak();
	intptr_t				nFindHandle(NULL);
	_finddata_t			stFoundData;
	bool						boIsDirectory(false);

	memset(&stFoundData,0,sizeof(_finddata_t));
	nFindHandle=pIPak->FindFirst(strPath,&stFoundData);
	// If it found nothing, no matter if it is a file or directory, it was not found.
	if (nFindHandle==-1)
	{
		return false;
	}
	pIPak->FindClose(nFindHandle);

	if (stFoundData.attrib&_A_SUBDIR)
	{
		boIsDirectory=true;
	}

	// If we are seeking directories...
	if (boDirectory)
	{
		// The return value will tell us if the found element is a directory.
		return boIsDirectory;
	}
	else
	{
		// If we are not seeking directories...
		// We return true if the found element is not a directory.
		return !boIsDirectory;
	}
}
//////////////////////////////////////////////////////////////////////////
bool   CFileUtil::FileExists(const CString& strFilePath)
{
	return Exists(strFilePath,false);
}
//////////////////////////////////////////////////////////////////////////
bool   CFileUtil::PathExists(const CString& strPath)
{
	return Exists(strPath,true);
}
//////////////////////////////////////////////////////////////////////////
bool   CFileUtil::CreatePath(const CString& strPath)
{
	CString									strDriveLetter;
	CString									strDirectory;
	CString									strFilename;
	CString									strExtension;
	CString									strCurrentDirectoryPath;
	std::vector<CString>		cstrDirectoryQueue;
	size_t									nCurrentPathQueue(0);
	size_t									nTotalPathQueueElements(0);
	BOOL										bnLastDirectoryWasCreated(FALSE);

	if (PathExists(strPath))
	{
		return true;
	}

	Path::SplitPath(strPath,strDriveLetter,strDirectory,strFilename,strExtension);
	Path::GetDirectoryQueue(strDirectory,cstrDirectoryQueue);

	if (strDriveLetter.GetLength()>0)
	{
		strCurrentDirectoryPath=strDriveLetter;
		strCurrentDirectoryPath+="\\";
	}


	nTotalPathQueueElements=cstrDirectoryQueue.size();
	for (nCurrentPathQueue=0;nCurrentPathQueue<nTotalPathQueueElements;++nCurrentPathQueue)
	{
		strCurrentDirectoryPath+=cstrDirectoryQueue[nCurrentPathQueue];
		strCurrentDirectoryPath+="\\";
		// The value which will go out of this loop is the result of the attempt to create the
		// last directory, only.
		bnLastDirectoryWasCreated=::CreateDirectory(strCurrentDirectoryPath,NULL);
	}

	if (!bnLastDirectoryWasCreated)
	{
		if (ERROR_ALREADY_EXISTS!=GetLastError())
		{
			return false;
		}
	}

	return true;
}
//////////////////////////////////////////////////////////////////////////
bool   CFileUtil::DeleteFile(const CString& strPath)
{
	BOOL bnAttributeChange(FALSE);
	BOOL bnFileDeletion(FALSE);

	bnAttributeChange=SetFileAttributes(strPath,FILE_ATTRIBUTE_NORMAL);
	bnFileDeletion=::DeleteFile(strPath);

	return (bnFileDeletion!=FALSE);
}
//////////////////////////////////////////////////////////////////////////
bool	CFileUtil::RemoveDirectory(const CString& strPath)
{
	BOOL bnRemoveResult(FALSE);
	BOOL bnAttributeChange(FALSE);

	bnAttributeChange=SetFileAttributes(strPath,FILE_ATTRIBUTE_NORMAL);
	bnRemoveResult=::RemoveDirectory(strPath);

	return (bnRemoveResult!=FALSE);
}
//////////////////////////////////////////////////////////////////////////
