/*************************************************************************
Crytek Source File.
Copyright (C), Crytek Studios, 2001-2004.
-------------------------------------------------------------------------
$Id: AssetModelItem.h,v 1.0 2009/04/15 11:00:00 PauloZaffari Exp wwwrun $
$DateTime$
Description: Source file for the class implementing IAssetDisplayItem
interface. It declares the headers of the actual used 
functions.
-------------------------------------------------------------------------
History:
- 15:04:2009   11:00 : Created by Paulo Zaffari

*************************************************************************/
#include "StdAfx.h"
#include "AssetModelItem.h"

#include "AssetModelDatabase.h"

#include "Util/PathUtil.h"

//////////////////////////////////////////////////////////////////////////
CAssetModelItem::CAssetModelItem():
m_boMustUpdateAssetInfo(true),
m_strFilename(""),
m_strExtension(""),
m_strRelativePath(""),
m_boUsedInLevel(false),
m_nFileSize(0),
m_boVisible(true),
m_boBitmapIsChached(false),
m_boIsSelected(false),
m_oDrawingRectangle(0,0,512,512),
m_piOwnerDatabase(NULL),
m_ref(1),




m_camTarget(0,0,0),
m_camRadius(1),
//m_camera,
//m_aabb,
m_object(NULL),
m_renderer(gEnv->pRenderer),

m_bGrid(true),
m_bAxis(true),

m_clearColor(0.5f,0.5f,0.5f),

//m_pCurrentMaterial(NULL)

m_bRotate(false),
m_rotateAngle(0),
m_bShowObject(true),
m_fov(60)
{
	m_aabb.Reset();

	CDLight l;
	l.m_Origin = Vec3(100,-100,100);
	float L = 1.0f;
	l.SetLightColor(ColorF(L,L,L,1));
	l.m_SpecMult = 1.0f;
	l.m_fRadius = 1000;
	//l.m_fStartRadius = 0;
	//l.m_fEndRadius = 1000;
	l.m_Flags |= DLF_POINT;
	m_lights.push_back( l );
}
//////////////////////////////////////////////////////////////////////////
CAssetModelItem::~CAssetModelItem()
{
	FreeData();
}
//////////////////////////////////////////////////////////////////////////
void CAssetModelItem::FreeData()
{
	if (m_boBitmapIsChached)
	{
		UnCacheAsset();
		m_boBitmapIsChached=false;		
	}	
}
//////////////////////////////////////////////////////////////////////////
IAssetDisplayDatabase*	CAssetModelItem::GetOwnerDisplayDatabse()
{
	return m_piOwnerDatabase;
}
//////////////////////////////////////////////////////////////////////////
void CAssetModelItem::SetOwnerDisplayDatabase(IAssetDisplayDatabase* piOwnerDisplayDatabase)
{
	m_piOwnerDatabase=piOwnerDisplayDatabase;
}
//////////////////////////////////////////////////////////////////////////
void		CAssetModelItem::GetFilename(string& rstrFilename) const 
{ 
	rstrFilename=m_strFilename; 
}
//////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////
void		CAssetModelItem::SetFilename(const string& val) 
{ 
	CString strSourceString(val);
	CString strDriveLetter;
	CString strDirectory;
	CString strFilename;
	CString strExtension;
	string	strResultingExtension;

	Path::SplitPath(strSourceString,strDriveLetter,strDirectory,strFilename,strExtension);

	m_strFilename = strFilename; 
	m_strFilename += strExtension;

	if (strExtension.GetLength())
	{
		strResultingExtension=strExtension;
		SetExtension(strResultingExtension.substr(1));
	}	
	SetRelativePath((string)(strDriveLetter+strDirectory));
}
//////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////
void		CAssetModelItem::GetExtension(string& rstrExtension) const 
{ 
	rstrExtension=m_strExtension; 
}
//////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////
void		CAssetModelItem::SetExtension(const string& val) 
{ 
	if (m_strExtension!=val)
	{
		m_boMustUpdateAssetInfo=true;
	}
	m_strExtension = val; 
}
//////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////
void 		CAssetModelItem::GetRelativePath(string& rstrRelativePath) const 
{ 
	rstrRelativePath=m_strRelativePath; 
}
//////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////
void		CAssetModelItem::SetRelativePath(const string& val) 
{ 
	if (m_strRelativePath!=val)
	{
		m_boMustUpdateAssetInfo=true;
	}

	m_strRelativePath = val; 
}
//////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////
bool		CAssetModelItem::GetUsedInLevel() const 
{ 
	return m_boUsedInLevel; 
}
//////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////
void		CAssetModelItem::SetUsedInLevel(bool val) 
{ 
	if (m_boUsedInLevel!=val)
	{
		m_boMustUpdateAssetInfo=true;
	}

	m_boUsedInLevel = val; 
}
//////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////
long long CAssetModelItem::GetFileSize() const
{ 
	return m_nFileSize; 
}
//////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////
void CAssetModelItem::SetFileSize(long long val )
{
	if (m_nFileSize!=val)
	{
		m_boMustUpdateAssetInfo=true;
	}
	m_nFileSize = val; 
}
//////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////
bool	CAssetModelItem::GetVisibleStatus() const
{
	return m_boVisible;
}
//////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////
void	CAssetModelItem::SetVisibleStatus(bool val)
{
	m_boVisible=val;
}
//////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////
bool	CAssetModelItem::GetSelectedStatus()
{ 
	return m_boIsSelected; 
}
//////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////
void	CAssetModelItem::SetSelectedStatus(bool boIsSelected) 
{ 
	m_boIsSelected = boIsSelected; 
}
//////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////
CRect& CAssetModelItem::GetDrawingRectangle(CRect& rstDrawingRectangle) const
{ 
	rstDrawingRectangle=m_oDrawingRectangle;
	return rstDrawingRectangle; 
}
//////////////////////////////////////////////////////////////////////////
void	CAssetModelItem::SetDrawingRectangle(const CRect& crstDrawingRectangle)
{
	// When this changes we may need to recache the asset, in case the size
	// changes...
	assert(false);
	m_oDrawingRectangle=crstDrawingRectangle;
}
//////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////
bool CAssetModelItem::GetCachedStatus()
{
	return m_boBitmapIsChached;
}
//////////////////////////////////////////////////////////////////////////
bool			CAssetModelItem::CacheAsset(const XmlNodeRef& xmlCacheOptions)
{
	if (m_boBitmapIsChached)
	{
		return true;
	}

	// Cache the asset here (cache means load into a bitmap).
	if (m_boUsedInLevel)
	{
		assert(false);
	}
	else
	{
		IStatObj::SSubObject*	pSubObject(NULL);

		//m_object=gEnv->p3DEngine->LoadStatObj(m_strRelativePath+m_strFilename,NULL,&pSubObject,false,ELoadingFlagsPreviewMode);
		m_object=gEnv->p3DEngine->LoadStatObj(m_strRelativePath+m_strFilename,NULL,&pSubObject,true,IMaterialManager::ELoadingFlagsPreviewMode);
		//m_object=gEnv->p3DEngine->LoadStatObj(m_strRelativePath+m_strFilename,NULL,&pSubObject,true,0);
		//m_object=gEnv->p3DEngine->LoadStatObj(m_strRelativePath+m_strFilename,NULL,&pSubObject,false,0);
		if (!m_object)
		{
			return false;
		}
		m_object->AddRef();
		m_aabb.min=m_object->GetBoxMin();
		m_aabb.max=m_object->GetBoxMax();
	}
	//SetCameraLookAt( 2.0f,Vec3(1,1,-0.5));
	//SetCameraLookAt( 1.1f,Vec3(0,1.0f,0));
	CalculateCameraPosition();

	m_boBitmapIsChached=true;

	return m_boBitmapIsChached;
}
//////////////////////////////////////////////////////////////////////////
bool      CAssetModelItem::UnCacheAsset()
{
	BOOL bnReturn(FALSE);
	if (!m_boBitmapIsChached)
	{
		return true;
	}

	m_boBitmapIsChached=false;
	SAFE_RELEASE(m_object);

	return true;
}
//////////////////////////////////////////////////////////////////////////
bool	CAssetModelItem::Render(HWND hRenderWindow,const CRect& rstFullViewport,const CRect& rstViewport)
{
	if (!m_boBitmapIsChached)
	{
		return false;
	}

	//if (!m_object)
	//{
	//	IStatObj::SSubObject*	pSubObject(NULL);
	//	m_object=gEnv->p3DEngine->LoadStatObj(m_strRelativePath+m_strFilename,NULL,&pSubObject,true,ELoadingFlagsPreviewMode);
	//	if (!m_object)
	//	{
	//		return false;
	//	}
	//	m_object->AddRef();
	//	//m_object->StartStreaming(false,&m_piStream);
	//	m_aabb.min=m_object->GetBoxMin();
	//	m_aabb.max=m_object->GetBoxMax();
	//	SetCameraLookAt( 2.0f,Vec3(1,1,-0.5));
	//}
	

	CRect rc(rstViewport);

	int		nX(0);
	int		nY(0);
	int		nWidth(0);
	int		nHeight(0);

	if (rc.top<0)
	{
		rc.top=0;
	}

	if (rc.left<0)
	{
		rc.left=0;
	}

	if ((rc.bottom<0)||(rc.right<0))
	{
		return true;
	}

	if (rc.bottom>rstFullViewport.bottom)
	{
		rc.bottom=rstFullViewport.bottom;
	}

	if (rc.right>rstFullViewport.right)
	{
		rc.right=rstFullViewport.right;
	}

	CalculateCameraPosition();

	//::GetClientRect(hRenderWindow,&rc);
	//if (rc.bottom-rc.top < 2 || rc.right-rc.left < 2)
	//	return false;

	//if (!m_bContextCreated)
	//{
	//	if (!CreateContext())
	//		return false;
	//}

	//IRenderer *pr = CSystem::Instance()->SetCurrentRenderer( m_renderer );
	SetCamera( m_camera, rstViewport);

	m_renderer->GetViewport(&nX,&nY,&nWidth,&nHeight);
	m_renderer->ChangeViewport(rc.left,rc.top,rc.Width(),rc.Height());
	
	m_renderer->ClearBuffer(FRT_CLEAR, &m_clearColor);

	//m_renderer->SetCurrentContext(hRenderWindow);
	//m_renderer->ChangeViewport(0,0,rc.right,rc.bottom);
	//m_renderer->SetClearColor( Vec3(m_clearColor.r,m_clearColor.g,m_clearColor.b) );
	//m_renderer->BeginFrame();

	//m_renderer->SetCameraOffCenter( m_camera );
	m_renderer->SetCamera( m_camera );

	// Render grid. Explicitly clear color and depth buffer first
	// (otherwise ->EndEf3D() will do that and thereby clear the grid).
	//m_renderer->ClearBuffer(FRT_CLEAR, &m_clearColor);
	//if (m_bGrid || m_bAxis)
	if (false)
	{
		DrawGrid();
		//m_renderer->GetIRenderAuxGeom()->Flush(false);
	}

	// Render object.
	m_renderer->EF_StartEf();
	m_renderer->ResetToDefault();

	m_renderer->SetWireframeMode( R_SOLID_MODE );

	// Add lights.
	for (int i = 0; i < m_lights.size(); i++)
	{
		m_renderer->EF_ADDDlight( &m_lights[i] );
	}

	_smart_ptr<IMaterial> pMaterial(NULL);
	//if (m_pCurrentMaterial)
	//	pMaterial = m_pCurrentMaterial->GetMatInfo();

	SRendParams rp;
	rp.nDLightMask = 0x3;
	rp.AmbientColor = ColorF(1,1,1,1);
	rp.dwFObjFlags |= FOB_TRANS_MASK;
	rp.pMaterial = pMaterial;

	Matrix34 tm;
	tm.SetIdentity();
	rp.pMatrix = &tm;

	if (m_bRotate)
	{
		tm.SetRotationXYZ(Ang3( 0,0,m_rotateAngle ));
		m_rotateAngle += 0.1f;
	}

	if (m_bShowObject)
	{
		if (m_object)
			m_object->Render( rp );

		//if (m_entity)
		//	m_entity->Render( rp );

		//if (m_character)
		//	m_character->Render( rp, QuatTS(IDENTITY)  );

		//if (m_pEmitter)
		//{
		//	m_pEmitter->Update();
		//	m_pEmitter->Render( rp );
		//}
	}

	m_renderer->EF_EndEf3D(SHDF_SORT | SHDF_NOASYNC | SHDF_STREAM_SYNC, -1);

	// Returning the rendering to the original viewport (probably not, but well...).
	m_renderer->ChangeViewport(rstFullViewport.left,rstFullViewport.top,rstFullViewport.Width(),rstFullViewport.Height());

	//m_renderer->FlushTextMessages();

	//m_renderer->RenderDebug();
	//m_renderer->EndFrame();

	//// Restore main context.
	//m_renderer->MakeMainContextActive();

	return true;
}
//////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////
bool	CAssetModelItem::FilterAsset(const XmlNodeRef& xmlFilterOptions)
{
	bool				boNextVisibleStatus(m_piOwnerDatabase->GetVisibleStatus());
	bool				boCurrentVisibleStatus(m_boVisible);

	const char*	szFilanameFilter(NULL);

	string			strFilenameFilter;
	string			strTempString;

	if (!xmlFilterOptions)
	{
		// If the current visible flag is different of the next, we must
		// set it to the next flag.
		if (boCurrentVisibleStatus!=boNextVisibleStatus)
		{
			m_boVisible=boNextVisibleStatus;
			// Yes, the visible flag changed.
			return true;
		}	
		else
		{
			return false;
		}
	}

	// If we have something in the filename filter (and thus we must filter the filenames)...
	if (xmlFilterOptions->getAttr("Filename",&szFilanameFilter))
	{
		strFilenameFilter=szFilanameFilter;
		if (!strFilenameFilter.empty())
		{
			strTempString=m_strRelativePath;
			strTempString+=m_strFilename;
			//strTempString=m_strFilename;
			strTempString.MakeLower();
			strFilenameFilter.MakeLower();

			if (strFilenameFilter[0]!='*')
			{
				strFilenameFilter.insert(0,'*');
			}
			if (strFilenameFilter[strFilenameFilter.size()-1]!='*')
			{
				strFilenameFilter.push_back('*');
			}

			if (!PathUtil::MatchWildcard(strTempString.c_str(),strFilenameFilter.c_str()))
			{
				boNextVisibleStatus=false;
			}
		}
	}

	if (!m_piOwnerDatabase->GetVisibleStatus())
	{
		boNextVisibleStatus=false;
	}

	// If the current visible flag is different of the next, we must
	// set it to the next flag.
	if (boCurrentVisibleStatus!=boNextVisibleStatus)
	{
		m_boVisible=boNextVisibleStatus;
		// In case it's not visible, we will make it's position invalid.
		if (!boNextVisibleStatus)
		{
			// If may be required for us to keep some drawing rectangle
			// different of the actual drawing rectangle when the bitmap is
			// cached.
			assert(false);
			m_oDrawingRectangle.SetRect(-65535,-65535,-65535,-65535);
		}

		// Yes, the visible flag changed.
		return true;
	}

	//No, the visible flag didn't change.
	return false;
}
//////////////////////////////////////////////////////////////////////////
const	XmlNodeRef&  CAssetModelItem::GetAssetInfo()
{
	if (m_boMustUpdateAssetInfo)
	{
		if (!m_xmlAssetInfo)
		{
			m_xmlAssetInfo=gEnv->pSystem->CreateXmlNode("AssetInformation");
		}
		m_xmlAssetInfo->setAttr("RelativePath",m_strRelativePath);
		m_xmlAssetInfo->setAttr("Filename",m_strFilename);
		m_xmlAssetInfo->setAttr("Extension",m_strExtension);
		m_xmlAssetInfo->setAttr("FileSize",m_nFileSize);
		m_xmlAssetInfo->setAttr("Width",(int)512);
		m_xmlAssetInfo->setAttr("Height",(int)512);
		m_xmlAssetInfo->setAttr("UsedInLevel",m_boUsedInLevel);
		m_boMustUpdateAssetInfo=false;
	}
	return m_xmlAssetInfo;
}
//////////////////////////////////////////////////////////////////////////
const			XmlNodeRef&  CAssetModelItem::GetAssetDisplayInfo()
{
	if (!m_xmlAssedDisplayInfo)
	{
		// As the XML node is ref counted, the old asset info will be be released
		// just as needed.
		XmlNodeRef	xmlCurrentNode;

		m_xmlAssedDisplayInfo=gEnv->pSystem->CreateXmlNode("AssetDisplayInformation");

		xmlCurrentNode=m_xmlAssedDisplayInfo->newChild("Filename");
		xmlCurrentNode->setAttr("DisplayName","Filename:");
		xmlCurrentNode->setAttr("MultilineDisplay",true);
		xmlCurrentNode->setAttr("DisplayValue",m_strFilename);

		xmlCurrentNode=m_xmlAssedDisplayInfo->newChild("Filesize");
		xmlCurrentNode->setAttr("DisplayName","Size:");
		xmlCurrentNode->setAttr("MultilineDisplay",false);
		xmlCurrentNode->setAttr("ValueType","int");
		xmlCurrentNode->setAttr("MaySum",true);
		xmlCurrentNode->setAttr("DisplayValue",m_nFileSize);
	}
	return m_xmlAssedDisplayInfo;
}
//////////////////////////////////////////////////////////////////////////
bool CAssetModelItem::HitTest(int nX,int nY)
{
	CRect oTestRect(nX,nY,nX,nY);
	CRect oIntersection;

	if (
		((nX>=m_oDrawingRectangle.left)&&(nX<=m_oDrawingRectangle.right))
		&&
		((nY>=m_oDrawingRectangle.top)&&(nY<=m_oDrawingRectangle.bottom))
		)
	{
		return true;
	}
	else
	{
		return false;
	}
}
//////////////////////////////////////////////////////////////////////////
bool CAssetModelItem::HitTest(const CRect& roTestRect)
{
	CRect oIntersection;
	return (oIntersection.IntersectRect(m_oDrawingRectangle,roTestRect)!=FALSE);
}
//////////////////////////////////////////////////////////////////////////
void CAssetModelItem::SetCameraLookAt( float fRadiusScale,const Vec3& fromDir)
{
	AABB box(m_aabb);
	Vec3 v = box.max - box.min;
	float radius = v.GetLength()/2.0f;	

	m_camTarget = (box.max + box.min) * 0.5f;
	m_camRadius = (radius*fRadiusScale)+gEnv->p3DEngine->GetCurrentCamera().GetNearPlane();

	Vec3 dir = fromDir.GetNormalized();
	Matrix34 tm = Matrix33::CreateRotationVDir(dir,0);
	tm.SetTranslation( m_camTarget - dir*m_camRadius );
	m_camera.SetMatrix(tm);
}
//////////////////////////////////////////////////////////////////////////
void	CAssetModelItem::CalculateCameraPosition()
{
	AABB		box(m_aabb);

	Vec3		stCameraDirection(0.0f,0.0f,0.0f);
	float		fCameraDistance(0);

	float		afAABBDimensions[3];
	float		afRatios[3];
	float   fSmallestRatio(FLT_MAX);

	float		afCameraDistances[2];

	float		fSign(-1.0f);

	int32		nCount(0);
	int32		nSmallestRationIndex(0);
	int32		nCameraDimensionIndex(0);

	// If m_fov==45+180*K, FPE.
	float		fHalfFovTangent(tanf(DEG2RAD(m_fov/2.0f)));

	for (nCount=0;nCount<3;++nCount)
	{
		afAABBDimensions[nCount]=box.max[nCount]-box.min[nCount];
	}

	for (nCount=0;nCount<3;++nCount)
	{
		if (afAABBDimensions[(nCount+1)%3]!=0)
		{
			afRatios[nCount]=fabs(fabs(afAABBDimensions[nCount]/afAABBDimensions[(nCount+1)%3])-1.0f);
		}
		else
		{
			afRatios[nCount]=FLT_MAX;
		}
	}

	for (nCount=0;nCount<3;++nCount)
	{
		if (fSmallestRatio<afRatios[nCount])
		{
			fSmallestRatio=afRatios[nCount];
			nSmallestRationIndex=nCount;
		}
	}

	nSmallestRationIndex=2;
	nCameraDimensionIndex=(nSmallestRationIndex+2)%3;
	if (((nCameraDimensionIndex+1)%3)==2)
	{
		fSign*=fSign;
	}

	stCameraDirection[nCameraDimensionIndex]=1.0f*fSign;

	// The placement of the camera must be calculated from a point
	// on the top of the bounding box.
	if (box.min[nCameraDimensionIndex]>box.max[nCameraDimensionIndex])
	{
		box.min[nCameraDimensionIndex]=box.max[nCameraDimensionIndex];
	}
	else
	{
		box.max[nCameraDimensionIndex]=box.min[nCameraDimensionIndex];
	}

	// We want to look at the center of the front face of the AABB.
	m_camTarget = (box.max + box.min) * 0.5f;

	afCameraDistances[0]=m_camTarget[(nCameraDimensionIndex+1)%3]-m_aabb.min[(nCameraDimensionIndex+1)%3];
	afCameraDistances[1]=m_camTarget[(nCameraDimensionIndex+2)%3]-m_aabb.min[(nCameraDimensionIndex+2)%3];

 fCameraDistance=MAX(afCameraDistances[0],afCameraDistances[1])/fHalfFovTangent;

 if (fCameraDistance<m_camera.GetNearPlane())
 {
 	fCameraDistance=m_camera.GetNearPlane();
 }

 Matrix34 tm = Matrix33::CreateRotationVDir(stCameraDirection,0);
 tm.SetTranslation( m_camTarget - stCameraDirection*fCameraDistance);
 m_camera.SetMatrix(tm);


	//// The placement of the camera must be calculated from a point
	//// on the top of the bounding box.
	//if (box.min.y>box.max.y)
	//{
	//	box.min.y=box.max.y;
	//}
	//else
	//{
	//	box.max.y=box.min.y;
	//}

	//// We want to look at the center of the front face of the AABB.
	//m_camTarget = (box.max + box.min) * 0.5f;

	//float fRadius = (box.max-box.min).GetLength()/2.0f;	

	//float fDistanceToTop(m_camTarget.y-box.max.y);
	//float fHeight(m_camTarget.z-m_aabb.min.z);
	//float fWidth(m_camTarget.x-m_aabb.min.x);
	//float fHalfFovTangent(tanf(DEG2RAD(m_fov/2.0f)));

	//fVerticalCameraDistance=(fabs(fHeight)/fHalfFovTangent);
	//fHorizontalCameraDistance=(fabs(fWidth)/fHalfFovTangent);

	//fCameraDistance=MAX(fVerticalCameraDistance,fHorizontalCameraDistance);
	//if (fCameraDistance<m_camera.GetNearPlane())
	//{
	//	fCameraDistance=m_camera.GetNearPlane();
	//}

	//Matrix34 tm = Matrix33::CreateRotationVDir(stCameraDirection,0);
	//tm.SetTranslation( m_camTarget - stCameraDirection*fCameraDistance);
	//m_camera.SetMatrix(tm);
}
//////////////////////////////////////////////////////////////////////////
void CAssetModelItem::DrawGrid()
{
	// Draw grid.
	float step = 0.1f;
	float XR = 5;
	float YR = 5;

	IRenderAuxGeom * pRag = m_renderer->GetIRenderAuxGeom();
	SAuxGeomRenderFlags nRendFlags = pRag->GetRenderFlags();

	pRag->SetRenderFlags( e_Def3DPublicRenderflags );
	SAuxGeomRenderFlags nNewFlags = pRag->GetRenderFlags();
	nNewFlags.SetAlphaBlendMode( e_AlphaBlended );
	pRag->SetRenderFlags( nNewFlags );

	int nGridAlpha = 40;
	if (m_bGrid)
	{
		// Draw grid.
		for (float x = -XR; x < XR; x+=step)
		{
			if (fabs(x) > 0.01)
				//m_renderer->DrawLine( Vec3(x,-YR,0),Vec3(x,YR,0) );
				pRag->DrawLine( Vec3(x,-YR,0), ColorB(150, 150, 150 , nGridAlpha), Vec3(x,YR,0) , ColorB(150, 150, 150 , nGridAlpha) );
		}
		for (float y = -YR; y < YR; y+=step)
		{
			if (fabs(y) > 0.01)
				//m_renderer->DrawLine( Vec3(-XR,y,0),Vec3(XR,y,0) );
				pRag->DrawLine( Vec3(-XR,y,0), ColorB(150, 150, 150 , nGridAlpha), Vec3(XR,y,0) , ColorB(150, 150, 150 , nGridAlpha) );
		}

	}

	nGridAlpha = 60;
	if (m_bAxis)
	{
		// Draw axis.
		//m_renderer->SetMaterialColor( 1,0,0,1.0f );
		//m_renderer->DrawLine( Vec3(-XR,0,0),Vec3(XR,0,0) );
		pRag->DrawLine( Vec3(0,0,0), ColorB(255, 0, 0 ,nGridAlpha), Vec3(XR,0,0) , ColorB(255, 0, 0 ,nGridAlpha) );

		//m_renderer->SetMaterialColor( 0,1,0,1.0f );
		//m_renderer->DrawLine( Vec3(0,-YR,0),Vec3(0,YR,0) );
		pRag->DrawLine( Vec3(0,0,0), ColorB(0, 255, 0 ,nGridAlpha), Vec3(0,YR,0) , ColorB(0, 255, 0 ,nGridAlpha) );

		//m_renderer->SetMaterialColor( 0,0,1,1.0f );
		//m_renderer->DrawLine( Vec3(0,0,-YR),Vec3(0,0,YR) );
		pRag->DrawLine(Vec3(0,0,0), ColorB(0, 0, 255 ,nGridAlpha), Vec3(0,0,YR) , ColorB(0, 0, 255 ,nGridAlpha) );
	}
	pRag->SetRenderFlags( nRendFlags );
}
//////////////////////////////////////////////////////////////////////////
void	CAssetModelItem::SetCamera( CCamera &cam,const CRect &rcViewportRect )
{
	m_camera.SetPosition( cam.GetPosition() );
	//m_camera.SetAngle( cam.GetAngles() );

	CRect rc(rcViewportRect);
	//m_camera.SetFov(m_stdFOV);
	int w = rc.Width();
	int h = rc.Height();
	//float proj = (float)h/(float)w;
	//if (proj > 1.2f) proj = 1.2f;

	m_camera.SetFrustum( w,h, DEG2RAD(m_fov), m_camera.GetNearPlane(),m_camera.GetFarPlane() );

	m_camera.SetViewPort(rcViewportRect.left,rcViewportRect.top,
											 rcViewportRect.Width(),rcViewportRect.Height());

	//m_camera.UpdateFrustum();
}
//////////////////////////////////////////////////////////////////////////
