/*************************************************************************
Crytek Source File.
Copyright (C), Crytek Studios, 2001-2004.
-------------------------------------------------------------------------
$Id: AssetTextureDatabase.h,v 1.0 2009/04/15 11:00:00 PauloZaffari Exp wwwrun $
$DateTime$
Description: Source file for the class implementing IAssetDisplay
interface. It declares the headers of the actual used 
functions.
-------------------------------------------------------------------------
History:
- 15:04:2009   11:00 : Created by Paulo Zaffari

*************************************************************************/
#include "stdafx.h"

#include "Include/IEditorClassFactory.h"
#include "AssetTextureDatabse.h"
#include "AssetTextureItem.h"

#include "ITexture.h"
#include "IRenderer.h"

#include "Util/PathUtil.h"

#include "ImageExtensionHelper.h"

#include "Include/IAssetViewer.h"

#include "Util/MemoryBlock.h"
#include "Util/Image.h"
#include "Util/ImageUtil.h"

//////////////////////////////////////////////////////////////////////////
CAssetDisplayDatabaseTexture::CAssetDisplayDatabaseTexture()
{
	m_poAssociatedViewer=NULL;
	m_boMustEndThread=false;
	m_ref=1;
	m_boNewItemsShouldBeVisible=true;
}
//////////////////////////////////////////////////////////////////////////
CAssetDisplayDatabaseTexture::~CAssetDisplayDatabaseTexture()
{
	FreeData();
}
//////////////////////////////////////////////////////////////////////////
void CAssetDisplayDatabaseTexture::FreeData()
{
	TDFilenameToItemMap::iterator		itCurrentAsset;
	TDFilenameToItemMap::iterator		itEndAsset;

	itEndAsset=m_cKnwonTextures.end();
	for (itCurrentAsset=m_cKnwonTextures.begin();itCurrentAsset!=itEndAsset;itCurrentAsset++)
	{
		itCurrentAsset->second->Release();
	}
	m_cKnwonTextures.clear();

	m_poAssociatedViewer=NULL;
}
//////////////////////////////////////////////////////////////////////////
bool CAssetDisplayDatabaseTexture::SetAssociatedViewer(IAssetViewer* poAssociatedViewer)
{
	// Hot changing of associated viewer is not allowed, for now.
	if (m_poAssociatedViewer)
	{
		return false;
	}

	m_poAssociatedViewer=poAssociatedViewer;

	return true;
}
//////////////////////////////////////////////////////////////////////////
void CAssetDisplayDatabaseTexture::AddPreLoadedTextures(TDFilenameToItemMap& rcFilenameSet)
{
	// If no associated viewer, no need to add textures at all...
	if (!m_poAssociatedViewer)
	{
		// TOD: Must add a log entry about this.
		return;
	}

	std::vector<ITexture*>	cpiTextures;

	int															nCurrentTexture(0);
	int															nNumberOfTextures(0);
	int															nTextureWidth(0);
	int															nTextureHeight(0);
	const char*											szTextureName(NULL);
	string													strIntermediateTextureName;
	CString													strOutputTextureName;
	CString													strExtension;
	CAssetTextureItem*							poTextureDatabaseItem(NULL);
	CCryFile												oFile;
	size_t													nFileSize(0);

	nNumberOfTextures=*(int *)gEnv->pRenderer->EF_Query(EFQ_GetAllTextures,NULL);

	cpiTextures.resize(nNumberOfTextures,NULL);
	gEnv->pRenderer->EF_Query(EFQ_GetAllTextures,(INT_PTR)&cpiTextures.front());

	for (nCurrentTexture=0;nCurrentTexture<nNumberOfTextures;++nCurrentTexture)
	{
		ITexture*& piCurrentTexture=cpiTextures[nCurrentTexture];
		if (piCurrentTexture->IsTextureLoaded())
		{
			if (piCurrentTexture->GetTextureType()==eTT_3D)
			{
				// Currently the texture browser does not support 3D textures, as they make
				// everything crash when calling ITexture::GetData32().
				continue;
			}
			nTextureWidth=piCurrentTexture->GetWidth();
			nTextureHeight=piCurrentTexture->GetHeight();

			// Any texture with a side equals or less than 0 is impossible to display...
			// .. so we won't wate our time on them...
			if ((nTextureWidth<=0)||(nTextureHeight<=0))
			{
				continue;	
			}

			// Textures without names are not important for us.
			szTextureName=piCurrentTexture->GetName();
			if (!szTextureName)
			{
				continue;
			}

			// We don't want empty names as well.
			if (strlen(szTextureName)==0)
			{
				continue;
			}

			// Textures which name starts with '$' are dynamic textures or in any way textures that
			// don't interest us.
			if (szTextureName[0]=='$')
			{
				continue;
			}
		}
		else
		{
			// For now we are not loading textures, just using the ones already loaded.
		}

		strIntermediateTextureName=szTextureName;
		strIntermediateTextureName.MakeLower();
		strOutputTextureName=strIntermediateTextureName;
		Path::ConvertSlashToBackSlash(strOutputTextureName);
		strExtension=Path::GetExt(strOutputTextureName);

		if (rcFilenameSet.find(strOutputTextureName.GetBuffer())!=rcFilenameSet.end())
		{
			continue;
		}

		poTextureDatabaseItem=new CAssetTextureItem();

		//OBS: To get the file size we will need to check with the OS, as this information
		// is not stored anywhere else.
		// This may be QUITE slow.
		oFile.Open(szTextureName,"rb");
		if (oFile.GetHandle())
		{
			nFileSize=oFile.GetLength();

		}
		else
		{
			nFileSize=0;
		}
		poTextureDatabaseItem->SetFileSize(nFileSize); // A safe invalid value for now.
		oFile.Close();


		poTextureDatabaseItem->SetSurfaceType(piCurrentTexture->GetFormatName());
		poTextureDatabaseItem->SetMips(piCurrentTexture->GetNumMips());
		poTextureDatabaseItem->SetTextureWidth(piCurrentTexture->GetWidth());
		poTextureDatabaseItem->SetTextureHeight(piCurrentTexture->GetHeight());
		poTextureDatabaseItem->SetOwnerDisplayDatabase(this);
		poTextureDatabaseItem->SetExtension(strExtension.GetBuffer());

		// We are storing the original filename here for textures which come from the
		// the engine (without using backslashes) as it is done with all other  textures.
		// For some bizarre reason, if we did, the engine would have problems cashing the
		// textures. On the other hand, if we adopted as a standard the normal slashes
		// it seems that some textures loaded from the disk would have problems... so...
		// there doesn't seem to be a perfect cenario here.
		poTextureDatabaseItem->SetFilename(strIntermediateTextureName.c_str());
		//poTextureDatabaseItem->SetFilename(strOutputTextureName.c_str());

		poTextureDatabaseItem->SetUsedInLevel(true);
		poTextureDatabaseItem->SetHasAlphaChannelStatus(CImageExtensionHelper::HasAlphaForName(piCurrentTexture->GetFormatName()));

		poTextureDatabaseItem->SetVisibleStatus(m_boNewItemsShouldBeVisible);

		m_poAssociatedViewer->BeginDatabaseLock();

		rcFilenameSet.insert(TDInsertionElement(strOutputTextureName.GetBuffer(),poTextureDatabaseItem));

		m_poAssociatedViewer->AddAssetDatabaseItem(poTextureDatabaseItem);

		m_poAssociatedViewer->EndDatabaseLock();
	}
}
//////////////////////////////////////////////////////////////////////////
void CAssetDisplayDatabaseTexture::AddDiskTexture( TDFilenameToItemMap &rcFilenameSet )
{
	// If no associated viewer, no need to add textures at all...
	if (!m_poAssociatedViewer)
	{
		// TOD: Must add a log entry about this.
		return;
	}

	int																			nTextureWidth(0);
	int																			nTextureHeight(0);
	const char															*szTextureName(NULL);
	CString																	strExtension;
	CString																	strFilename;
	string																	strSurfaceType;

	CFileUtil::FileArray										cFiles;
	int																			nTotalFiles(0);
	int																			nCurrentFile(0);
	string																	strIntermediateFilename;
	CString																	strOutputTextureName;

	CCryFile																file;
	CImageExtensionHelper::DDS_FILE_DESC		stFileDesc;

	CAssetTextureItem*											poTextureDatabaseItem(NULL);


	// Will first look for all files and just add those which are actually textures...
	// Responsiveness optimization: instead of traversing directories just once and check
	// all files, in order to have faster response times we do it twice: once for the .dds
	// files and once for .tif. As this is threaded, both searches will be MUCH faster alone
	// even though the combined time will be bigger.
	// Still, the method used will have some results MUCH faster and thus the user will
	// have time to have fun with them before the final results will be there.
	//CFileUtil::ScanDirectory(PathUtil::GetGameFolder().c_str(),"*.*"/*"*.*"*/,cFiles,true);

	CFileUtil::ScanDirectory(PathUtil::GetGameFolder().c_str(),"*.dds"/*"*.*"*/,cFiles,true);
	// For every file, check if is a texture...
	nTotalFiles=cFiles.size();
	for (nCurrentFile=0;nCurrentFile<nTotalFiles;++nCurrentFile)
	{
		// Checks if we must do a premature thread killing.
		if (m_boMustEndThread)
		{
			return;
		}

		CFileUtil::FileDesc& rstFileDescriptor=cFiles[nCurrentFile];

		strIntermediateFilename=rstFileDescriptor.filename.GetBuffer();
		strIntermediateFilename.MakeLower();
		strOutputTextureName=strIntermediateFilename;
		Path::ConvertSlashToBackSlash(strOutputTextureName);

		// No need to load files already loaded by the engine...
		if (rcFilenameSet.find(strOutputTextureName.GetBuffer())!=rcFilenameSet.end())
		{
			continue;
		}

		// No need for temp files.
		if (strstr(rstFileDescriptor.filename,"~~~TempFile~~~")!=NULL)
		{
			continue;
		}

		strExtension=Path::GetExt(strOutputTextureName.GetBuffer());

		if (!file.Open( rstFileDescriptor.filename,"rb" ))
		{
			continue;
		}

		// Read magic number
		file.ReadTypeRaw( &stFileDesc );
		if (!stFileDesc.IsValid())
		{
			file.Close();
			continue;
		}
		file.Close();

		ETEX_Format format = DDSFormats::GetFormatByDesc(stFileDesc.header.ddspf);
		int  nHorizontalFacesMultiplier(1);
		int  nVerticalFacesMultiplier(1);
		bool boIsCubemap(false);
		if ((stFileDesc.header.dwSurfaceFlags & DDS_SURFACE_FLAGS_CUBEMAP) && (stFileDesc.header.dwCubemapFlags & DDS_CUBEMAP_ALLFACES))
		{
			boIsCubemap=true;

			if (
				(format == eTF_A8R8G8B8)
				||
				(format == eTF_X8R8G8B8)
				||
				(format == eTF_R8G8B8)
				||
				(format == eTF_L8)
				||
				(format == eTF_A8)
				)
			{
				nHorizontalFacesMultiplier=3;
				nVerticalFacesMultiplier=2;
			}
		}

		// Textures without names are not important for us.
		szTextureName=strOutputTextureName.GetBuffer();
		if (!szTextureName)
		{
			continue;
		}

		nTextureWidth=stFileDesc.header.dwWidth;
		nTextureHeight=stFileDesc.header.dwHeight;

		// Any texture with a side equals or less than 0 is impossible to display...
		// .. so we won't wate our time on them...
		if ((nTextureWidth<=0)||(nTextureHeight<=0))
		{
			continue;	
		}

		// Textures which name starts with '$' are dynamic textures or in any way textures that
		// don't interest us.
		if (szTextureName[0]=='$')
		{
			continue;
		}

		poTextureDatabaseItem=new CAssetTextureItem();

		strSurfaceType=CImageExtensionHelper::NameForDesc(stFileDesc.header.ddspf);
		poTextureDatabaseItem->SetFileSize(rstFileDescriptor.size);
		poTextureDatabaseItem->SetSurfaceType(strSurfaceType);
		poTextureDatabaseItem->SetMips(stFileDesc.header.GetMipCount());
		poTextureDatabaseItem->SetTextureWidth(nTextureWidth*nHorizontalFacesMultiplier);
		poTextureDatabaseItem->SetTextureHeight(nTextureHeight*nVerticalFacesMultiplier);
		poTextureDatabaseItem->SetFilename(strOutputTextureName.GetBuffer());
		poTextureDatabaseItem->SetUsedInLevel(false);
		poTextureDatabaseItem->SetIsCubeMap(boIsCubemap);
		poTextureDatabaseItem->SetHasAlphaChannelStatus(CImageExtensionHelper::HasAlphaForName(strSurfaceType.c_str()));
		poTextureDatabaseItem->SetOwnerDisplayDatabase(this);
		poTextureDatabaseItem->SetExtension(strExtension.GetBuffer());
		poTextureDatabaseItem->SetVisibleStatus(m_boNewItemsShouldBeVisible);

		m_poAssociatedViewer->BeginDatabaseLock();
		if (!m_boMustEndThread)
		{
			rcFilenameSet.insert(TDInsertionElement(strOutputTextureName.GetBuffer(),poTextureDatabaseItem));
			m_poAssociatedViewer->AddAssetDatabaseItem(poTextureDatabaseItem);
		}
		else
		{
			m_poAssociatedViewer->EndDatabaseLock();
			delete poTextureDatabaseItem;
			return;
		}
		
		m_poAssociatedViewer->EndDatabaseLock();
	}



	CFileUtil::ScanDirectory(PathUtil::GetGameFolder().c_str(),"*.tif",cFiles,true);
	// For every file, check if is a texture...
	nTotalFiles=cFiles.size();
	for (nCurrentFile=0;nCurrentFile<nTotalFiles;++nCurrentFile)
	{
		// Checks if we must do a premature thread killing.
		if (m_boMustEndThread)
		{
			return;
		}

		CFileUtil::FileDesc& rstFileDescriptor=cFiles[nCurrentFile];

		strIntermediateFilename=rstFileDescriptor.filename.GetBuffer();
		strIntermediateFilename.MakeLower();
		strOutputTextureName=strIntermediateFilename;
		Path::ConvertSlashToBackSlash(strOutputTextureName);

		// No need to load files already loaded...
		if (rcFilenameSet.find(strOutputTextureName.GetBuffer())!=rcFilenameSet.end())
		{
			continue;
		}

		// No need for temp files.
		if (strstr(strOutputTextureName.GetBuffer(),"~~~TempFile~~~")!=NULL)
		{
			continue;
		}

		strFilename=strOutputTextureName.GetBuffer();
		strExtension=Path::GetExt(strOutputTextureName.GetBuffer());
		strFilename=Path::ReplaceExtension(strFilename,"dds");

		// No need to load files already loaded as it's equivalent dds.
		if (rcFilenameSet.find(strFilename.GetBuffer())!=rcFilenameSet.end())
		{
			continue;
		}

		CImage		oImage;
		CImageUtil::LoadImage(rstFileDescriptor.filename,oImage);

		// Textures without names are not important for us.
		szTextureName=(char*)rstFileDescriptor.filename.GetBuffer();
		if (!szTextureName)
		{
			continue;
		}

		nTextureWidth=oImage.GetWidth();
		nTextureHeight=oImage.GetHeight();

		// Any texture with a side equals or less than 0 is impossible to display...
		// .. so we won't wate our time on them...
		if ((nTextureWidth<=0)||(nTextureHeight<=0))
		{
			continue;	
		}

		// Textures which name starts with '$' are dynamic textures or in any way textures that
		// don't interest us.
		if (szTextureName[0]=='$')
		{
			continue;
		}

		strSurfaceType=oImage.GetFormatDescription();
		poTextureDatabaseItem=new CAssetTextureItem();
		poTextureDatabaseItem->SetFileSize(rstFileDescriptor.size);
		poTextureDatabaseItem->SetSurfaceType(strSurfaceType);
		poTextureDatabaseItem->SetMips(oImage.GetNumberOfMipMaps()); // Currently setting to one... but we could have more accurate info on that.
		poTextureDatabaseItem->SetTextureWidth(oImage.GetWidth());
		poTextureDatabaseItem->SetTextureHeight(oImage.GetHeight());
		poTextureDatabaseItem->SetFilename(strOutputTextureName.GetBuffer());
		poTextureDatabaseItem->SetUsedInLevel(false);
		poTextureDatabaseItem->SetIsCubeMap(oImage.IsCubemap());
		poTextureDatabaseItem->SetHasAlphaChannelStatus(CImageExtensionHelper::HasAlphaForName(strSurfaceType.c_str()));
		poTextureDatabaseItem->SetOwnerDisplayDatabase(this);
		poTextureDatabaseItem->SetExtension(strExtension.GetBuffer());
		poTextureDatabaseItem->SetVisibleStatus(m_boNewItemsShouldBeVisible);

		// This is a good place to check if we should stop processing files.
		if (m_boMustEndThread)
		{
			delete poTextureDatabaseItem;
			return;	
		}

		m_poAssociatedViewer->BeginDatabaseLock();
		rcFilenameSet.insert(TDInsertionElement(strOutputTextureName.GetBuffer(),poTextureDatabaseItem));
		m_poAssociatedViewer->AddAssetDatabaseItem(poTextureDatabaseItem);
		m_poAssociatedViewer->EndDatabaseLock();
	}
}
//////////////////////////////////////////////////////////////////////////
void CAssetDisplayDatabaseTexture::NotifyShutDown()
{
	m_poAssociatedViewer->BeginDatabaseLock();
	m_boMustEndThread=true;
	m_poAssociatedViewer->EndDatabaseLock();
}
//////////////////////////////////////////////////////////////////////////
void CAssetDisplayDatabaseTexture::Lock()
{
	CryThread<CAssetDisplayDatabaseTexture>::Lock();
}
//////////////////////////////////////////////////////////////////////////
void CAssetDisplayDatabaseTexture::Unlock()
{
	CryThread<CAssetDisplayDatabaseTexture>::Unlock();
}
//////////////////////////////////////////////////////////////////////////
void CAssetDisplayDatabaseTexture::WaitForThread()
{
	CryThread<CAssetDisplayDatabaseTexture>::WaitForThread();
}
//////////////////////////////////////////////////////////////////////////
void CAssetDisplayDatabaseTexture::Start()
{
	CryThread<CAssetDisplayDatabaseTexture>::Start();
}
//////////////////////////////////////////////////////////////////////////
void CAssetDisplayDatabaseTexture::Stop()
{
	CryThread<CAssetDisplayDatabaseTexture>::Stop();
}


//////////////////////////////////////////////////////////////////////////
const XmlNodeRef&		CAssetDisplayDatabaseTexture::GetFilterOptionsDescription()
{
	if (!m_xmlFilterOptionsDescription)
	{
		XmlNodeRef	xmlCurrentNode;
		XmlNodeRef	xmlGroupNode;


		m_xmlFilterOptionsDescription=gEnv->pSystem->CreateXmlNode("FilterOptionsDescription");
		m_xmlFilterOptionsDescription->setAttr("AssetTypeName","Texture");

		xmlGroupNode=m_xmlFilterOptionsDescription->newChild("Group");
		xmlGroupNode->setAttr("Name","Filename");

		xmlCurrentNode=xmlGroupNode->newChild("Parameter");
		xmlCurrentNode->setAttr("Name","Filename");
		//xmlCurrentNode->setAttr("DisplayName","");
		xmlCurrentNode->setAttr("Type","string");
		//xmlCurrentNode->setAttr("Default","");


		xmlGroupNode=m_xmlFilterOptionsDescription->newChild("Group");
		xmlGroupNode->setAttr("Name","Texture Dimensions");

		//////////////////////////////////////////////////////////////////////////
		// Enum sample
		xmlCurrentNode=xmlGroupNode->newChild("Parameter");
		xmlCurrentNode->setAttr("Name","Minimum Width");
		xmlCurrentNode->setAttr("DisplayName","Minimum Width");
		xmlCurrentNode->setAttr("Type","enum");
		xmlCurrentNode->setAttr("Default",0);

		xmlCurrentNode=xmlCurrentNode->newChild("enum");
		xmlCurrentNode->setAttr("0","Any");
		xmlCurrentNode->setAttr("1","1");
		xmlCurrentNode->setAttr("2","2");
		xmlCurrentNode->setAttr("3","4");
		xmlCurrentNode->setAttr("4","8");
		xmlCurrentNode->setAttr("5","16");
		xmlCurrentNode->setAttr("6","32");
		xmlCurrentNode->setAttr("7","64");
		xmlCurrentNode->setAttr("8","128");
		xmlCurrentNode->setAttr("9","256");
		xmlCurrentNode->setAttr("10","512");
		xmlCurrentNode->setAttr("11","1024");
		xmlCurrentNode->setAttr("12","2048");
		xmlCurrentNode->setAttr("13","4096");
		xmlCurrentNode->setAttr("14","8192");
		xmlCurrentNode->setAttr("15","16384");
		//////////////////////////////////////////////////////////////////////////


		//////////////////////////////////////////////////////////////////////////
		// Enum sample
		xmlCurrentNode=xmlGroupNode->newChild("Parameter");
		xmlCurrentNode->setAttr("Name","Maximum Width");
		xmlCurrentNode->setAttr("DisplayName","Maximum Width");
		xmlCurrentNode->setAttr("Type","enum");
		xmlCurrentNode->setAttr("Default",0);

		xmlCurrentNode=xmlCurrentNode->newChild("enum");
		xmlCurrentNode->setAttr("0","Any");
		xmlCurrentNode->setAttr("1","1");
		xmlCurrentNode->setAttr("2","2");
		xmlCurrentNode->setAttr("3","4");
		xmlCurrentNode->setAttr("4","8");
		xmlCurrentNode->setAttr("5","16");
		xmlCurrentNode->setAttr("6","32");
		xmlCurrentNode->setAttr("7","64");
		xmlCurrentNode->setAttr("8","128");
		xmlCurrentNode->setAttr("9","256");
		xmlCurrentNode->setAttr("10","512");
		xmlCurrentNode->setAttr("11","1024");
		xmlCurrentNode->setAttr("12","2048");
		xmlCurrentNode->setAttr("13","4096");
		xmlCurrentNode->setAttr("14","8192");
		xmlCurrentNode->setAttr("15","16384");
		//////////////////////////////////////////////////////////////////////////


		//////////////////////////////////////////////////////////////////////////
		// Enum sample
		xmlCurrentNode=xmlGroupNode->newChild("Parameter");
		xmlCurrentNode->setAttr("Name","Minimum Height");
		xmlCurrentNode->setAttr("DisplayName","Minimum Height");
		xmlCurrentNode->setAttr("Type","enum");
		xmlCurrentNode->setAttr("Default",0);

		xmlCurrentNode=xmlCurrentNode->newChild("enum");
		xmlCurrentNode->setAttr("0","Any");
		xmlCurrentNode->setAttr("1","1");
		xmlCurrentNode->setAttr("2","2");
		xmlCurrentNode->setAttr("3","4");
		xmlCurrentNode->setAttr("4","8");
		xmlCurrentNode->setAttr("5","16");
		xmlCurrentNode->setAttr("6","32");
		xmlCurrentNode->setAttr("7","64");
		xmlCurrentNode->setAttr("8","128");
		xmlCurrentNode->setAttr("9","256");
		xmlCurrentNode->setAttr("10","512");
		xmlCurrentNode->setAttr("11","1024");
		xmlCurrentNode->setAttr("12","2048");
		xmlCurrentNode->setAttr("13","4096");
		xmlCurrentNode->setAttr("14","8192");
		xmlCurrentNode->setAttr("15","16384");
		//////////////////////////////////////////////////////////////////////////


		//////////////////////////////////////////////////////////////////////////
		// Enum sample
		xmlCurrentNode=xmlGroupNode->newChild("Parameter");
		xmlCurrentNode->setAttr("Name","Maximum Height");
		xmlCurrentNode->setAttr("DisplayName","Maximum Height");
		xmlCurrentNode->setAttr("Type","enum");
		xmlCurrentNode->setAttr("Default",0);

		xmlCurrentNode=xmlCurrentNode->newChild("enum");
		xmlCurrentNode->setAttr("0","Any");
		xmlCurrentNode->setAttr("1","1");
		xmlCurrentNode->setAttr("2","2");
		xmlCurrentNode->setAttr("3","4");
		xmlCurrentNode->setAttr("4","8");
		xmlCurrentNode->setAttr("5","16");
		xmlCurrentNode->setAttr("6","32");
		xmlCurrentNode->setAttr("7","64");
		xmlCurrentNode->setAttr("8","128");
		xmlCurrentNode->setAttr("9","256");
		xmlCurrentNode->setAttr("10","512");
		xmlCurrentNode->setAttr("11","1024");
		xmlCurrentNode->setAttr("12","2048");
		xmlCurrentNode->setAttr("13","4096");
		xmlCurrentNode->setAttr("14","8192");
		xmlCurrentNode->setAttr("15","16384");
		//////////////////////////////////////////////////////////////////////////


		//////////////////////////////////////////////////////////////////////////
		xmlGroupNode=m_xmlFilterOptionsDescription->newChild("Group");
		xmlGroupNode->setAttr("Name","Texture Semantics");

		xmlCurrentNode=xmlGroupNode->newChild("Parameter");
		xmlCurrentNode->setAttr("Name","Show Only Diffuse");
		xmlCurrentNode->setAttr("DisplayName","Show Only Diffuse");
		xmlCurrentNode->setAttr("Type","bool");
		xmlCurrentNode->setAttr("Default",false);

		xmlCurrentNode=xmlGroupNode->newChild("Parameter");
		xmlCurrentNode->setAttr("Name","Show Only Specular");
		xmlCurrentNode->setAttr("DisplayName","Show Only Specular");
		xmlCurrentNode->setAttr("Type","bool");
		xmlCurrentNode->setAttr("Default",false);

		xmlCurrentNode=xmlGroupNode->newChild("Parameter");
		xmlCurrentNode->setAttr("Name","Show Only Bump");
		xmlCurrentNode->setAttr("DisplayName","Show Only Bump");
		xmlCurrentNode->setAttr("Type","bool");
		xmlCurrentNode->setAttr("Default",false);

		xmlCurrentNode=xmlGroupNode->newChild("Parameter");
		xmlCurrentNode->setAttr("Name","Show Cube Map");
		xmlCurrentNode->setAttr("DisplayName","Show Cube Map");
		xmlCurrentNode->setAttr("Type","bool");
		xmlCurrentNode->setAttr("Default",false);
		//////////////////////////////////////////////////////////////////////////


		//////////////////////////////////////////////////////////////////////////
		xmlGroupNode=m_xmlFilterOptionsDescription->newChild("Group");
		xmlGroupNode->setAttr("Name","Texture Files");

		xmlCurrentNode=xmlGroupNode->newChild("Parameter");
		xmlCurrentNode->setAttr("Name","Show Only DDS");
		xmlCurrentNode->setAttr("DisplayName","Show Only DDS");
		xmlCurrentNode->setAttr("Type","bool");
		xmlCurrentNode->setAttr("Default",false);

		xmlCurrentNode=xmlGroupNode->newChild("Parameter");
		xmlCurrentNode->setAttr("Name","Show Only TIF");
		xmlCurrentNode->setAttr("DisplayName","Show Only TIF");
		xmlCurrentNode->setAttr("Type","bool");
		xmlCurrentNode->setAttr("Default",false);

		xmlCurrentNode=xmlGroupNode->newChild("Parameter");
		xmlCurrentNode->setAttr("Name","Used in Level");
		xmlCurrentNode->setAttr("DisplayName","Used in Level");
		xmlCurrentNode->setAttr("Type","bool");
		xmlCurrentNode->setAttr("Default",false);

		xmlCurrentNode=xmlGroupNode->newChild("Parameter");
		xmlCurrentNode->setAttr("Name","Has Alpha Channel");
		xmlCurrentNode->setAttr("DisplayName","Has Alpha Channel");
		xmlCurrentNode->setAttr("Type","bool");
		xmlCurrentNode->setAttr("Default",false);

		xmlCurrentNode=xmlGroupNode->newChild("Parameter");
		xmlCurrentNode->setAttr("Name","Show Alpha Channel");
		xmlCurrentNode->setAttr("DisplayName","Show Alpha Channel");
		xmlCurrentNode->setAttr("Type","bool");
		xmlCurrentNode->setAttr("Default",false);
		//////////////////////////////////////////////////////////////////////////
	}

	return m_xmlFilterOptionsDescription;
}
//////////////////////////////////////////////////////////////////////////
const XmlNodeRef&		CAssetDisplayDatabaseTexture::GetDisplayInformationTemplate()
{
	if (!m_xmlInformationDisplayTemplate)
	{
		XmlNodeRef	xmlCurrentNode;

		//static XmlNodeRef	m_xmlInformationDisplayTemplate;
		m_xmlInformationDisplayTemplate=(XmlNodeRef)gEnv->pSystem->CreateXmlNode("AssetDisplayInformation");

		xmlCurrentNode=m_xmlInformationDisplayTemplate->newChild("Filename");
		xmlCurrentNode->setAttr("DisplayName","Filename:");
		xmlCurrentNode->setAttr("MultilineDisplay",true);
		xmlCurrentNode->setAttr("DisplayValue","No selection");

		xmlCurrentNode=m_xmlInformationDisplayTemplate->newChild("SurfaceType");
		xmlCurrentNode->setAttr("DisplayName","Type:");
		xmlCurrentNode->setAttr("MultilineDisplay",false);
		xmlCurrentNode->setAttr("DisplayValue","");

		xmlCurrentNode=m_xmlInformationDisplayTemplate->newChild("Mips");
		xmlCurrentNode->setAttr("DisplayName","Mips:");
		xmlCurrentNode->setAttr("MultilineDisplay",false);
		xmlCurrentNode->setAttr("DisplayValue","");

		xmlCurrentNode=m_xmlInformationDisplayTemplate->newChild("Filesize");
		xmlCurrentNode->setAttr("DisplayName","Size:");
		xmlCurrentNode->setAttr("MultilineDisplay",false);
		xmlCurrentNode->setAttr("ValueType","int");
		xmlCurrentNode->setAttr("MaySum",true);
		xmlCurrentNode->setAttr("DisplayValue",0);

		xmlCurrentNode=m_xmlInformationDisplayTemplate->newChild("Resolution");
		xmlCurrentNode->setAttr("DisplayName","Resolution:");
		xmlCurrentNode->setAttr("MultilineDisplay",false);
		xmlCurrentNode->setAttr("DisplayValue","");
	}

	return m_xmlInformationDisplayTemplate;
}
//////////////////////////////////////////////////////////////////////////
const char*					CAssetDisplayDatabaseTexture::GetDatabaseName()
{
	return "Textures";
}
//////////////////////////////////////////////////////////////////////////
void								CAssetDisplayDatabaseTexture::SetVisibleStatus(bool bVisible)
{
	TDFilenameToItemMap::iterator	itCurrentIterator;
	TDFilenameToItemMap::iterator	itEndIterator;

	m_poAssociatedViewer->BeginDatabaseLock();
	m_boNewItemsShouldBeVisible=bVisible;

	itEndIterator=m_cKnwonTextures.end();
	for 
			(
				itCurrentIterator=m_cKnwonTextures.begin();
				itCurrentIterator!=itEndIterator;
				itCurrentIterator++
			)
	{
		if (m_boMustEndThread)
		{
			m_poAssociatedViewer->EndDatabaseLock();
			return;
		}
		itCurrentIterator->second->SetVisibleStatus(bVisible);
	}

	m_poAssociatedViewer->EndDatabaseLock();
}
//////////////////////////////////////////////////////////////////////////
bool	CAssetDisplayDatabaseTexture::GetVisibleStatus()
{
	return m_boNewItemsShouldBeVisible;
}
//////////////////////////////////////////////////////////////////////////
IAssetDisplay*	CAssetDisplayDatabaseTexture::GetItem(const char* szAddItem)
{
	if (!m_poAssociatedViewer)
	{
		return NULL;
	}

	if (strlen(szAddItem)==0)
	{
		return NULL;
	}

	m_poAssociatedViewer->BeginDatabaseLock();

	CAssetTextureItem*											poReturn(NULL);
	CString                                 strOriginalAddItem(szAddItem);
	CString                                 strAddItem(szAddItem);
	CString																  strExtension;
	CString																	strOutputFilename;

	// Replace the tif extension for the dds extension, as in other modules.
	strExtension=Path::GetExt(strAddItem);
	if (strExtension.GetLength()>0)
	{
		if (strExtension.CompareNoCase("tif")==0)
		{
			strAddItem=Path::ReplaceExtension(strAddItem,"dds");
		}			
	}		


	TDFilenameToItemMap::iterator				itIterator;
	itIterator=m_cKnwonTextures.find(strAddItem.GetBuffer());
	if (itIterator!=m_cKnwonTextures.end())
	{
		m_poAssociatedViewer->EndDatabaseLock();
		return itIterator->second;
	}

	itIterator=m_cKnwonTextures.find(strOriginalAddItem.GetBuffer());
	if (itIterator!=m_cKnwonTextures.end())
	{
		m_poAssociatedViewer->EndDatabaseLock();
		return itIterator->second;
	}


	CFileUtil::FileArray										cFiles;
	CString																	strPath(PathUtil::GetGameFolder().c_str());
	string																	strSurfaceType;

	strOutputFilename=strAddItem.GetBuffer();
	CFileUtil::ScanDirectory(strPath,strAddItem,cFiles,false);
	if (cFiles.empty())
	{	
		strOutputFilename=strOriginalAddItem.GetBuffer();
		CFileUtil::ScanDirectory(strPath,strOriginalAddItem,cFiles,false);
		if (cFiles.empty())
		{			
			m_poAssociatedViewer->EndDatabaseLock();
			return NULL;
		}				
	}

	int										nTextureWidth(0),nTextureHeight(0);
	CFileUtil::FileDesc&	rstFileDescriptor=cFiles[0];
	CImage								oImage;
	CImageUtil::LoadImage(strOutputFilename.GetBuffer(),oImage);

	Path::ConvertSlashToBackSlash(strOutputFilename);
	strlwr((char*)strOutputFilename.GetBuffer());

	nTextureWidth=oImage.GetWidth();
	nTextureHeight=oImage.GetHeight();

	strSurfaceType=oImage.GetFormatDescription();
	poReturn=new CAssetTextureItem();
	poReturn->SetFileSize(rstFileDescriptor.size);
	poReturn->SetSurfaceType(strSurfaceType);
	poReturn->SetMips(1); // Currently setting to one... but we could have more accurate info on that.
	poReturn->SetTextureWidth(oImage.GetWidth());
	poReturn->SetTextureHeight(oImage.GetHeight());
	poReturn->SetFilename(strOutputFilename.GetBuffer());
	poReturn->SetUsedInLevel(true);
	poReturn->SetOwnerDisplayDatabase(this);
	poReturn->SetExtension(strExtension.GetBuffer());
	poReturn->SetVisibleStatus(m_boNewItemsShouldBeVisible);

	if (m_boMustEndThread)
	{				
		delete poReturn;
		m_poAssociatedViewer->EndDatabaseLock();
		return NULL;	
	}

	m_poAssociatedViewer->AddAssetDatabaseItem(poReturn);
	m_cKnwonTextures.insert(TDInsertionElement(strOutputFilename.GetBuffer(),poReturn));

	m_poAssociatedViewer->EndDatabaseLock();

	return poReturn;
}
//////////////////////////////////////////////////////////////////////////
void CAssetDisplayDatabaseTexture::Run()
{
	if (!m_poAssociatedViewer)
	{
		return;
	}

	m_boMustEndThread=false;

	AddPreLoadedTextures(m_cKnwonTextures);
	AddDiskTexture(m_cKnwonTextures);

	m_poAssociatedViewer->FinishedDatabaseUpdate(this);
}
//////////////////////////////////////////////////////////////////////////
