#include "StdAfx.h"
#include "RTPSimpleServer.h"

#include <StlUtils.h>

#include <winsock2.h>
#include <ws2tcpip.h>
#include <iphlpapi.h>
#pragma comment( lib, "ws2_32.lib" )

int CRTPSimpleServer::rtp_server_debug( 0 );

CRTPSimpleServer::CRTPSimpleServer( int port )
: m_socket( INVALID_SOCKET )
, m_pRtpPacketListener( NULL )
, m_wallclockTimestamp( 0 )
{

	REGISTER_CVAR( rtp_server_debug, rtp_server_debug, VF_DUMPTODISK, "" );

	m_pRtpPacketListener = this;

	WSADATA wsaData;
	const WORD winsockVersion = MAKEWORD( 2, 2 );
	int startupResult = WSAStartup( winsockVersion, &wsaData );
	if ( startupResult != 0 )
	{
		CryLog( "RTP Server: Failed on WSAStartup" );
		return;
	}

	addrinfo hints;
	ZeroMemory( &hints, sizeof( hints ) );
	hints.ai_flags = AI_PASSIVE;
	hints.ai_family = AF_INET;
	hints.ai_socktype = SOCK_DGRAM;
	hints.ai_protocol = IPPROTO_UDP;

	char portString[ 16 ];
	sprintf_s( portString, 16, "%d", port );

	addrinfo* addressInfo; 
	int getAddressResult = getaddrinfo( NULL, portString, &hints, &addressInfo );
	if ( getAddressResult != 0 )
	{
		CryLog( "RTP Server: Failed on getaddrinfo" );
		return;
	}


	m_socket = socket( addressInfo->ai_family, addressInfo->ai_socktype, addressInfo->ai_protocol );
	if ( m_socket == INVALID_SOCKET )
	{
		CryLog( "RTP Server: Failed on socket creation" );
		freeaddrinfo( addressInfo );
		return;
	}

	unsigned long nonBlocking = 1;
	ioctlsocket( m_socket, FIONBIO, &nonBlocking );

	int bindResult = bind( m_socket, addressInfo->ai_addr, ( int )( addressInfo->ai_addrlen ) );
	if ( bindResult == SOCKET_ERROR )
	{
		CryLog( "RTP Server: Failed on bind" );
		freeaddrinfo( addressInfo );
		return;
	}

	freeaddrinfo( addressInfo );
}


CRTPSimpleServer::~CRTPSimpleServer()
{
	closesocket( m_socket );
	WSACleanup();
}

void CRTPSimpleServer::SetPacketListener( IRTPPacketListener* pRtpPacketListener )
{
	if ( pRtpPacketListener == NULL )
	{
		m_pRtpPacketListener = this;
	}
	else
	{
		m_pRtpPacketListener = pRtpPacketListener;
	}
}

void CRTPSimpleServer::Update()
{
	m_wallclockTimestamp = GetAsynchTimestamp();

	ReadData();
}

bool CRTPSimpleServer::ReadData()
{
	bool keepReading = true;

	while ( keepReading )
	{
		const int BUFFER_SIZE = 2048;
		byte buffer[ BUFFER_SIZE ];

		int bytesReceived = recvfrom( m_socket, ( char* )( buffer ), BUFFER_SIZE - 1, 0, NULL, 0);

		if ( bytesReceived == SOCKET_ERROR )
		{
			int errorCode = WSAGetLastError();
			if ( errorCode == WSAEWOULDBLOCK )
			{
				return true;
			}
			CryLog( "RTP Server( Read Data ): recv failed with error code '%d'", errorCode );
			return false;
		}

		if ( bytesReceived == 0 )
		{
			CryLog( "RTP Server( Read Data ): recv 0 bytes" );
			return false;
		}

		buffer[ bytesReceived ] = 0; //Hack: Null terminating buffer to make string parsing easier.

		if (bytesReceived<sizeof( SFixedRTPHeader ))
			CryLog( "bytesReceived too small(%d)", bytesReceived );

		SFixedRTPHeader* pRtpHeader = ( SFixedRTPHeader* )( buffer );
		pRtpHeader->timestamp = ntohl( pRtpHeader->timestamp );
		pRtpHeader->sequenceNumber = ntohs( pRtpHeader->sequenceNumber );
		pRtpHeader->synchornisationSourceId = ntohl( pRtpHeader->synchornisationSourceId );

		int totalRTPHeaderBytes = sizeof( SFixedRTPHeader ) + pRtpHeader->CSRCCount * 4;
		int payloadBytes = bytesReceived - totalRTPHeaderBytes - pRtpHeader->padding;
		if (payloadBytes>480)
			CryLog( "payloadBytes too long(%d)", payloadBytes );

		byte* pRtpPayload = buffer + totalRTPHeaderBytes;


		if ( rtp_server_debug != 0 )
		{
			CryLog( "[RTP]: payload %d sequenceNum %d bytesReceived %d timestamp %d ssid %d received at %d", pRtpHeader->payloadType, pRtpHeader->sequenceNumber, bytesReceived, pRtpHeader->timestamp, pRtpHeader->synchornisationSourceId, m_wallclockTimestamp );
		}

		CreateSynchronisationEntry( pRtpHeader );

		m_pRtpPacketListener->OnRTPPacket( pRtpHeader, pRtpPayload, payloadBytes );
	}

	return true;
}


int64 CRTPSimpleServer::GetWallclockTimestamp() const
{
	return m_wallclockTimestamp;
}


int64 CRTPSimpleServer::ConvertToWallclockTimestamp( SFixedRTPHeader* pRtpHeader, int64 samplingFrequency ) const
{
	const int64 SECONDS_TO_MILLISECONDS_FACTOR = 1000;

	const SSynchronisationInfo& synchronisationInfo = stl::find_in_map( m_synchronisationSources, pRtpHeader->synchornisationSourceId, m_dummySynchronisationInfo );
	
	int64 relativeRtpTimestamp = ( ( int64 )( pRtpHeader->timestamp ) * SECONDS_TO_MILLISECONDS_FACTOR ) / samplingFrequency;
	int64 absoluteRtpTimestamp = synchronisationInfo.wallclockStartTimestamp + relativeRtpTimestamp;

	return absoluteRtpTimestamp;
}


void CRTPSimpleServer::ClearSynchronisationEntries()
{
	m_synchronisationSources.clear();
}


void CRTPSimpleServer::CreateSynchronisationEntry( SFixedRTPHeader* pRtpHeader )
{
	//TODO: debug info when new synchronisation queue created.
	if ( rtp_server_debug != 0 )
	{
		if ( m_synchronisationSources.find( pRtpHeader->synchornisationSourceId ) == m_synchronisationSources.end() )
		{
			CryLog( "[RTP]: new ssid %d deteceted", pRtpHeader->synchornisationSourceId );
		}
	}
	m_synchronisationSources.insert( std::pair< int32, SSynchronisationInfo >( pRtpHeader->synchornisationSourceId, SSynchronisationInfo( m_wallclockTimestamp ) ) );
}

int64 CRTPSimpleServer::GetAsynchTimestamp() const
{
	return GetISystem()->GetITimer()->GetAsyncTime().GetMilliSecondsAsInt64();
}