#include "StdAfx.h"
#include "Talkie/Talkie.h"

#include "Game.h"
#include "GameCVars.h"

#include <IEntitySystem.h>
#include <ICryAnimation.h>
#include <IFacialAnimation.h>
#include <IEntity.h>
#include <IActorSystem.h>
#include <IViewSystem.h>
#define NOMINMAX
#include <windows.h>
#undef PlaySound

#define PAYLOAD_TYPE_AUDIO 101
#define PAYLOAD_TYPE_VISIM 102
#define PAYLOAD_TYPE_ANIMATION 103
#define PAYLOAD_TYPE_EMOTION 104
#define PAYLOAD_TYPE_LOOKAT 105

#define AUDIO_PACKET_SIZE 480
#define AUDIO_BYTES_PER_SAMPLE 2
#define AUDIO_BITS_PER_SAMPLE ( AUDIO_BYTES_PER_SAMPLE * 8 )
#define AUDIO_SAMPLE_PER_PACKET ( AUDIO_PACKET_SIZE / AUDIO_BITS_PER_SAMPLE )
#define AUDIO_SAMPLING_RATE_HZ 22050

#define VISIM_SAMPLING_RATE_HZ AUDIO_SAMPLING_RATE_HZ
#define EMOTION_SAMPLING_RATE_HZ AUDIO_SAMPLING_RATE_HZ
#define ANIMATION_SAMPLING_RATE_HZ AUDIO_SAMPLING_RATE_HZ

#define EMOTION_NAME_BUFFER_SIZE 256
#define EMOTION_EXPIRATION_MILLISECONDS 200

#define ANIMATION_NAME_BUFFER_SIZE 256
#define ANIMATION_EXPIRATION_MILLISECONDS 200

#define LOOK_AT_RANGE_DEGREES 120

#define TGALLERY_SCRIPT_FILE "Scripts/tgallery.lua"
/* jenss - made a new file for CAudioBuffer TODO: delete if sound plays
// 
// //////////////////////////////////////////////////////////////////////////
// class CAudioBuffer
// {
// public:
// 	CAudioBuffer( int bufferBytes )
// 		: m_bufferBytes( bufferBytes )
// 		, m_dataStart( 0 )
// 		, m_dataBytes( 0 )
// 	{
// 		assert( 0 < GetBufferBytes() );
// 
// 		m_buffer = new byte[ GetBufferBytes() ];
// 		memset( m_buffer, 0, GetBufferBytes() );
// 	}
// 
// 	~CAudioBuffer()
// 	{
// 		delete[] m_buffer;
// 	}
// 
// 	void WriteData( const byte* pDataIn, int dataBytes, int offsetBytes = 0 );
// 	void ReadData( byte* pDataOut, int dataBytes, int offsetBytes = 0 );
// 	
// 	void Advance( int dataBytes )
// 	{
// 		int maxAdvance = std::min( dataBytes, m_dataBytes );
// 		m_dataStart += maxAdvance;
// 		m_dataBytes -= maxAdvance; 
// 	}
// 
// 	void Clear()
// 	{
// 		m_dataBytes = 0;
// 		m_dataStart = 0;
// 	}
// 
// 	int GetBytesInBuffer() const { return m_dataBytes; }
// 	int GetBufferBytes() const { return m_bufferBytes; }
// 
// 	byte* m_buffer;
// 	int m_bufferBytes;
// 	int m_dataStart;
// 	int m_dataBytes;
// };
// 
// void CAudioBuffer::WriteData( const byte* pDataIn, int dataBytes, int offsetBytes )
// {
// 	if ( GetBufferBytes() < dataBytes )
// 	{
// 		if ( g_pGame->GetCVars()->tgallery_debugAudioPackets == 1 )
// 		{
// 			CryLog( "audio couldn't be written to buffer: Buffer full." );
// 		}
// 		return;
// 	}
// 
// 	if ( dataBytes == 0 )
// 	{
// 		if ( g_pGame->GetCVars()->tgallery_debugAudioPackets == 1 )
// 		{
// 			CryLog( "audio couldn't be written to buffer: No Data." );
// 		}
// 		return;
// 	}
// 
// 	int writeStart = ( ( m_dataStart + offsetBytes ) + GetBufferBytes() ) % GetBufferBytes();
// 	bool wrapAroundWrite = ( GetBufferBytes() < writeStart + dataBytes );
// 	if ( wrapAroundWrite )
// 	{
// 		int bytesBeforeWrapAround = GetBufferBytes() - writeStart;
// 		memcpy( m_buffer + writeStart, pDataIn, bytesBeforeWrapAround );
// 		memcpy( m_buffer, pDataIn + bytesBeforeWrapAround, dataBytes - bytesBeforeWrapAround );
// 	}
// 	else
// 	{
// 		memcpy( m_buffer + writeStart, pDataIn, dataBytes );
// 	}
// 
// 	m_dataBytes = std::min( std::max( m_dataBytes, offsetBytes + dataBytes ), GetBufferBytes() );
// }
// 
// void CAudioBuffer::ReadData( byte* pDataOut, int dataBytes, int offsetBytes )
// {
// 	if ( GetBufferBytes() < dataBytes )
// 	{
// 		if ( g_pGame->GetCVars()->tgallery_debugAudioPackets == 1 )
// 		{
// 			CryLog( "audio couldn't be read from buffer." );
// 		}
// 		return;
// 	}
// 
// 	if ( dataBytes == 0 )
// 	{
// 		return;
// 	}
// 
// 	int readStart = ( ( m_dataStart + offsetBytes ) + GetBufferBytes() ) % GetBufferBytes();
// 	bool wrapAroundRead = ( GetBufferBytes() < readStart + dataBytes );
// 
// 	if ( wrapAroundRead )
// 	{
// 		int bytesBeforeWrapAround = GetBufferBytes() - readStart;
// 		memcpy( pDataOut, m_buffer + readStart, bytesBeforeWrapAround );
// 		memcpy( pDataOut + bytesBeforeWrapAround, m_buffer, dataBytes - bytesBeforeWrapAround );
// 	}
// 	else
// 	{
// 		memcpy( pDataOut, m_buffer + readStart, dataBytes );
// 	}
// }
*/

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

int ConvertToAudioBytes( int64 milliseconds )
{
	return ( int )( ( milliseconds * AUDIO_SAMPLING_RATE_HZ * AUDIO_BYTES_PER_SAMPLE ) / 1000 );
}

struct SAudioPayload
{
	// TODO: this is misleading, as it's not really safe to use the data when another packet is received.
	byte data[ AUDIO_PACKET_SIZE ];
};

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

CTalkie::CTalkie()
: m_pEntity( NULL )
, m_pCharacterInstance( NULL )
, m_pAnimationSet( NULL )
, m_pSkeletonAnim( NULL )
, m_pFacialInstance( NULL )
, m_pFacialEffectorsLibrary( NULL )
, m_rtpServer( m_config.GetRTPPort() )
, m_sound( NULL )
, m_pAudioBuffer( NULL )
, m_audioBytesRead( 0 )
, m_audioPacketSSRC( 0 )
, m_emotionEffectorChannel( -1 )
, m_faceDetected( false )
, m_lookAtScreenX( 0.5f )
, m_lookAtScreenY( 0.5f )
{
	GetISystem()->GetIGame()->GetIGameFramework()->RegisterListener( this, "CTalkie", FRAMEWORKLISTENERPRIORITY_GAME );
	GetISystem()->GetIGame()->GetIGameFramework()->GetILevelSystem()->AddListener( this );
	m_rtpServer.SetPacketListener( this );
	m_pAudioBuffer = new CAudioBuffer( m_config.GetAudioBufferBytes() );
	GetISystem()->GetIScriptSystem()->ExecuteFile( TGALLERY_SCRIPT_FILE );
//  LoadMap();
}

//////////////////////////////////////////////////////////////////////////
CTalkie::~CTalkie()
{
	GetISystem()->GetIGame()->GetIGameFramework()->GetILevelSystem()->RemoveListener( this );
	GetISystem()->GetIGame()->GetIGameFramework()->UnregisterListener( this );

	delete m_pAudioBuffer;
}

//////////////////////////////////////////////////////////////////////////
void CTalkie::LoadMap()
{
	if ( m_config.GetMapName() == NULL )
	{
		return;
	}

	string mapCommand = string( "map " ) + string( m_config.GetMapName() );
	gEnv->pConsole->ExecuteString( mapCommand );
}

//////////////////////////////////////////////////////////////////////////
bool CTalkie::SetLocalSound()
{
  m_sound = GetISystem()->GetISoundSystem()->CreateNetworkSound(
    this,
    AUDIO_BITS_PER_SAMPLE,
    AUDIO_SAMPLING_RATE_HZ,
    ConvertToAudioBytes( m_config.GetNetworkBufferMilliseconds() ) / AUDIO_BYTES_PER_SAMPLE,
    m_pEntity->GetId() );
  return (m_sound == NULL);
}

//////////////////////////////////////////////////////////////////////////
void CTalkie::TryPlaySound()
{
  IEntitySoundProxy* pSoundProxy = ( IEntitySoundProxy* ) m_pEntity->GetProxy( ENTITY_PROXY_SOUND );
  if ( pSoundProxy == NULL )
  {
    pSoundProxy = ( IEntitySoundProxy* ) m_pEntity->CreateProxy( ENTITY_PROXY_SOUND );
  }

  if ( pSoundProxy && m_sound )
  {
    pSoundProxy->PlaySound( m_sound );
    pSoundProxy->SetEffectRadius( 10 );
  }

  /*	m_sound->Play();*/
}

//////////////////////////////////////////////////////////////////////////
void CTalkie::FillListOfEffectors()
{
  for ( int visimId = 0; visimId < m_config.GetNumVisims(); visimId++ )
  {
    const char* effectorName = m_config.GetVisimEffectorName( visimId );
    if ( effectorName == NULL )
    {
      m_effectors.push_back( NULL );
    }
    else
    {
      m_effectors.push_back( m_pFacialEffectorsLibrary->Find( effectorName ) );
    }
  }
}

//////////////////////////////////////////////////////////////////////////
bool CTalkie::SetupLocalInstances()
{
  m_pEntity = GetISystem()->GetIEntitySystem()->FindEntityByName( m_config.GetCharacterName() );
  if ( m_pEntity == NULL )
  {
    return false;
  }
  //GetISystem()->GetIEntitySystem()->AddEntityEventListener( m_pEntity->GetId(), ENTITY_EVENT_DONE, this );

  m_pCharacterInstance = m_pEntity->GetCharacter( 0 );
  if ( m_pCharacterInstance == NULL )
  {
    return false;
  }

  m_pAnimationSet = m_pCharacterInstance->GetIAnimationSet();
  if ( m_pAnimationSet == NULL )
  {
    return false;
  }

  m_pSkeletonAnim = m_pCharacterInstance->GetISkeletonAnim();
  if ( m_pSkeletonAnim == NULL )
  {
    return false;
  }

  m_pSkeletonPose = m_pCharacterInstance->GetISkeletonPose();
  if ( m_pSkeletonPose == NULL )
  {
    return false;
  }

  m_pCharacterInstance->EnableFacialAnimation( true );
  m_pFacialInstance = m_pCharacterInstance->GetFacialInstance();
  if ( m_pFacialInstance == NULL )
  {
    return false;
  }

  m_pFacialEffectorsLibrary = m_pFacialInstance->GetFacialModel()->GetLibrary();
  if ( m_pFacialEffectorsLibrary == NULL )
  {
    return false;
  }

  return true;
}

//////////////////////////////////////////////////////////////////////////
void CTalkie::OnLoadingComplete( ILevel* pLevel )
{
  if ( pLevel == NULL )
    return;

  if ( !SetupLocalInstances() )
    return;

  FillListOfEffectors();
	ClearVisimQueue();
	ClearEmotionQueue();
	ClearAnimationQueue();
	SetDefaultAnimation();
  if ( !SetLocalSound() )
    return;

  TryPlaySound();

	m_pAudioBuffer->Clear();
	gEnv->bMultiplayer = true;
}

//////////////////////////////////////////////////////////////////////////
void CTalkie::OnEntityEvent( IEntity* pEntity, SEntityEvent& event )
{
	if ( event.event == ENTITY_EVENT_DONE )
	{
		assert( pEntity != NULL );
		GetISystem()->GetIEntitySystem()->RemoveEntityEventListener( pEntity->GetId(), ENTITY_EVENT_DONE, this );

		m_pEntity = NULL;
		m_pCharacterInstance = NULL;
		m_pAnimationSet = NULL;
		m_pSkeletonAnim = NULL;
		m_pSkeletonPose = NULL;
		m_pFacialInstance = NULL;
		m_pFacialEffectorsLibrary = NULL;
		m_effectors.clear();
	}
}

//////////////////////////////////////////////////////////////////////////
void CTalkie::Init(){}

//////////////////////////////////////////////////////////////////////////
void CTalkie::OnPostUpdate( float deltaTime )
{
	if ( m_pCharacterInstance == NULL )
	{
		return;
	}

	if ( m_pFacialInstance == NULL )
	{
		return;
	}

	if ( GetISystem()->GetIInput()->InputState( TKeyName( "1" ), eIS_Down ) )
	{
		m_audioPacketSSRC = 0;
		m_rtpServer.ClearSynchronisationEntries();

		m_pAudioBuffer->Clear();
		RemoveVisimEffectors();
		RemoveEmotionEffectors();
		SetDefaultAnimation();

		ClearVisimQueue();
		ClearEmotionQueue();
		ClearAnimationQueue();

		GetISystem()->GetIScriptSystem()->UnloadScript( TGALLERY_SCRIPT_FILE );
		GetISystem()->GetIScriptSystem()->ExecuteFile( TGALLERY_SCRIPT_FILE );

	}

// 	static int safePress = 0;
// 	safePress++;
// 	if ( GetISystem()->GetIInput()->InputState( TKeyName( "2" ), eIS_Down ) )
// 	{
// 		if ( 100 < safePress )
// 		{
// 			static bool selectA = true;
// 			if ( selectA )
// 			{
// 				SetAnimation( "prone_idleSelectUB_blowtorch_01" );
// 			}
// 			else
// 			{
// 				SetDefaultAnimation();
// 			}
// 			selectA = ! selectA;
// 			safePress = 0;
// 		}
// 	}

	m_rtpServer.Update();

	int64 wallclockTimestamp = m_rtpServer.GetWallclockTimestamp();
	UpdateVisims( wallclockTimestamp );
	UpdateEmotions( wallclockTimestamp );
	UpdateAnimations( wallclockTimestamp );
	UpdateLookAt();

	IRenderer* pRenderer = GetISystem()->GetIRenderer();

// 	pRenderer->Set2DMode( true, 800, 600 );
// 	pRenderer->Draw2dImage( 0, 0, 100, 50, 0 );
// 	pRenderer->Draw2dImage( 0, 0, float( ( 100 * m_pAudioBuffer->GetBytesInBuffer() ) / m_pAudioBuffer->GetBufferBytes() ), 50, 0, 0.f, 0.f, 1.f, 1.f, 0.f, 1.f, 0.f, 0.f );
// 	pRenderer->Set2DMode( false, 800, 600 );
}

//////////////////////////////////////////////////////////////////////////
void CTalkie::OnRTPPacket( SFixedRTPHeader* pRtpHeader, byte* pRtpPayload, int payloadBytes )
{
	if ( pRtpHeader->payloadType == PAYLOAD_TYPE_AUDIO )
	{
		SAudioPayload* pAudioPayload = ( SAudioPayload* )( pRtpPayload );
		OnRTPAudio( pRtpHeader, pAudioPayload, payloadBytes );
	}
	else if ( pRtpHeader->payloadType == PAYLOAD_TYPE_VISIM )
  {
    OnRTPVisim( pRtpHeader, pRtpPayload, payloadBytes );
	}
	else if ( pRtpHeader->payloadType == PAYLOAD_TYPE_ANIMATION )
	{
		OnRTPAnimation( pRtpHeader, pRtpPayload, payloadBytes );
	}
	else if ( pRtpHeader->payloadType == PAYLOAD_TYPE_EMOTION )
	{
		OnRTPEmotion( pRtpHeader, pRtpPayload, payloadBytes );
	}
	else if ( pRtpHeader->payloadType == PAYLOAD_TYPE_LOOKAT )
	{
		OnRTPLookAt( pRtpHeader, pRtpPayload, payloadBytes );
	}
}

//////////////////////////////////////////////////////////////////////////
bool CTalkie::FillDataBuffer( unsigned int bitsPerSample, unsigned int samplesPerSecond, unsigned int numSamples, void* pData )
{
	CryAutoCriticalSection lock( m_audioLock );

	int bytesToRead = numSamples * bitsPerSample / 8;

	int64 millisecondsToAudioStart = m_wallclockOfAudioStart - m_rtpServer.GetWallclockTimestamp();
	int bytesToAudioStart = ConvertToAudioBytes( millisecondsToAudioStart );
	bool inBufferingTime = ( 0 < bytesToAudioStart );
	if ( inBufferingTime )
	{
		bool completelyInBufferingTime = ( bytesToRead < bytesToAudioStart );
		if ( completelyInBufferingTime )
		{
			if ( g_pGame->GetCVars()->tgallery_debugAudioPackets == 1 )
			{
				CryLog( "silent %d", bytesToRead );
			}
			memset( pData, 0, bytesToRead );
		}
		else
		{
			if ( g_pGame->GetCVars()->tgallery_debugAudioPackets == 1 )
			{
				CryLog( "Playing %d silent, %d buffered", bytesToAudioStart, bytesToRead - bytesToAudioStart );
			}
			memset( pData, 0, bytesToAudioStart );
			m_pAudioBuffer->ReadData( ( byte* )( pData ) + bytesToAudioStart, bytesToRead - bytesToAudioStart );
			m_pAudioBuffer->Advance( bytesToRead - bytesToAudioStart );
			m_audioBytesRead += ( bytesToRead - bytesToAudioStart );
		}
	}
	else
	{
		int bytesInBuffer = m_pAudioBuffer->GetBytesInBuffer();
		bool readingMoreBytesThanAvailable = ( bytesInBuffer < bytesToRead );
		if ( readingMoreBytesThanAvailable )
		{
			m_pAudioBuffer->ReadData( ( byte* )( pData ), bytesInBuffer );
			memset( ( byte* )( pData ) + bytesInBuffer, 0, bytesToRead - bytesInBuffer );
			//m_pAudioBuffer->Advance( bytesInBuffer ); // tmp
			//m_audioBytesRead += bytesInBuffer; // tmp
		}
		else
		{
			if ( g_pGame->GetCVars()->tgallery_debugAudioPackets == 1 )
			{
				CryLog( "Playing %d buffered", bytesToRead );
			}
			m_pAudioBuffer->ReadData( ( byte* )( pData ), bytesToRead );
			//m_pAudioBuffer->Advance( bytesToRead ); //tmp here
			//m_audioBytesRead += bytesToRead; // tmp here
		}

		m_pAudioBuffer->Advance( bytesToRead ); //tmp here
		m_audioBytesRead += bytesToRead; // tmp here
		
	}

	return true;
}

//////////////////////////////////////////////////////////////////////////
void CTalkie::OnRTPVisim( SFixedRTPHeader* pHeader, byte* pData, int payloadBytes )
{
	QueueVisim(
    m_rtpServer.GetWallclockTimestamp(),
    ConvertToWallclockTimestamp(pHeader, VISIM_SAMPLING_RATE_HZ),
    (int) pData[ 0 ] );
}

//////////////////////////////////////////////////////////////////////////
void CTalkie::OnRTPAudio( SFixedRTPHeader* pHeader, SAudioPayload* pPayload, int payloadBytes )
{
	CryAutoCriticalSection lock( m_audioLock );

	if ( g_pGame->GetCVars()->tgallery_changeAudioPackageEndianness != 0 )
	{
		int16* audioData = ( int16* )( pPayload );
		for ( int i = 0; i < ( payloadBytes / 2 ); i++ )
		{
			audioData[ i ] = ntohs( audioData[ i ] );
		}
	}

	bool audioSourceHasChanged = ( pHeader->synchornisationSourceId != m_audioPacketSSRC );
	if ( audioSourceHasChanged )
	{
		m_audioBytesRead = 0;
		m_pAudioBuffer->Clear();
		int64 wallclockTimestamp = m_rtpServer.GetWallclockTimestamp();
		m_wallclockOfAudioStart = wallclockTimestamp + m_config.GetBufferMilliseconds() - m_config.GetNetworkBufferMilliseconds();
		m_audioPacketSSRC = pHeader->synchornisationSourceId;

		if ( g_pGame->GetCVars()->tgallery_debugAudioPackets == 1 )
		{
			CryLog( "Audio source changed. Starting audio at %d, current time %d", m_wallclockOfAudioStart, wallclockTimestamp );
		}
	}

	int byteOffsetOfPacket = pHeader->timestamp * AUDIO_BYTES_PER_SAMPLE;
	int byteOffsetOfPacketInAudioQueue = byteOffsetOfPacket - m_audioBytesRead;
	if ( g_pGame->GetCVars()->tgallery_debugAudioPackets == 1 )
	{
		CryLog( "Audio: absolute offset: %d offset in queue: %d", byteOffsetOfPacket, byteOffsetOfPacketInAudioQueue );
	}
	m_pAudioBuffer->WriteData( pPayload->data, payloadBytes, byteOffsetOfPacketInAudioQueue );
}

//////////////////////////////////////////////////////////////////////////
int64 CTalkie::ConvertToWallclockTimestamp(SFixedRTPHeader* pHeader, int64 frequency)
{
  return m_rtpServer.ConvertToWallclockTimestamp( pHeader, frequency ) + m_config.GetBufferMilliseconds();
}

//////////////////////////////////////////////////////////////////////////
void CTalkie::OnRTPEmotion( SFixedRTPHeader* pHeader, byte* pData, int payloadBytes )
{
	if ( m_pFacialEffectorsLibrary == NULL )
	{
		return;
	}

	char emotionNameBuffer[ EMOTION_NAME_BUFFER_SIZE ];
	float intensity = 1;

	int elementsRead = sscanf_s( ( char* )( pData ), "%s %f", emotionNameBuffer, sizeof( emotionNameBuffer ), &intensity );
	if ( elementsRead == 2 )
	{
    QueueEmotion(m_rtpServer.GetWallclockTimestamp(),
      ConvertToWallclockTimestamp(pHeader, EMOTION_SAMPLING_RATE_HZ),
      m_pFacialEffectorsLibrary->Find( emotionNameBuffer ),
      intensity);
	}
}

//////////////////////////////////////////////////////////////////////////
void CTalkie::OnRTPAnimation( SFixedRTPHeader* pHeader, byte* pData, int payloadBytes )
{
	char animationNameBuffer[ ANIMATION_NAME_BUFFER_SIZE ];

	int elementsRead = sscanf_s( ( char* )( pData ), "%s", animationNameBuffer, sizeof( animationNameBuffer ) );
	if ( elementsRead == 1 )
	{
    QueueAnimation(m_rtpServer.GetWallclockTimestamp(),
      ConvertToWallclockTimestamp(pHeader, ANIMATION_SAMPLING_RATE_HZ),
      animationNameBuffer );
	}
}

//////////////////////////////////////////////////////////////////////////
float ExtractFloat(byte* pData)
{
  char dataString[ 7 ] = { 0 };
  memcpy( dataString, pData, 6 );
  float returnValue;
  sscanf_s( dataString, "%f", &returnValue );
  return returnValue;
}

//////////////////////////////////////////////////////////////////////////
void CTalkie::OnRTPLookAt( SFixedRTPHeader* pHeader, byte* pData, int payloadBytes )
{
	// Play the latest received look at packet right away ( ignore timestamp info )
  float lookAtX = ExtractFloat(pData + 1);
	float lookAtY = ExtractFloat(pData + 7); 

  bool enableLookAt = ( pData[ 0 ] != '1' );
	SetLookAtParameters( enableLookAt, lookAtX, lookAtY );
}

//////////////////////////////////////////////////////////////////////////
void CTalkie::ClearVisimQueue()
{
	while ( ! m_visimPlayQueue.empty() )
	{
		m_visimPlayQueue.pop();
	}
}

//////////////////////////////////////////////////////////////////////////
void CTalkie::QueueVisim( int64 wallclockTimestamp, int64 visimWallclockTimestamp, int visimId )
{
	bool isVeryOldVisim = ( visimWallclockTimestamp + m_config.GetVisimExpirationMilliseconds() < wallclockTimestamp );
	if ( isVeryOldVisim )
	{
		return;
	}

	VisimInfo vi;
	vi.wallclockTimestamp = visimWallclockTimestamp;
	vi.visimId = visimId;

	m_visimPlayQueue.push( vi );
}

void CTalkie::RemoveVisimEffectors()
{
	m_pFacialInstance->PreviewEffector( NULL, 0, 0 );
}

//////////////////////////////////////////////////////////////////////////
void CTalkie::UpdateVisims( int64 wallclockTimestamp )
{
	if ( m_pFacialInstance == NULL )
	{
		return;
	}

	if ( m_visimPlayQueue.empty() )
	{
		return;
	}

	int visimId = -1;
	bool lookAtNextVisim = true;
	while ( lookAtNextVisim )
	{
		const VisimInfo& vi = m_visimPlayQueue.top();
		if ( vi.wallclockTimestamp < wallclockTimestamp )
		{
			visimId = vi.visimId;
			m_visimPlayQueue.pop();
			if ( m_visimPlayQueue.empty() )
			{
				lookAtNextVisim = false;
			}
		}
		else
		{
			lookAtNextVisim = false;
		}
	}

	if ( visimId != -1 )
	{
		if ( 0 <= visimId && visimId < m_effectors.size() )
		{
			m_pFacialInstance->PreviewEffector( m_effectors[ visimId ], 1, 0 );
		}
		else
		{
			m_pFacialInstance->PreviewEffector( NULL, 1, 0 );
		}
	}
}
//////////////////////////////////////////////////////////////////////////
void CTalkie::ClearEmotionQueue()
{
	while ( ! m_emotionPlayQueue.empty() )
	{
		m_emotionPlayQueue.pop();
	}
}

//////////////////////////////////////////////////////////////////////////
void CTalkie::QueueEmotion( int64 wallclockTimestamp, int64 emotionWallclockTimestamp, IFacialEffector* emotionEffector, float intensity )
{
	bool isVeryOldEmotion = ( emotionWallclockTimestamp + EMOTION_EXPIRATION_MILLISECONDS < wallclockTimestamp );
	if ( isVeryOldEmotion )
	{
		return;
	}

	EmotionInfo ei;
	ei.wallclockTimestamp = emotionWallclockTimestamp;
	ei.emotionEffector = emotionEffector;
	ei.emotionIntensity = intensity;

	m_emotionPlayQueue.push( ei );
}

//////////////////////////////////////////////////////////////////////////
void CTalkie::UpdateEmotions( int64 wallclockTimestamp )
{
	if ( m_pFacialInstance == NULL )
	{
		return;
	}

	if ( m_emotionPlayQueue.empty() )
	{
		return;
	}

	IFacialEffector* emotionEffector = NULL;
	float intensity = 0;
	bool emotionFound = false;
	bool lookAtNextEmotion = true;
	while ( lookAtNextEmotion )
	{
		const EmotionInfo& ei = m_emotionPlayQueue.top();
		if ( ei.wallclockTimestamp < wallclockTimestamp )
		{
			emotionFound = true;
			emotionEffector = ei.emotionEffector;
			intensity = ei.emotionIntensity;
			
			m_emotionPlayQueue.pop();
			if ( m_emotionPlayQueue.empty() )
			{
				lookAtNextEmotion = false;
			}
		}
		else
		{
			lookAtNextEmotion = false;
		}
	}

	if ( emotionFound )
	{
		RemoveEmotionEffectors();
		m_emotionEffectorChannel = ( int32 )( m_pFacialInstance->StartEffectorChannel( emotionEffector, intensity, 0.2f ) );
	}
}

//////////////////////////////////////////////////////////////////////////
void CTalkie::RemoveEmotionEffectors()
{
	if ( 0 <= m_emotionEffectorChannel )
	{
		m_pFacialInstance->StopEffectorChannel( m_emotionEffectorChannel, 0.2f );
	}
	m_emotionEffectorChannel = -1;
}

//////////////////////////////////////////////////////////////////////////

void CTalkie::ClearAnimationQueue()
{
	while ( ! m_emotionPlayQueue.empty() )
	{
		m_emotionPlayQueue.pop();
	}
}

//////////////////////////////////////////////////////////////////////////
void CTalkie::QueueAnimation( int64 wallclockTimestamp, int64 animationWallclockTimestamp, const char* animationName )
{
	bool isVeryOldEmotion = ( animationWallclockTimestamp + ANIMATION_EXPIRATION_MILLISECONDS < wallclockTimestamp );
	if ( isVeryOldEmotion )
	{
		return;
	}

	AnimationInfo ai;
	ai.wallclockTimestamp = animationWallclockTimestamp;
	sprintf_s( ai.animationName, "%s", animationName );
	ai.animationIntensity = 1;

	m_animationPlayQueue.push( ai );
}

//////////////////////////////////////////////////////////////////////////
void CTalkie::UpdateAnimations( int64 wallclockTimestamp )
{
	if ( m_pSkeletonAnim == NULL )
	{
		return;
	}

	if ( m_animationPlayQueue.empty() )
	{
		return;
	}

	string animationName;
	float intensity = 0;
	bool animationFound = false;
	bool lookAtNextAnimation = true;
	while ( lookAtNextAnimation )
	{
		const AnimationInfo& ai = m_animationPlayQueue.top();
		if ( ai.wallclockTimestamp < wallclockTimestamp )
		{
			animationFound = true;
			animationName = ai.animationName;
			intensity = ai.animationIntensity;

			m_animationPlayQueue.pop();
			if ( m_animationPlayQueue.empty() )
			{
				lookAtNextAnimation = false;
			}
		}
		else
		{
			lookAtNextAnimation = false;
		}
	}

	if ( animationFound )
	{
		SetAnimation( animationName.c_str() );
	}
}

//////////////////////////////////////////////////////////////////////////
void CTalkie::SetDefaultAnimation()
{
	SetAnimation( m_config.GetInitialAnimationName() );
}

//////////////////////////////////////////////////////////////////////////
void CTalkie::SetAnimation( const char* animationName )
{
	if ( m_pSkeletonAnim == NULL )
	{
		return;
	}

	if ( animationName == NULL )
	{
    assert( false);
		return;
	}

	CryCharAnimationParams animationParams;
	animationParams.m_nFlags |= CA_LOOP_ANIMATION;
	animationParams.m_fTransTime = 0.2f;
	m_pSkeletonAnim->StartAnimation( animationName, animationParams );
}

//////////////////////////////////////////////////////////////////////////
void CTalkie::SetLookAtParameters( bool faceDetected, float screenX, float screenY )
{
	m_faceDetected = faceDetected;
	m_lookAtScreenX = screenX;
	m_lookAtScreenY = 1 - screenY;
}

//////////////////////////////////////////////////////////////////////////
void GetVectorFromScriptTable(IScriptTable *table, Vec3* vec)
{
  table->GetValue( "x", vec->x );
  table->GetValue( "y", vec->y );
  table->GetValue( "z", vec->z );
}

//////////////////////////////////////////////////////////////////////////
void UpdateLookAtPositionFromScriptSystem(Vec3* lookAtPosition)
{
  ScriptAnyValue out;
  GetISystem()->GetIScriptSystem()->EndCallAny( out );

  if ( out.type == ANY_TVECTOR )
  {
    lookAtPosition->x = out.vec3.x;
    lookAtPosition->y = out.vec3.y;
    lookAtPosition->z = out.vec3.z;
  }
  else if ( out.type == ANY_TTABLE )
  {
    GetVectorFromScriptTable(out.table, lookAtPosition);
  }
}

//////////////////////////////////////////////////////////////////////////
void CTalkie::UpdateLookAt()
{
	if ( m_pSkeletonPose == NULL )
	{
		return;
	}

	if ( g_pGame->GetCVars()->tgallery_enableLookAt == 0 )
	{
		return;
	}

	assert( m_pEntity != NULL );

	Vec3 lookAtPosition = Vec3( 0, 0, 0 );
	const CCamera& camera = gEnv->pRenderer->GetCamera();

	Vec3 cameraPosition = camera.GetPosition();
	Vec3 entityPosition = m_pEntity->GetPos();

	if ( m_faceDetected )
	{
		float faceX = ( m_lookAtScreenX * 2 ) - 1;
		float faceY = ( m_lookAtScreenY * 2 ) - 1;

		const Vec3 up( 0, 0, 1 );
		Vec3 cameraToEntity = ( entityPosition - cameraPosition );
		cameraToEntity.z = 0;
		cameraToEntity.NormalizeSafe( Vec3( 1, 0, 0 ) );
		Vec3 right = cameraToEntity.Cross( up );

    lookAtPosition = cameraPosition + right * faceX * g_pGame->GetCVars()->tgallery_virtualScreenWidth + up * faceY * g_pGame->GetCVars()->tgallery_virtualScreenHeight;

		IScriptSystem* pScriptSystem = GetISystem()->GetIScriptSystem();
		if ( pScriptSystem->BeginCall( "TGalleryLookAt" ) )
		{
			pScriptSystem->PushFuncParam( cameraPosition );
			pScriptSystem->PushFuncParam( entityPosition );
			pScriptSystem->PushFuncParam( m_lookAtScreenX );
			pScriptSystem->PushFuncParam( m_lookAtScreenY );
			pScriptSystem->PushFuncParam( lookAtPosition );

      UpdateLookAtPositionFromScriptSystem(&lookAtPosition);
		}
	}
	else
	{
		Vec3 entityForwardDirection = m_pEntity->GetForwardDir();

		lookAtPosition = entityPosition + entityForwardDirection;
		lookAtPosition.z = cameraPosition.z;

		IScriptSystem* pScriptSystem = GetISystem()->GetIScriptSystem();
		if ( pScriptSystem->BeginCall( "TGalleryLookAt_NoFaceDetected" ) )
		{
			pScriptSystem->PushFuncParam( cameraPosition );
			pScriptSystem->PushFuncParam( entityPosition );
			pScriptSystem->PushFuncParam( entityForwardDirection );
			pScriptSystem->PushFuncParam( lookAtPosition );

      UpdateLookAtPositionFromScriptSystem(&lookAtPosition);
		}
	}

	m_pSkeletonPose->SetLookIK( true, DEG2RAD( LOOK_AT_RANGE_DEGREES ), lookAtPosition );
}