#ifndef __Talkie__h__
#define __Talkie__h__

#include <ILevelSystem.h>
#include <IGameFramework.h>
#include <IEntitySystem.h>
#include <ISound.h>

#include <Talkie/RTPSimpleServer.h>
#include <Talkie/TGalleryConfiguration.h>
#include <Talkie/TalkieAudioBuffer.h>

struct IEntity;
struct ICharacterInstance;
struct IAnimationSet;
struct ISkeletonAnim;
struct ISkeletonPose;
struct IFacialEffectorsLibrary;

struct IFacialEffector;
struct IFacialInstance;

struct SAudioPayload;

//class CAudioBuffer;

class CTalkie
	: public ILevelSystemListener
	, public IGameFrameworkListener
	, public IRTPPacketListener
	, public INetworkSoundListener
	, public IEntityEventListener
{
	struct ClientConnection;
public:
	CTalkie();
	virtual ~CTalkie();

	void LoadMap();

	// ILevelSystemListener
	virtual void OnLevelNotFound( const char* levelName ) {}
	virtual void OnLoadingStart( ILevelInfo* pLevel ) {}
	virtual void OnLoadingComplete( ILevel* pLevel );
	virtual void OnLoadingError( ILevelInfo* pLevel, const char* error ) {}
	virtual void OnLoadingProgress( ILevelInfo* pLevel, int progressAmount ) {}
	// ~ILevelSystemListener

	// IGameFrameworkListener
	virtual void OnPostUpdate( float deltaTime );
	virtual void OnSaveGame( ISaveGame* pSaveGame ) {}
	virtual void OnLoadGame( ILoadGame* pLoadGame ) {}
	virtual void OnLevelEnd( const char* nextLevel ) {}
	virtual void OnActionEvent( const SActionEvent& event ) {}
	// ~IGameFrameworkListener

	// IRTPPacketListener
	virtual void OnRTPPacket( SFixedRTPHeader* pRtpHeader, byte* pRtpPayload, int payloadBytes );
	// ~IRTPPacketListener

	// INetworkSoundListener
	virtual bool FillDataBuffer( unsigned int bitsPerSample, unsigned int samplesPerSecond, unsigned int numSamples, void* pData );
	virtual void OnActivate( bool active ) {}
	virtual void SetSoundPlaybackDistances( float max3dDistance, float min2dDistance ) {}
	virtual void UpdateSound3dPan() {}
	// ~INetworkSoundListener

	// IEntityEventListener
	virtual void OnEntityEvent( IEntity* pEntity, SEntityEvent& event );
	// ~IEntityEventListener


	// VisimeController
	void ClearVisimQueue();
	void QueueVisim( int64 wallclockTimestamp, int64 visimWallclockTimestamp, int visimId );
	void UpdateVisims( int64 wallclockTimestamp );
	void RemoveVisimEffectors();
	// ~VisimeController

	// EmotionController
	void ClearEmotionQueue();
	void QueueEmotion( int64 wallclockTimestamp, int64 emotionWallclockTimestamp, IFacialEffector* emotionEffector, float intensity );
	void UpdateEmotions( int64 wallclockTimestamp );
	void RemoveEmotionEffectors();
	// ~EmotionController

	// AnimationController
	void ClearAnimationQueue();
	void QueueAnimation( int64 wallclockTimestamp, int64 animationWallclockTimestamp, const char* animationName );
	void UpdateAnimations( int64 wallclockTimestamp );
	void SetDefaultAnimation();
	void SetAnimation( const char* animationName );
	// ~AnimationController

	// LookAtController
	void SetLookAtParameters( bool faceDetected, float screenX, float screenY );
	void UpdateLookAt();
	// ~LookAtController


protected:

  void Init();

  void OnRTPVisim( SFixedRTPHeader* pHeader, byte* pData, int payloadBytes );
	void OnRTPAudio( SFixedRTPHeader* pHeader, SAudioPayload* pPayload, int payloadBytes );
	void OnRTPEmotion( SFixedRTPHeader* pHeader, byte* pData, int payloadBytes );
	void OnRTPAnimation( SFixedRTPHeader* pHeader, byte* pData, int payloadBytes );
	void OnRTPLookAt( SFixedRTPHeader* pHeader, byte* pData, int payloadBytes );

private:
  bool SetLocalSound();
  void TryPlaySound();
  void FillListOfEffectors();
  bool SetupLocalInstances();
  int64 ConvertToWallclockTimestamp(SFixedRTPHeader* pHeader, int64 frequency);

	CTGalleryConfiguration m_config;

	IEntity* m_pEntity;
	ICharacterInstance* m_pCharacterInstance;
	IAnimationSet* m_pAnimationSet;
	ISkeletonAnim* m_pSkeletonAnim;
	ISkeletonPose* m_pSkeletonPose;
	IFacialEffectorsLibrary* m_pFacialEffectorsLibrary;

	IFacialInstance* m_pFacialInstance;

	std::vector< IFacialEffector* > m_effectors;

	struct VisimInfo
	{
		int64 wallclockTimestamp;
		int visimId;

		bool operator < ( const VisimInfo& vi ) const
		{
			bool isLowerPriority = ( vi.wallclockTimestamp < wallclockTimestamp );
			return isLowerPriority;
		}
	};

	std::priority_queue< VisimInfo, std::vector< VisimInfo > > m_visimPlayQueue;

	CRTPSimpleServer m_rtpServer;

	ISound* m_sound;

	int m_audioBytesRead;
	CAudioBuffer* m_pAudioBuffer;
	uint32 m_audioPacketSSRC;
	int64 m_wallclockOfAudioStart;

	CryCriticalSection m_audioLock;

	struct EmotionInfo
	{
		int64 wallclockTimestamp;
		IFacialEffector* emotionEffector;
		float emotionIntensity;

		bool operator < ( const EmotionInfo& ei ) const
		{
			bool isLowerPriority = ( ei.wallclockTimestamp < wallclockTimestamp );
			return isLowerPriority;
		}
	};

	std::priority_queue< EmotionInfo, std::vector< EmotionInfo > > m_emotionPlayQueue;

	int32 m_emotionEffectorChannel;


	struct AnimationInfo
	{
		int64 wallclockTimestamp;
		char animationName[ 64 ];
		float animationIntensity;

		bool operator < ( const AnimationInfo& ai ) const
		{
			bool isLowerPriority = ( ai.wallclockTimestamp < wallclockTimestamp );
			return isLowerPriority;
		}
	};

	std::priority_queue< AnimationInfo, std::vector< AnimationInfo > > m_animationPlayQueue;

	bool m_faceDetected;
	float m_lookAtScreenX;
	float m_lookAtScreenY;
/*
public:
	static float tgallery_virtualScreenWidth;
	static float tgallery_virtualScreenHeight;
	static int tgallery_enableLookAt;

	static int tgallery_changeAudioPackageEndianness;

	static int tgallery_debugAudioPackets;
	static int tgallery_debugVisimPackets;
	static int tgallery_debugAnimationPackets;
	static int tgallery_debugEmotionPackets;
	static int tgallery_debugLookAtPackets;
  */
};

#endif