/*************************************************************************
  Crytek Source File.
  Copyright (C), Crytek Studios, 2001-2010.
 -------------------------------------------------------------------------
  History:
  - Jan/2010: Created by Jens Schbel

*************************************************************************/

#include "StdAfx.h"

#include "Game.h"
#include "GameCVars.h"

//#include <IEntitySystem.h>

#include "Talkie/TalkieActor.h"

#include <windows.h>
#undef PlaySound

#define PAYLOAD_TYPE_AUDIO 101
#define PAYLOAD_TYPE_VISIM 102
#define PAYLOAD_TYPE_ANIMATION 103
#define PAYLOAD_TYPE_EMOTION 104
#define PAYLOAD_TYPE_LOOKAT 105

#define AUDIO_BYTES_PER_SAMPLE 2
#define AUDIO_BITS_PER_SAMPLE ( AUDIO_BYTES_PER_SAMPLE * 8 )

#define AUDIO_SAMPLING_RATE_HZ 22050

#define VISIM_SAMPLING_RATE_HZ AUDIO_SAMPLING_RATE_HZ
#define EMOTION_SAMPLING_RATE_HZ AUDIO_SAMPLING_RATE_HZ
#define ANIMATION_SAMPLING_RATE_HZ AUDIO_SAMPLING_RATE_HZ

#define EMOTION_NAME_BUFFER_SIZE 256
#define EMOTION_EXPIRATION_MILLISECONDS 200

#define ANIMATION_NAME_BUFFER_SIZE 256
#define ANIMATION_EXPIRATION_MILLISECONDS 200

#define LOOK_AT_RANGE_DEGREES 120

#define TGALLERY_SCRIPT_FILE "Scripts/tgallery.lua"

namespace Talkie
{
  //////////////////////////////////////////////////////////////////////////
  int ConvertToAudioBytes( int64 milliseconds )
  {
    return ( int )( ( milliseconds * AUDIO_SAMPLING_RATE_HZ * AUDIO_BYTES_PER_SAMPLE ) / 1000 );
  }

  //////////////////////////////////////////////////////////////////////////
  float ExtractFloat(byte* pData)
  {
    char dataString[ 7 ] = { 0 };
    memcpy( dataString, pData, 6 );
    float returnValue;
    sscanf_s( dataString, "%f", &returnValue );
    return returnValue;
  }

  //////////////////////////////////////////////////////////////////////////
  void GetVectorFromScriptTable(IScriptTable *table, Vec3* vec)
  {
    table->GetValue( "x", vec->x );
    table->GetValue( "y", vec->y );
    table->GetValue( "z", vec->z );
  }

  //////////////////////////////////////////////////////////////////////////
  void UpdateLookAtPositionFromScriptSystem(Vec3* lookAtPosition)
  {
    ScriptAnyValue out;
    GetISystem()->GetIScriptSystem()->EndCallAny( out );

    if ( out.type == ANY_TVECTOR )
    {
      lookAtPosition->x = out.vec3.x;
      lookAtPosition->y = out.vec3.y;
      lookAtPosition->z = out.vec3.z;
    }
    else if ( out.type == ANY_TTABLE )
    {
      GetVectorFromScriptTable(out.table, lookAtPosition);
    }
  }

  //////////////////////////////////////////////////////////////////////////
  void LoadMap(const char* mapName)
  {
    if ( mapName != NULL )
    {
      gEnv->pConsole->ExecuteString( string( "map " ) + string( mapName ) );
    }
  }

} // namespace Talkie

using namespace Talkie;

//////////////////////////////////////////////////////////////////////////
CTalkieActor::CTalkieActor()
: m_pTalkieEntity( NULL )
, m_pCharacterInstance( NULL )
, m_pAnimationSet( NULL )
, m_pSkeletonAnim( NULL )
, m_pFacialInstance( NULL )
, m_pFacialEffectorsLibrary( NULL )
, m_rtpServer( m_config.GetRTPPort() )
, m_sound( NULL )
, m_pAudioBuffer( NULL )
, m_audioBytesRead( 0 )
, m_audioPacketSSRC( 0 )
, m_emotionEffectorChannel( -1 )
, m_faceDetected( false )
, m_lookAtScreenX( 0.5f )
, m_lookAtScreenY( 0.5f )
{
  GetISystem()->GetIGame()->GetIGameFramework()->RegisterListener( this, "CTalkie", FRAMEWORKLISTENERPRIORITY_GAME );
  GetISystem()->GetIGame()->GetIGameFramework()->GetILevelSystem()->AddListener( this );
  m_rtpServer.SetPacketListener( this );
  m_pAudioBuffer = new CAudioBuffer( m_config.GetAudioBufferBytes() );
  GetISystem()->GetIScriptSystem()->ExecuteFile( TGALLERY_SCRIPT_FILE );

  // TODO: hack to immediately load the map
  Talkie::LoadMap(m_config.GetMapName());
}

//////////////////////////////////////////////////////////////////////////
CTalkieActor::~CTalkieActor()
{
  GetISystem()->GetIGame()->GetIGameFramework()->GetILevelSystem()->RemoveListener( this );
  GetISystem()->GetIGame()->GetIGameFramework()->UnregisterListener( this );

  delete m_pAudioBuffer;
}

//////////////////////////////////////////////////////////////////////////
bool CTalkieActor::SetupLocalInstances()
{
  m_pTalkieEntity = GetISystem()->GetIEntitySystem()->FindEntityByName( m_config.GetCharacterName() );
  if ( m_pTalkieEntity == NULL )
  {
    return false;
  }

  m_pCharacterInstance = m_pTalkieEntity->GetCharacter( 0 );
  if ( m_pCharacterInstance == NULL )
  {
    return false;
  }

  m_pAnimationSet = m_pCharacterInstance->GetIAnimationSet();
  if ( m_pAnimationSet == NULL )
  {
    return false;
  }

  m_pSkeletonAnim = m_pCharacterInstance->GetISkeletonAnim();
  if ( m_pSkeletonAnim == NULL )
  {
    return false;
  }

  m_pSkeletonPose = m_pCharacterInstance->GetISkeletonPose();
  if ( m_pSkeletonPose == NULL )
  {
    return false;
  }

  m_pCharacterInstance->EnableFacialAnimation( true );
  m_pFacialInstance = m_pCharacterInstance->GetFacialInstance();
  if ( m_pFacialInstance == NULL )
  {
    return false;
  }

  m_pFacialEffectorsLibrary = m_pFacialInstance->GetFacialModel()->GetLibrary();
  if ( m_pFacialEffectorsLibrary == NULL )
  {
    return false;
  }

  return true;
}

//////////////////////////////////////////////////////////////////////////
void CTalkieActor::FillListOfEffectors()
{
  for ( int visimId = 0; visimId < m_config.GetNumVisims(); visimId++ )
  {
    const char* effectorName = m_config.GetVisimEffectorName( visimId );
    if ( effectorName == NULL )
    {
      m_effectors.push_back( NULL );
    }
    else
    {
      m_effectors.push_back( m_pFacialEffectorsLibrary->Find( effectorName ) );
    }
  }
}

//////////////////////////////////////////////////////////////////////////
bool CTalkieActor::SetLocalSound()
{
  m_sound = GetISystem()->GetISoundSystem()->CreateNetworkSound(
    this,
    AUDIO_BITS_PER_SAMPLE,
    AUDIO_SAMPLING_RATE_HZ,
    ConvertToAudioBytes( m_config.GetNetworkBufferMilliseconds() ) / AUDIO_BYTES_PER_SAMPLE,
    m_pTalkieEntity->GetId() );
  return (m_sound == NULL);
}

//////////////////////////////////////////////////////////////////////////
void CTalkieActor::TryPlaySound()
{
  IEntitySoundProxy* pSoundProxy = ( IEntitySoundProxy* ) m_pTalkieEntity->GetProxy( ENTITY_PROXY_SOUND );
  if ( pSoundProxy == NULL )
  {
    pSoundProxy = ( IEntitySoundProxy* ) m_pTalkieEntity->CreateProxy( ENTITY_PROXY_SOUND );
  }

  if ( pSoundProxy && m_sound )
  {
    pSoundProxy->PlaySound( m_sound );
    pSoundProxy->SetEffectRadius( 10 );
  }
}

// from ILevelSystemListener /////////////////////////////////////////////
void CTalkieActor::OnLoadingComplete( ILevel* pLevel )
{
  if ( pLevel == NULL )
    return;

  if ( !SetupLocalInstances() )
    return;

  FillListOfEffectors();
  ClearVisimQueue();
  ClearEmotionQueue();
  ClearAnimationQueue();
  SetDefaultAnimation();
  if ( !SetLocalSound() )
    return;

  TryPlaySound();

  m_pAudioBuffer->Clear();
  gEnv->bMultiplayer = true;
}

// from IGameFrameworkListener ///////////////////////////////////////////
void CTalkieActor::OnPostUpdate( float deltaTime )
{
  if ( m_pCharacterInstance == NULL )
  {
    return;
  }

  if ( m_pFacialInstance == NULL )
  {
    return;
  }

  if ( GetISystem()->GetIInput()->InputState( TKeyName( "1" ), eIS_Down ) )
  {
    m_audioPacketSSRC = 0;
    m_rtpServer.ClearSynchronisationEntries();

    m_pAudioBuffer->Clear();
    RemoveVisimEffectors();
    RemoveEmotionEffectors();
    SetDefaultAnimation();

    ClearVisimQueue();
    ClearEmotionQueue();
    ClearAnimationQueue();

    GetISystem()->GetIScriptSystem()->UnloadScript( TGALLERY_SCRIPT_FILE );
    GetISystem()->GetIScriptSystem()->ExecuteFile( TGALLERY_SCRIPT_FILE );

  }

  m_rtpServer.Update();

  int64 wallclockTimestamp = m_rtpServer.GetWallclockTimestamp();
  UpdateVisims( wallclockTimestamp );
  UpdateEmotions( wallclockTimestamp );
  UpdateAnimations( wallclockTimestamp );
  UpdateLookAt();
}

// from IRTPPacketListener ///////////////////////////////////////////////
void CTalkieActor::OnRTPPacket( SFixedRTPHeader* pRtpHeader, byte* pRtpPayload, int payloadBytes )
{

  switch (pRtpHeader->payloadType)
  {
  case PAYLOAD_TYPE_AUDIO:
    OnRTPAudio( pRtpHeader, ( SAudioPayload* )( pRtpPayload ), payloadBytes );
    break;
  case PAYLOAD_TYPE_VISIM:
    OnRTPVisim( pRtpHeader, pRtpPayload, payloadBytes );
    break;
  case PAYLOAD_TYPE_ANIMATION:
    OnRTPAnimation( pRtpHeader, pRtpPayload, payloadBytes );
    break;
  case PAYLOAD_TYPE_EMOTION:
    OnRTPEmotion( pRtpHeader, pRtpPayload, payloadBytes );
    break;
  case PAYLOAD_TYPE_LOOKAT:
    OnRTPLookAt( pRtpHeader, pRtpPayload, payloadBytes );
    break;
  }
}

//////////////////////////////////////////////////////////////////////////
void CTalkieActor::OnRTPVisim( SFixedRTPHeader* pHeader, byte* pData, int payloadBytes )
{
  QueueVisim(
    m_rtpServer.GetWallclockTimestamp(),
    ConvertToWallclockTimestamp(pHeader, VISIM_SAMPLING_RATE_HZ),
    (int) pData[ 0 ] );
}

//////////////////////////////////////////////////////////////////////////
void CTalkieActor::OnRTPAudio( SFixedRTPHeader* pHeader, SAudioPayload* pPayload, int payloadBytes )
{
  CryAutoCriticalSection lock( m_audioLock );

  if ( g_pGame->GetCVars()->tgallery_changeAudioPackageEndianness != 0 )
  {
    int16* audioData = ( int16* )( pPayload );
    for ( int i = 0; i < ( payloadBytes / 2 ); i++ )
    {
      audioData[ i ] = ntohs( audioData[ i ] );
    }
  }

  bool audioSourceHasChanged = ( pHeader->synchornisationSourceId != m_audioPacketSSRC );
  if ( audioSourceHasChanged )
  {
    m_audioBytesRead = 0;
    m_pAudioBuffer->Clear();
    int64 wallclockTimestamp = m_rtpServer.GetWallclockTimestamp();
    m_wallclockOfAudioStart = wallclockTimestamp + m_config.GetBufferMilliseconds() - m_config.GetNetworkBufferMilliseconds();
    m_audioPacketSSRC = pHeader->synchornisationSourceId;

    if ( g_pGame->GetCVars()->tgallery_debugAudioPackets == 1 )
    {
      CryLog( "Audio source changed. Starting audio at %d, current time %d", m_wallclockOfAudioStart, wallclockTimestamp );
    }
  }

  int byteOffsetOfPacket = pHeader->timestamp * AUDIO_BYTES_PER_SAMPLE;
  int byteOffsetOfPacketInAudioQueue = byteOffsetOfPacket - m_audioBytesRead;
  if ( g_pGame->GetCVars()->tgallery_debugAudioPackets == 1 )
  {
    CryLog( "Audio: absolute offset: %d offset in queue: %d", byteOffsetOfPacket, byteOffsetOfPacketInAudioQueue );
  }
  m_pAudioBuffer->WriteData( pPayload->data, payloadBytes, byteOffsetOfPacketInAudioQueue );
}

//////////////////////////////////////////////////////////////////////////
void CTalkieActor::OnRTPEmotion( SFixedRTPHeader* pHeader, byte* pData, int payloadBytes )
{
  if ( m_pFacialEffectorsLibrary == NULL )
  {
    return;
  }

  char emotionNameBuffer[ EMOTION_NAME_BUFFER_SIZE ];
  float intensity = 1;

  int elementsRead = sscanf_s( ( char* )( pData ), "%s %f", emotionNameBuffer, sizeof( emotionNameBuffer ), &intensity );
  if ( elementsRead == 2 )
  {
    QueueEmotion(m_rtpServer.GetWallclockTimestamp(),
      ConvertToWallclockTimestamp(pHeader, EMOTION_SAMPLING_RATE_HZ),
      m_pFacialEffectorsLibrary->Find( emotionNameBuffer ),
      intensity);
  }
}

//////////////////////////////////////////////////////////////////////////
void CTalkieActor::OnRTPAnimation( SFixedRTPHeader* pHeader, byte* pData, int payloadBytes )
{
  char animationNameBuffer[ ANIMATION_NAME_BUFFER_SIZE ];

  int elementsRead = sscanf_s( ( char* )( pData ), "%s", animationNameBuffer, sizeof( animationNameBuffer ) );
  if ( elementsRead == 1 )
  {
    QueueAnimation(m_rtpServer.GetWallclockTimestamp(),
      ConvertToWallclockTimestamp(pHeader, ANIMATION_SAMPLING_RATE_HZ),
      animationNameBuffer );
  }
}

//////////////////////////////////////////////////////////////////////////
void CTalkieActor::OnRTPLookAt( SFixedRTPHeader* pHeader, byte* pData, int payloadBytes )
{
  // Play the latest received look at packet right away ( ignore timestamp info )
  float lookAtX = Talkie::ExtractFloat(pData + 1);
  float lookAtY = Talkie::ExtractFloat(pData + 7); 

  bool enableLookAt = ( pData[ 0 ] != '1' );
  SetLookAtParameters( enableLookAt, lookAtX, lookAtY );
}

// from INetworkSoundListener ////////////////////////////////////////////
bool CTalkieActor::FillDataBuffer( unsigned int bitsPerSample, unsigned int samplesPerSecond, unsigned int numSamples, void* pData )
{
  CryAutoCriticalSection lock( m_audioLock );

  int bytesToRead = numSamples * bitsPerSample / 8;

  int64 millisecondsToAudioStart = m_wallclockOfAudioStart - m_rtpServer.GetWallclockTimestamp();
  int bytesToAudioStart = ConvertToAudioBytes( millisecondsToAudioStart );
  bool inBufferingTime = ( 0 < bytesToAudioStart );
  if ( inBufferingTime )
  {
    bool completelyInBufferingTime = ( bytesToRead < bytesToAudioStart );
    if ( completelyInBufferingTime )
    {
      if ( g_pGame->GetCVars()->tgallery_debugAudioPackets == 1 )
      {
        CryLog( "silent %d", bytesToRead );
      }
      memset( pData, 0, bytesToRead );
    }
    else
    {
      if ( g_pGame->GetCVars()->tgallery_debugAudioPackets == 1 )
      {
        CryLog( "Playing %d silent, %d buffered", bytesToAudioStart, bytesToRead - bytesToAudioStart );
      }
      memset( pData, 0, bytesToAudioStart );
      m_pAudioBuffer->ReadData( ( byte* )( pData ) + bytesToAudioStart, bytesToRead - bytesToAudioStart );
      m_pAudioBuffer->Advance( bytesToRead - bytesToAudioStart );
      m_audioBytesRead += ( bytesToRead - bytesToAudioStart );
    }
  }
  else
  {
    int bytesInBuffer = m_pAudioBuffer->GetBytesInBuffer();
    bool readingMoreBytesThanAvailable = ( bytesInBuffer < bytesToRead );
    if ( readingMoreBytesThanAvailable )
    {
      m_pAudioBuffer->ReadData( ( byte* )( pData ), bytesInBuffer );
      memset( ( byte* )( pData ) + bytesInBuffer, 0, bytesToRead - bytesInBuffer );
    }
    else
    {
      if ( g_pGame->GetCVars()->tgallery_debugAudioPackets == 1 )
      {
        CryLog( "Playing %d buffered", bytesToRead );
      }
      m_pAudioBuffer->ReadData( ( byte* )( pData ), bytesToRead );
    }

    m_pAudioBuffer->Advance( bytesToRead ); //tmp here
    m_audioBytesRead += bytesToRead; // tmp here

  }

  return true;
}

// from IEntityEventListener /////////////////////////////////////////////
void CTalkieActor::OnEntityEvent( IEntity* pEntity, SEntityEvent& event )
{
  if ( event.event == ENTITY_EVENT_DONE )
  {
    assert( pEntity != NULL );
    GetISystem()->GetIEntitySystem()->RemoveEntityEventListener( pEntity->GetId(), ENTITY_EVENT_DONE, this );

    m_pTalkieEntity = NULL;
    m_pCharacterInstance = NULL;
    m_pAnimationSet = NULL;
    m_pSkeletonAnim = NULL;
    m_pSkeletonPose = NULL;
    m_pFacialInstance = NULL;
    m_pFacialEffectorsLibrary = NULL;
    m_effectors.clear();
  }
}

//////////////////////////////////////////////////////////////////////////
void CTalkieActor::ClearVisimQueue()
{
  while ( !m_visimPlayQueue.empty() )
  {
    m_visimPlayQueue.pop();
  }
}

//////////////////////////////////////////////////////////////////////////
void CTalkieActor::QueueVisim( int64 wallclockTimestamp, int64 visimWallclockTimestamp, int visimId )
{
  bool isVeryOldVisim = ( visimWallclockTimestamp + m_config.GetVisimExpirationMilliseconds() < wallclockTimestamp );
  if ( isVeryOldVisim )
  {
    return;
  }

  VisimInfo vi;
  vi.wallclockTimestamp = visimWallclockTimestamp;
  vi.visimId = visimId;
  m_visimPlayQueue.push( vi );
}

//////////////////////////////////////////////////////////////////////////
void CTalkieActor::RemoveVisimEffectors()
{
  m_pFacialInstance->PreviewEffector( NULL, 0, 0 );
}

//////////////////////////////////////////////////////////////////////////
void CTalkieActor::ClearEmotionQueue()
{
  while ( ! m_emotionPlayQueue.empty() )
  {
    m_emotionPlayQueue.pop();
  }
}

//////////////////////////////////////////////////////////////////////////
void CTalkieActor::QueueEmotion( int64 wallclockTimestamp, int64 emotionWallclockTimestamp, IFacialEffector* emotionEffector, float intensity )
{
  bool isVeryOldEmotion = ( emotionWallclockTimestamp + EMOTION_EXPIRATION_MILLISECONDS < wallclockTimestamp );
  if ( isVeryOldEmotion )
  {
    return;
  }

  EmotionInfo ei;
  ei.wallclockTimestamp = emotionWallclockTimestamp;
  ei.emotionEffector = emotionEffector;
  ei.emotionIntensity = intensity;

  m_emotionPlayQueue.push( ei );
}

//////////////////////////////////////////////////////////////////////////
void CTalkieActor::RemoveEmotionEffectors()
{
  if ( 0 <= m_emotionEffectorChannel )
  {
    m_pFacialInstance->StopEffectorChannel( m_emotionEffectorChannel, 0.2f );
  }
  m_emotionEffectorChannel = -1;
}

//////////////////////////////////////////////////////////////////////////
void CTalkieActor::ClearAnimationQueue()
{
  while ( ! m_emotionPlayQueue.empty() )
  {
    m_emotionPlayQueue.pop();
  }
}

//////////////////////////////////////////////////////////////////////////
void CTalkieActor::QueueAnimation( int64 wallclockTimestamp, int64 animationWallclockTimestamp, const char* animationName )
{
  bool isVeryOldEmotion = ( animationWallclockTimestamp + ANIMATION_EXPIRATION_MILLISECONDS < wallclockTimestamp );
  if ( isVeryOldEmotion )
  {
    return;
  }

  AnimationInfo ai;
  ai.wallclockTimestamp = animationWallclockTimestamp;
  sprintf_s( ai.animationName, "%s", animationName );
  ai.animationIntensity = 1;

  m_animationPlayQueue.push( ai );
}

//////////////////////////////////////////////////////////////////////////
void CTalkieActor::SetDefaultAnimation()
{
  SetAnimation( m_config.GetInitialAnimationName() );
}

//////////////////////////////////////////////////////////////////////////
void CTalkieActor::SetAnimation( const char* animationName )
{
  if ( m_pSkeletonAnim == NULL )
  {
    return;
  }

  if ( animationName == NULL )
  {
    assert( false);
    return;
  }

  CryCharAnimationParams animationParams;
  animationParams.m_nFlags |= CA_LOOP_ANIMATION;
  animationParams.m_fTransTime = 0.2f;
  m_pSkeletonAnim->StartAnimation( animationName, animationParams );
}

//////////////////////////////////////////////////////////////////////////
void CTalkieActor::SetLookAtParameters( bool faceDetected, float screenX, float screenY )
{
  m_faceDetected = faceDetected;
  m_lookAtScreenX = screenX;
  m_lookAtScreenY = 1 - screenY;
}

//////////////////////////////////////////////////////////////////////////
bool CTalkieActor::IsAbleToProcessLookAt()
{
  return ( m_pSkeletonPose != NULL && g_pGame->GetCVars()->tgallery_enableLookAt != 0 );
}

//////////////////////////////////////////////////////////////////////////
void CTalkieActor::ProcessLookAtWithFace()
{
  const CCamera& camera = gEnv->pRenderer->GetCamera();
  Vec3 cameraPosition = camera.GetPosition();
  Vec3 entityPosition = m_pTalkieEntity->GetPos();

  const Vec3 up( 0, 0, 1 );
  Vec3 cameraToEntity = ( entityPosition - cameraPosition );
  cameraToEntity.z = 0;
  cameraToEntity.NormalizeSafe( Vec3( 1, 0, 0 ) );

  Vec3 right = cameraToEntity.Cross( up );
  float faceX = ( m_lookAtScreenX * 2 ) - 1;
  float faceY = ( m_lookAtScreenY * 2 ) - 1;

  Vec3 lookAtPosition = cameraPosition
                      + right * faceX * g_pGame->GetCVars()->tgallery_virtualScreenWidth
                      + up    * faceY * g_pGame->GetCVars()->tgallery_virtualScreenHeight;

  IScriptSystem* pScriptSystem = GetISystem()->GetIScriptSystem();
  if ( pScriptSystem->BeginCall( "TGalleryLookAt" ) )
  {
    pScriptSystem->PushFuncParam( cameraPosition );
    pScriptSystem->PushFuncParam( entityPosition );
    pScriptSystem->PushFuncParam( m_lookAtScreenX );
    pScriptSystem->PushFuncParam( m_lookAtScreenY );
    pScriptSystem->PushFuncParam( lookAtPosition );

    Talkie::UpdateLookAtPositionFromScriptSystem(&lookAtPosition);
  }

  m_pSkeletonPose->SetLookIK( true, DEG2RAD( LOOK_AT_RANGE_DEGREES ), lookAtPosition );
}

//////////////////////////////////////////////////////////////////////////
void CTalkieActor::ProcessLookAtWithOutFace()
{
  const CCamera& camera = gEnv->pRenderer->GetCamera();
  Vec3 cameraPosition = camera.GetPosition();
  Vec3 entityPosition = m_pTalkieEntity->GetPos();
  Vec3 entityForwardDirection = m_pTalkieEntity->GetForwardDir();
  Vec3 lookAtPosition = entityPosition + entityForwardDirection;
  lookAtPosition.z = cameraPosition.z;

  IScriptSystem* pScriptSystem = GetISystem()->GetIScriptSystem();
  if ( pScriptSystem->BeginCall( "TGalleryLookAt_NoFaceDetected" ) )
  {
    pScriptSystem->PushFuncParam( cameraPosition );
    pScriptSystem->PushFuncParam( entityPosition );
    pScriptSystem->PushFuncParam( entityForwardDirection );
    pScriptSystem->PushFuncParam( lookAtPosition );

    Talkie::UpdateLookAtPositionFromScriptSystem(&lookAtPosition);
  }

  m_pSkeletonPose->SetLookIK( true, DEG2RAD( LOOK_AT_RANGE_DEGREES ), lookAtPosition );
}

//////////////////////////////////////////////////////////////////////////
void CTalkieActor::UpdateLookAt()
{

  if ( !IsAbleToProcessLookAt() )
    return;

  assert( m_pTalkieEntity != NULL );

  if ( m_faceDetected )
  {
    ProcessLookAtWithFace();
  }
  else
  {
    ProcessLookAtWithOutFace();
  }
}

//////////////////////////////////////////////////////////////////////////
int64 CTalkieActor::ConvertToWallclockTimestamp(SFixedRTPHeader* pHeader, int64 frequency)
{
  return m_rtpServer.ConvertToWallclockTimestamp( pHeader, frequency ) + m_config.GetBufferMilliseconds();
}

//////////////////////////////////////////////////////////////////////////
void CTalkieActor::UpdateVisims( int64 wallclockTimestamp )
{
  if ( m_pFacialInstance == NULL )
  {
    return;
  }

  if ( m_visimPlayQueue.empty() )
  {
    return;
  }

  int visimId = -1;
  bool lookAtNextVisim = true;
  while ( lookAtNextVisim )
  {
    const VisimInfo& vi = m_visimPlayQueue.top();
    if ( vi.wallclockTimestamp < wallclockTimestamp )
    {
      visimId = vi.visimId;
      m_visimPlayQueue.pop();
      if ( m_visimPlayQueue.empty() )
      {
        lookAtNextVisim = false;
      }
    }
    else
    {
      lookAtNextVisim = false;
    }
  }

  if ( visimId != -1 )
  {
    if ( 0 <= visimId && visimId < m_effectors.size() )
    {
      m_pFacialInstance->PreviewEffector( m_effectors[ visimId ], 1, 0 );
    }
    else
    {
      m_pFacialInstance->PreviewEffector( NULL, 1, 0 );
    }
  }
}

//////////////////////////////////////////////////////////////////////////
void CTalkieActor::UpdateEmotions( int64 wallclockTimestamp )
{
  if ( m_pFacialInstance == NULL )
  {
    return;
  }

  if ( m_emotionPlayQueue.empty() )
  {
    return;
  }

  IFacialEffector* emotionEffector = NULL;
  float intensity = 0;
  bool emotionFound = false;
  bool lookAtNextEmotion = true;
  while ( lookAtNextEmotion )
  {
    const EmotionInfo& ei = m_emotionPlayQueue.top();
    if ( ei.wallclockTimestamp < wallclockTimestamp )
    {
      emotionFound = true;
      emotionEffector = ei.emotionEffector;
      intensity = ei.emotionIntensity;

      m_emotionPlayQueue.pop();
      if ( m_emotionPlayQueue.empty() )
      {
        lookAtNextEmotion = false;
      }
    }
    else
    {
      lookAtNextEmotion = false;
    }
  }

  if ( emotionFound )
  {
    RemoveEmotionEffectors();
    m_emotionEffectorChannel = ( int32 )( m_pFacialInstance->StartEffectorChannel( emotionEffector, intensity, 0.2f ) );
  }
}

//////////////////////////////////////////////////////////////////////////
void CTalkieActor::UpdateAnimations( int64 wallclockTimestamp )
{
  if ( m_pSkeletonAnim == NULL )
  {
    return;
  }

  if ( m_animationPlayQueue.empty() )
  {
    return;
  }

  string animationName;
  float intensity = 0;
  bool animationFound = false;
  bool lookAtNextAnimation = true;
  while ( lookAtNextAnimation )
  {
    const AnimationInfo& ai = m_animationPlayQueue.top();
    if ( ai.wallclockTimestamp < wallclockTimestamp )
    {
      animationFound = true;
      animationName = ai.animationName;
      intensity = ai.animationIntensity;

      m_animationPlayQueue.pop();
      if ( m_animationPlayQueue.empty() )
      {
        lookAtNextAnimation = false;
      }
    }
    else
    {
      lookAtNextAnimation = false;
    }
  }

  if ( animationFound )
  {
    SetAnimation( animationName.c_str() );
  }
}

