using System;
using System.Collections.Generic;
using System.Text;
using System.Text.RegularExpressions;

namespace AnimationTesting
{
	class CommandLineArgumentsException : Exception
	{
		public CommandLineArgumentsException(string message)
			: base(message)
		{

		}
	}

	class CommandLineArguments
	{
		class Option
		{
			private Regex m_regex;
			private string m_defaultValue;
			private string m_value;

			public Option(string name, string defaultValue)
			{
				m_regex = new Regex(string.Format("/{0}:(.*)", name));
				m_defaultValue = defaultValue;
				m_value = null;
			}

			public string Value
			{
				get { return m_value == null ? m_defaultValue : m_value; }
			}

			public bool Matches(string arg)
			{
				return m_regex.IsMatch(arg);
			}

			public void Update(string arg)
			{
				Match match = m_regex.Match(arg);
				if (match.Success)
				{
					m_value = match.Groups[1].Value;
				}
			}
		}

		private IDictionary<string, Option> m_options;

		public CommandLineArguments()
		{
			m_options = new Dictionary<string, Option>();
			
			AddOption("Report");
			AddOption("Suite", @"..\..\Code\Tools\AnimationTesting\CryAnimationTesting\suite.xml");
			AddOption("Binaries", @"Bin32");
		}

		private void AddOption(string name, string defaultvalue)
		{
			m_options.Add(name, new Option(name, defaultvalue));
		}

		private void AddOption(string name)
		{
			AddOption(name, null);
		}

		Option FindOption(string arg)
		{
			foreach (Option option in m_options.Values)
			{
				if (option.Matches(arg))
					return option;
			}
			return null;
		}

		public void ParseArguments(string[] args)
		{
			foreach (String arg in args)
			{
				Option option = FindOption(arg);
				if (option != null)
				{
					option.Update(arg);
				}
				else
				{
					throw new CommandLineArgumentsException("Argument not valid: " + arg);
				}
			}
		}

		public string Suite
		{
			get { return m_options["Suite"].Value; }
		}

		public string Binaries
		{
			get { return m_options["Binaries"].Value; }
		}

		public string Report
		{
			get { return m_options["Report"].Value; }
		}
	}
}
