using System;
using System.Collections.Generic;
using System.Text;
using CryAnimationTesting.Wrapper;

namespace AnimationTesting
{
	class ConsoleOutputWriter : OutputWriter
	{
		class FailureInfo
		{
			private TestContext m_testContext;
			private TestRunInfo m_testRunInfo;

			public FailureInfo(TestContext testContext, TestRunInfo testRunInfo)
			{
				m_testContext = testContext;
				m_testRunInfo = testRunInfo;
			}

			public TestContext TestContext
			{
				get { return m_testContext; }
			}

			public TestRunInfo TestRunInfo
			{
				get { return m_testRunInfo; }
			}
		}

		class FatalFailureInfo
		{
			private string m_failureMessage;

			public FatalFailureInfo(string failureMessage)
			{
				m_failureMessage = failureMessage;
			}

			public String FailureMessage
			{
				get { return m_failureMessage; }
			}
		}

		private ICollection<FailureInfo> m_failures;
		private ICollection<FatalFailureInfo> m_fatalFailures;
		private int m_testsPassed;

		public ConsoleOutputWriter()
		{
			m_failures = new List<FailureInfo>();
			m_fatalFailures = new List<FatalFailureInfo>();
			m_testsPassed = 0;
		}

		public void TestPassed(TestContext testContext, TestRunInfo testRunInfo)
		{
			m_testsPassed++;
			Console.Write(".");
		}

		public void TestFailed(TestContext testContext, TestRunInfo testRunInfo)
		{
			Console.Write("F");
			m_failures.Add(new FailureInfo(testContext, testRunInfo));
		}

		public void TestClassNotFound(Test test)
		{
			m_fatalFailures.Add(new FatalFailureInfo("Test class not found: " + test.Class));
		}

		public void FatalError(Test test, CharacterException characterException)
		{
			m_fatalFailures.Add(new FatalFailureInfo(characterException.Message + ": " + characterException.CharacterName));
		}

		public void FatalError(Test test, CharacterAssertException characterAssertException)
		{
			m_fatalFailures.Add(new FatalFailureInfo(characterAssertException.Message + " because of " + characterAssertException.Condition + ": " + characterAssertException.CharacterName));
		}

		public void Flush()
		{
			Console.WriteLine("\nTestPassed [{0}] TestFailed [{1}] FatalError [{2}]", m_testsPassed, m_failures.Count, m_fatalFailures.Count);

			if (m_fatalFailures.Count > 0 || m_failures.Count > 0)
			{
				Console.WriteLine("\n====================================================");
				foreach (FatalFailureInfo fatal in m_fatalFailures)
				{
					Console.WriteLine("FatalFailure {0}", fatal.FailureMessage);
				}

				foreach (FailureInfo failure in m_failures)
				{
                    Console.WriteLine("Failure Character [{0}] Animation [{1}] ErrorCondition [{2}] ErrorMessage [{3}] FileName [{4}] FileLineNumber [{5}]", failure.TestContext.CharacterName, failure.TestContext.AnimationName, failure.TestRunInfo.ErrorCondition, failure.TestRunInfo.ErrorMessage, failure.TestRunInfo.FileName, failure.TestRunInfo.FileLineNumber);
				}
				Console.WriteLine("====================================================");
			}
		}
	}
}
