using System;
using System.Collections.Generic;
using System.Text;
using System.Xml;
using CryAnimationTesting.Wrapper;

namespace AnimationTesting
{
	class XmlOutputWriter : OutputWriter
	{

		#region AbstractTestInfo

		abstract class AbstractTestInfo
		{
			public abstract void Write(XmlTextWriter writer);

			public virtual bool IsFailure()
			{
				return false;
			}

			public virtual bool IsError()
			{
				return false;
			}

			public void WriteTestCaseResult(XmlTextWriter writer)
			{
				writer.WriteStartElement("testcase");
				Write(writer);
				writer.WriteEndElement();
			}
		}

		class TestInfo : AbstractTestInfo
		{
			protected TestContext m_testContext;
			protected TestRunInfo m_testRunInfo;

			public TestInfo(TestContext testContext, TestRunInfo testRunInfo)
			{
				m_testContext = testContext;
				m_testRunInfo = testRunInfo;
			}

			public override void Write(XmlTextWriter writer)
			{
				writer.WriteAttributeString("name", m_testContext.TestName);
				writer.WriteAttributeString("class", m_testRunInfo.TestName);
				writer.WriteAttributeString("character", m_testContext.CharacterName);
				writer.WriteAttributeString("animation", m_testContext.AnimationName);

				WriteFailure(writer);
			}

			public virtual void WriteFailure(XmlTextWriter writer) 
			{
			
			}
		}

		class TestFailureInfo : TestInfo
		{
			public TestFailureInfo(TestContext testContext, TestRunInfo testRunInfo)
				: base(testContext, testRunInfo)
			{

			}

			public override void WriteFailure(XmlTextWriter writer)
			{
				writer.WriteStartElement("failure");
				
				writer.WriteAttributeString("type", "Assert");
				writer.WriteAttributeString("message", m_testRunInfo.ErrorMessage);

                writer.WriteString(string.Format("filename [{0}] line [{1}]", m_testRunInfo.FileName, m_testRunInfo.FileLineNumber.ToString()));
				
				writer.WriteEndElement();
			}

			public override bool IsFailure()
			{
				return true;
			}
		}

		abstract class TestErrorInfo : AbstractTestInfo
		{
			protected Test m_test;

			public TestErrorInfo(Test test)
			{
				m_test = test;
			}

			public override void Write(XmlTextWriter writer)
			{
				writer.WriteAttributeString("name", m_test.Name);
				writer.WriteAttributeString("class", m_test.Class);

				writer.WriteStartElement("failure");

				WriteFailureDetails(writer);

				writer.WriteEndElement();
			}

			public abstract void WriteFailureDetails(XmlTextWriter writer);

			public override bool IsError()
			{
				return true;
			}
		}

		class TestClassNotFoundInfo : TestErrorInfo
		{
			public TestClassNotFoundInfo(Test test)
				: base(test)
			{
			}

			public override void WriteFailureDetails(XmlTextWriter writer)
			{
				writer.WriteAttributeString("type", "ClassNotFound");
				writer.WriteAttributeString("message", m_test.Class);
                writer.WriteString(string.Format("TestCase class {0} not found in C++ code. Check CRY_TEST macros.", m_test.Class));
			}
		}

		class CharacterExceptionInfo : TestErrorInfo
		{
			private CharacterException m_characterException;

			public CharacterExceptionInfo(Test test, CharacterException characterException)
				: base(test)
			{
				m_characterException = characterException;
			}

			public override void WriteFailureDetails(XmlTextWriter writer)
			{
				writer.WriteAttributeString("type", "CharacterException");
				writer.WriteAttributeString("message", m_characterException.Message);
                writer.WriteString(string.Format("CharacterName [{0}]", m_characterException.CharacterName));
			}
		}

		class CharacterAssertExceptionInfo : TestErrorInfo
		{
			private CharacterAssertException m_characterAssertException;

			public CharacterAssertExceptionInfo(Test test, CharacterAssertException characterAssertException)
				: base(test)
			{
				m_characterAssertException = characterAssertException;
			}

			public override void WriteFailureDetails(XmlTextWriter writer)
			{
				writer.WriteAttributeString("type", "CharacterAssertException");
				writer.WriteAttributeString("message", m_characterAssertException.Message);
                writer.WriteString(string.Format("Condition [{0}] Filename [{1}] Line [{2}]", m_characterAssertException.Condition, m_characterAssertException.FileName, m_characterAssertException.FileLineNumber.ToString()));
			}
		}

		#endregion

		private XmlTextWriter m_writer;
		private ICollection<AbstractTestInfo> m_testInfos;
		private DateTime m_startTime;

		public XmlOutputWriter(string xmlFileName)
		{
			m_writer = new XmlTextWriter(xmlFileName, Encoding.Default);
			m_testInfos = new List<AbstractTestInfo>();
			m_startTime = DateTime.Now;
		}

		public void TestPassed(TestContext testContext, TestRunInfo testRunInfo)
		{
			m_testInfos.Add(new TestInfo(testContext, testRunInfo));
		}

		public void TestFailed(TestContext testContext, TestRunInfo testRunInfo)
		{
			m_testInfos.Add(new TestFailureInfo(testContext, testRunInfo));
		}

		public void TestClassNotFound(Test test)
		{
			m_testInfos.Add(new TestClassNotFoundInfo(test));
		}

		public void FatalError(Test test, CharacterException characterException)
		{
			m_testInfos.Add(new CharacterExceptionInfo(test, characterException));
		}

		public void FatalError(Test test, CharacterAssertException characterException)
		{
			m_testInfos.Add(new CharacterAssertExceptionInfo(test, characterException));
		}

		private delegate bool FilterDelegate(AbstractTestInfo testInfo);

		private int GetTestNumber(FilterDelegate filter)
		{
			int res = 0;
			foreach (AbstractTestInfo testInfo in m_testInfos)
			{
				if (filter(testInfo))
					res++;
			}
			return res;
		}

		private int GetFailingTestNumber()
		{
			return GetTestNumber(delegate(AbstractTestInfo testInfo)
			{
				return testInfo.IsFailure();
			});
		}

		private int GetErrorTestNumber()
		{
			return GetTestNumber(delegate(AbstractTestInfo testInfo)
			{
				return testInfo.IsError();
			});
		}

		private int GetTestNumber()
		{
			return m_testInfos.Count;
		}

		public void Flush()
		{
			TimeSpan time = DateTime.Now.Subtract(m_startTime);

			m_writer.WriteStartDocument();
			m_writer.WriteStartElement("testsuite");

			m_writer.WriteAttributeString("name", "AnimationTesting");
			m_writer.WriteAttributeString("tests", GetTestNumber().ToString());
			m_writer.WriteAttributeString("failures", GetFailingTestNumber().ToString());
			m_writer.WriteAttributeString("errors", GetErrorTestNumber().ToString());
			m_writer.WriteAttributeString("time", time.TotalSeconds.ToString());

			foreach (AbstractTestInfo testInfo in m_testInfos)
			{
				testInfo.WriteTestCaseResult(m_writer);
			}

			m_writer.WriteEndElement();

			m_writer.WriteEndDocument();
			m_writer.Flush();
			m_writer.Close();
		}
	}
}
