using System;
using System.Xml;
using System.Collections.Generic;

namespace AnimationTesting
{
	public class SuiteParser
	{
		private XmlElement m_root;
		private IDictionary<string, Test> m_tests;

		public SuiteParser(XmlElement root, IDictionary<string, Test> tests)
		{
			m_root = root;
			m_tests = tests;
		}

		public void Parse()
		{
			Dictionary<string, List<string>> characters = new Dictionary<string, List<string>>();
			Dictionary<string, List<string>> animations = new Dictionary<string, List<string>>();

			foreach (XmlNode node in m_root.ChildNodes)
			{
				if (node.Name.Equals("Character"))
				{
					ParseCharacterNode(node, characters);
				}
				else if (node.Name.Equals("Animation"))
				{
					ParseAnimationNode(node, animations);
				}
			}

			foreach (XmlNode node in m_root.ChildNodes)
			{
				if (node.Name.Equals("Test"))
				{
					ParseTestNode(node, characters, animations);
				}
			}
		}

		private void ParseTestNode(XmlNode testNode, Dictionary<string, List<string>> characters, Dictionary<string, List<string>> animations)
		{
			XmlAttribute name = testNode.Attributes["name"];
			XmlAttribute clazz = testNode.Attributes["class"];
			Test test = new Test(name.Value, clazz.Value);

			foreach (XmlNode node in testNode)
			{
				if (node.Name.Equals("Character"))
				{
					XmlAttribute characterName = node.Attributes["name"];
					AddCharacters(test, characterName.Value, characters);
				}
				else if (node.Name.Equals("Animation"))
				{
					XmlAttribute animationName = node.Attributes["name"];
					AddAnimations(test, animationName.Value, animations);
				}
				else if (node.Name.Equals("ConfigInfo"))
				{
					ParseCustomNode(test, node.ChildNodes);
				}
			}

			m_tests.Add(name.Value, test);
		}

		private void ParseCharacterNode(XmlNode characterNode, Dictionary<string, List<string>> characters)
		{
			XmlAttribute name = characterNode.Attributes["name"];
			if (!characters.ContainsKey(name.Value))
			{
				characters.Add(name.Value, new List<string>());
			}

			List<string> files = characters[name.Value];

			foreach (XmlNode node in characterNode)
			{
				if (node.Name.Equals("File"))
				{
					XmlAttribute fileName = node.Attributes["name"];
					files.Add(fileName.Value);
				}
			}
		}

		private void AddCharacters(Test test, string characterName, Dictionary<string, List<string>> characters)
		{
			if (characters.ContainsKey(characterName))
			{
				foreach (string file in characters[characterName])
				{
					test.Characters.Add(file);
				}
			}
			else
			{
				throw new Exception("Character not found: " + characterName);
			}
		}

		private void ParseAnimationNode(XmlNode animationNode, Dictionary<string, List<string>> animations)
		{
			XmlAttribute name = animationNode.Attributes["name"];
			if (!animations.ContainsKey(name.Value))
			{
				animations.Add(name.Value, new List<string>());
			}

			List<string> files = animations[name.Value];

			foreach (XmlNode node in animationNode)
			{
				if (node.Name.Equals("Pattern"))
				{
					XmlAttribute pattern = node.Attributes["value"];
					files.Add(pattern.Value);
				}
			}
		}

		private void AddAnimations(Test test, string animationName, Dictionary<string, List<string>> animations)
		{
			if (animations.ContainsKey(animationName))
			{
				foreach (string pattern in animations[animationName])
				{
					test.Animations.Add(pattern);
				}
			}
			else
			{
				throw new Exception("Animation not found: " + animationName);
			}
		}

		private void ParseCustomNode(Test test, XmlNodeList nodeList)
		{
			foreach (XmlNode node in nodeList)
			{
				ConfigElement configElement = new ConfigElement(node.Name);
				foreach (XmlAttribute attribute in node.Attributes)
				{
					configElement.Add(attribute.Name, attribute.Value);
				}
				test.ConfigElementCollection.Add(configElement);
			}
		}
	}
}
