using NUnit.Framework;
using System.Collections.Generic;
using System.IO;
using System;

namespace UnitTest
{
	[TestFixture]
	public class SuiteTest
	{

		private AnimationTesting.Suite CreateSuite(string xml)
		{
			AnimationTesting.Suite suite = new AnimationTesting.Suite();
			suite.Load(new StringReader(xml));
			return suite;
		}

		private AnimationTesting.Test GetTest(string xml, string testName)
		{
			AnimationTesting.Suite suite = CreateSuite(xml);
			return suite[testName];
		}

		[Test]
		public void ParseOneTest()
		{
			string xml = @"	<Suite>
								<Test name='TestName' class='TestClass' />
							</Suite>";

			AnimationTesting.Suite suite = CreateSuite(xml);

			IEnumerator<AnimationTesting.Test> enumerator = suite.GetEnumerator();
			Assert.IsTrue(enumerator.MoveNext());
			Assert.AreEqual("TestName", enumerator.Current.Name);
			Assert.AreEqual("TestClass", enumerator.Current.Class);
		}

		[Test]
		public void ParseMoreThanOneTest()
		{
			string xml = @"	<Suite>
								<Test name='TestName1' class='TestClass1' />
								<Test name='TestName2' class='TestClass2' />
							</Suite>";

			AnimationTesting.Suite suite = CreateSuite(xml);

			IEnumerator<AnimationTesting.Test> enumerator = suite.GetEnumerator();
			Assert.IsTrue(enumerator.MoveNext());
			Assert.IsTrue(enumerator.MoveNext());
			Assert.IsFalse(enumerator.MoveNext());
		}

		[Test]
		public void ParseNoMappedCharactersInTest()
		{
			string xml = @"	<Suite>
								<Test name='TestName' class='TestClass'>
									<Character name='Objects/default.chr' />
									<Character name='Objects/aliens/stalker.chr' />
								</Test>
							</Suite>";

			try
			{
				AnimationTesting.Suite suite = CreateSuite(xml);
				Assert.Fail("Should throw");
			}
			catch (Exception) {}
		}

		[Test]
		public void ParseMappedCharactersInTest()
		{
			string xml = @"	<Suite>
								<Test name='TestName' class='TestClass'>
									<Character name='alien' />
								</Test>
								<Character name='alien'>
									<File name='Objects/default.chr' />
									<File name='Objects/aliens/stalker.chr' />
								</Character>
							</Suite>";

			AnimationTesting.Suite suite = CreateSuite(xml);

			AnimationTesting.Test test = suite["TestName"];
			Assert.IsNotNull(test);
			Assert.IsTrue(test.Characters.Contains("Objects/default.chr"));
			Assert.IsTrue(test.Characters.Contains("Objects/aliens/stalker.chr"));
		}

		[Test]
		public void ParseNoMappedAnimationsInTest()
		{
			string xml = @"	<Suite>
								<Test name='TestName' class='TestClass'>
									<Animation name='.*walk_x.*' />
									<Animation name='.*walk_y.*' />
								</Test>
							</Suite>";

			try
			{
				AnimationTesting.Suite suite = CreateSuite(xml);
				Assert.Fail("Should throw");
			}
			catch (Exception) {}
		}

		[Test]
		public void ParseMappedAnimationsInTest()
		{
			string xml = @"	<Suite>
								<Test name='TestName' class='TestClass'>
									<Animation name='walk' />
								</Test>
								<Animation name='walk'>
									<Pattern value='.*walk_x.*' />
									<Pattern value='.*walk_y.*' />
								</Animation>
							</Suite>";

			AnimationTesting.Suite suite = CreateSuite(xml);

			AnimationTesting.Test test = suite["TestName"];
			Assert.IsNotNull(test);
			Assert.IsTrue(test.Animations.Contains(".*walk_x.*"));
			Assert.IsTrue(test.Animations.Contains(".*walk_y.*"));
		}

        [Test]
        public void ParseMultiMappedAnimationsInTest()
        {
            string xml = @"	<Suite>
								<Test name='TestName' class='TestClass'>
									<Animation name='first-person' />
                                    <Animation name='slide' />
								</Test>
								<Animation name='first-person'>
									<Pattern value='.*fp.*' />
								</Animation>
								<Animation name='slide'>
									<Pattern value='.*slide.*' />
								</Animation>
							</Suite>";

            AnimationTesting.Suite suite = CreateSuite(xml);
            AnimationTesting.Test test = suite["TestName"];
            Assert.IsTrue(test.Animations.Contains(".*fp.*"));
            Assert.IsTrue(test.Animations.Contains(".*slide.*"));
        }

        [Test]
        public void ParseMultiMixedAnimationsInTest()
        {
            string xml = @"	<Suite>
								<Test name='TestName' class='TestClass'>
									<Animation name='first-person' />
                                    <Animation name='slide' />
								</Test>
								<Animation name='first-person'>
									<Pattern value='.*fp.*' />
								</Animation>
								<Animation name='slide'>
									<Pattern value='.*slide_1.*' />
                                    <Pattern value='.*slide_2.*' />
								</Animation>
							</Suite>";

            AnimationTesting.Suite suite = CreateSuite(xml);
            AnimationTesting.Test test = suite["TestName"];
            Assert.IsTrue(test.Animations.Contains(".*fp.*"));
            Assert.IsTrue(test.Animations.Contains(".*slide_1.*"));
            Assert.IsTrue(test.Animations.Contains(".*slide_2.*"));
        }

		[Test]
		public void ParseConfigInfo()
		{
			string xml = @"	<Suite>
								<Test name='TestName' class='TestClass'>
									<ConfigInfo><Node/></ConfigInfo>
								</Test>
							</Suite>";

			AnimationTesting.Test test = GetTest(xml, "TestName");

			IEnumerator<AnimationTesting.ConfigElement> configElementEnumerator = test.ConfigElementCollection.GetEnumerator();
			Assert.IsTrue(configElementEnumerator.MoveNext());
			Assert.AreEqual("Node", configElementEnumerator.Current.Name);
			Assert.IsFalse(configElementEnumerator.MoveNext());
		}

		[Test]
		public void ParseConfigInfoMultiNode()
		{
			string xml = @"	<Suite>
								<Test name='TestName' class='TestClass'>
									<ConfigInfo>
										<Node1/>
										<Node2/>
									</ConfigInfo>
								</Test>
							</Suite>";

			AnimationTesting.Test test = GetTest(xml, "TestName");

			IEnumerator<AnimationTesting.ConfigElement> configElementEnumerator = test.ConfigElementCollection.GetEnumerator();
			configElementEnumerator.MoveNext();
			Assert.AreEqual("Node1", configElementEnumerator.Current.Name);
			configElementEnumerator.MoveNext();
			Assert.AreEqual("Node2", configElementEnumerator.Current.Name);
			Assert.IsFalse(configElementEnumerator.MoveNext());
		}

		[Test]
		public void ParseConfigInfoAttribute()
		{
			string xml = @"	<Suite>
								<Test name='TestName' class='TestClass'>
									<ConfigInfo><Node Key='Value'/></ConfigInfo>
								</Test>
							</Suite>";

			AnimationTesting.Test test = GetTest(xml, "TestName");

			IEnumerator<AnimationTesting.ConfigElement> configElementEnumerator = test.ConfigElementCollection.GetEnumerator();
			configElementEnumerator.MoveNext();
			Assert.AreEqual("Value", configElementEnumerator.Current["Key"]);		
		}

		[Test]
		public void ParseConfigInfoNotFoundAttribute()
		{
			string xml = @"	<Suite>
								<Test name='TestName' class='TestClass'>
									<ConfigInfo><Node Key='Value'/></ConfigInfo>
								</Test>
							</Suite>";

			AnimationTesting.Test test = GetTest(xml, "TestName");

			IEnumerator<AnimationTesting.ConfigElement> configElementEnumerator = test.ConfigElementCollection.GetEnumerator();
			configElementEnumerator.MoveNext();
			Assert.IsNull(configElementEnumerator.Current["KeyNotFoud"]);
		}

		[Test]
		public void ParseConfigInfoMultiAttribute()
		{
			string xml = @"	<Suite>
								<Test name='TestName' class='TestClass'>
									<ConfigInfo><Node Key1='Value1' Key2='Value2'/></ConfigInfo>
								</Test>
							</Suite>";

			AnimationTesting.Test test = GetTest(xml, "TestName");

			IEnumerator<AnimationTesting.ConfigElement> configElementEnumerator = test.ConfigElementCollection.GetEnumerator();
			configElementEnumerator.MoveNext();
			Assert.AreEqual("Value1", configElementEnumerator.Current["Key1"]);
			Assert.AreEqual("Value2", configElementEnumerator.Current["Key2"]);
		}
	}
}
