#include "receivefile.h"
#include "config.h"
#include "hashcache.h"
#include "client.h"
#include <sstream>

static char NULL_BYTES[1024];

CReceiveFile::CReceiveFile( const std::string& filename, unsigned packets, const CHash& hash )
: m_filename(filename), m_file(new CFile(filename, false)), m_hash(hash), m_nPackets(packets), m_bValidated(false)
{
	m_queue.Insert( CDataRange(0, packets) );
}


bool CReceiveFile::Complete( IHashProgressCallback * pProgress, IClientListener * pListener )
{
	bool ok = false;
	try
	{
		if (!m_file.get())
			throw std::runtime_error("Attempt to complete a receive twice");
		m_file.reset();
		if (!m_queue.Empty())
			throw std::runtime_error("Attempt to complete with a non-empty packet-remaining queue");
		CHash fileHash = GetHash( m_filename, pProgress );
		if (fileHash != m_hash)
			throw std::runtime_error("Hash mismatch");
		ok = true;
	}
	catch (std::exception& e)
	{
		std::cout << "Exception completing file " << m_filename << ": " << e.what() << std::endl;
		if (pListener)
			pListener->Notification( "Error Completing Job", m_filename + "\n" + e.what() );
	}
	catch (...)
	{
		std::cout << "Exception completing file " << m_filename << std::endl;
		if (pListener)
			pListener->Notification( "Error Completing Job", m_filename );
	}
	if (!ok)
	{
		std::cout << "Removing file " << m_filename << std::endl;
		_unlink( m_filename.c_str() );
	}
	else if (pListener)
	{
		pListener->Notification( "Completed Job", m_filename );
	}
	return ok;
}

void CReceiveFile::Timeout( IClientListener * pListener )
{
	if (pListener)
		pListener->Notification( "Timeout on Job", m_filename );
	_unlink( m_filename.c_str() );
}

bool CReceiveFile::Receive( unsigned pkt, const void * data, size_t sz )
{
	try
	{
		if (sz != DATA_SIZE && pkt != (m_nPackets-1))
			throw std::runtime_error("Incorrect data size");

		if (m_queue.Remove(pkt))
		{
			m_file->Seek( 0, SEEK_END );
			fpos_t pos = fpos_t(pkt) * DATA_SIZE;
			while (m_file->GetPos() < pos)
			{
				size_t wLen = size_t( min( fpos_t(sizeof(NULL_BYTES)), pos-m_file->GetPos() ) );
				m_file->Write( NULL_BYTES, wLen );
			}
			if (m_file->GetPos() > pos)
				m_file->SetPos(pos);
			m_file->Write( data, sz );
		}
		return m_queue.Empty();
	}
	catch (...)
	{
		m_queue.Insert( CDataRange(pkt) );
		throw;
	}
}

std::string CReceiveFile::GetProgressMessage()
{
	std::ostringstream out;
	out << m_filename << ": ";
	unsigned nPacketsLeft = m_queue.Count();
	float fPercentComplete = 100.0f * (m_nPackets - nPacketsLeft) / m_nPackets;
	out << int(fPercentComplete) << "%";
	return out.str();
}
