// BuildManagerDlg.cpp : implementation file
//

#include "stdafx.h"
#include "BuildManager.h"
#include "BuildManagerDlg.h"
#include "FileUtil.h"
#include "TaskManager.h"

#include <list>

#ifdef _DEBUG
#define new DEBUG_NEW
#endif



// ConnectionsRefreshTask

class ConnectionsRefreshTask : public TaskManager::ITaskObject
{
	public:
		static ITaskObject * CreateTask(CBuildManagerDlg * buildDialog);
		
		virtual void Process();
		virtual void PostProcess();

	private:
		ConnectionsRefreshTask(CBuildManagerDlg * buildDialog);
		CBuildManagerDlg * m_buildDialog;
};


TaskManager::ITaskObject * ConnectionsRefreshTask::CreateTask(CBuildManagerDlg * buildDialog)
{
	return new ConnectionsRefreshTask(buildDialog);
}

ConnectionsRefreshTask::ConnectionsRefreshTask(CBuildManagerDlg * buildDialog)
{
	m_buildDialog = buildDialog;
}

void ConnectionsRefreshTask::Process()
{
}

void ConnectionsRefreshTask::PostProcess()
{
	if ( m_buildDialog )
	{
		m_buildDialog->ConnectionsRefreshCallback();	
	}
}

///////////////////////////////////////////////////////////////


// CBuildManagerDlg dialog
UINT CBuildManagerDlg::s_checkBuildUploadFinishedTimer = 0x01;

//

void __stdcall CBuildManagerDlg::CheckBuildUploadFinished(HWND, UINT, UINT_PTR, DWORD)
{
	bool bWasSuccessful = false;
	if (CProjectManager::Instance().IsUploadingBuild(&bWasSuccessful))
		return;

	::KillTimer(::AfxGetMainWnd()->GetSafeHwnd(), s_checkBuildUploadFinishedTimer);

	CBuildManagerDlg *pBM = (CBuildManagerDlg *)::AfxGetMainWnd();

	pBM->ConnectionRefreshCurrent();

	pBM->m_upload.SetWindowText("Copy Files");
	pBM->EnableControlsAuto();

	pBM->LogOutput(bWasSuccessful ?
		"$3Copy successfully finished." :
		"$4Copy terminated.");

	if (pBM->m_bAutoUpload)
	{
		if (!bWasSuccessful)
			::AfxMessageBox("Copy files aborted.");

		exit(0);
	}
}

/*

  CBuildManagerDlg

*/

CBuildManagerDlg::CBuildManagerDlg(CWnd *pParent) :
	CDialog(CBuildManagerDlg::IDD, pParent)
{
	m_hIcon = AfxGetApp()->LoadIcon(IDR_MAINFRAME);

	m_pConnection = NULL;

	m_controlDescs.reserve(16);

	m_bAutoUpload = false;
	m_autoUploadExclude = NULL;

	CommandListClear();
	m_timer = ::SetTimer(NULL,IDT_TASKMANAGER_TIMER,200,NULL);
}

CBuildManagerDlg::~CBuildManagerDlg()
{
	ConfigWrite();

	RemoteControl::CConnectionManager::Instance().Shutdown();

	::DeleteCriticalSection(&m_logCriticalSection);

	::KillTimer(NULL,m_timer);
}

void CBuildManagerDlg::DoDataExchange(CDataExchange* pDX)
{
	CDialog::DoDataExchange(pDX);

	DDX_Control(pDX, IDC_PROJECTS, m_projects);
	DDX_Control(pDX, IDC_PROJECTSREFRESH, m_projectsRefresh);
	DDX_Control(pDX, IDC_CONNECTION, m_connections);
	DDX_Control(pDX, IDC_CONNECTIONSREFRESH, m_connectionsRefresh);
	DDX_Control(pDX, IDC_BUILDS, m_builds);
	DDX_Control(pDX, IDC_BUILDSREFRESH, m_buildsRefresh);
	DDX_Control(pDX, IDC_BUILDVERSION, m_connectionBuilds);
	DDX_Control(pDX, IDC_UPLOADPATH, m_connectionPath);
	DDX_Control(pDX, IDC_LEVELS, m_connectionLevels);

	DDX_Control(pDX, IDC_UPLOAD, m_upload);
	DDX_Control(pDX, IDC_LAUNCH, m_launch);

	DDX_Control(pDX, IDC_LOG, m_log);
	DDX_Control(pDX, IDC_COMMAND, m_command);

	DDX_Control(pDX, IDC_NO_SYSTEM_CFG,m_noSystemCfg);
	DDX_Control(pDX, IDC_FRONTEND,m_frontend);
	DDX_Control(pDX, IDC_NOBINARIES, m_noBinaries);
	DDX_Control(pDX, IDC_NOLEVELS, m_noLevels);
	DDX_Control(pDX, IDC_CLEAN_BUILD, m_cleanBuild);
	DDX_Control(pDX, IDC_FILTER_BUILDS, m_filterBuilds);
}

BOOL CBuildManagerDlg::PreTranslateMessage(MSG *pMsg)
{
	if (pMsg->message == WM_KEYDOWN && GetFocus() == &m_command)
	{
		if (pMsg->wParam == VK_ESCAPE)
		{
			m_command.SetWindowText("");
			return TRUE;
		}
		if (pMsg->wParam == VK_UP)
		{
			return TRUE;
		}
		if (pMsg->wParam == VK_TAB)
		{
			CommandComplete();
			return true;
		}
		if (pMsg->wParam == VK_RETURN)
		{
			CString s;
			m_command.GetWindowText(s);
			m_command.SetWindowText("");

			if (!s.GetLength())
				return FALSE;

			char response[MAX_PATH];
			size_t responseLength = MAX_PATH;
			if (m_pConnection)
//				m_pConnection->SendCommand(s, response, responseLength);
				m_pConnection->SendCommand(s);
		}
	}

	if (pMsg->message == WM_KEYDOWN)
	{
		m_commandListFind = "";
		m_commandListLast = m_commandList;
	}

	if ( pMsg->message == WM_TIMER )
	{
		TaskManager::CTaskManager::Instance().Tick();
	}

	return CDialog::PreTranslateMessage(pMsg);
}

BEGIN_MESSAGE_MAP(CBuildManagerDlg, CDialog)
	ON_WM_SIZING()
	ON_WM_SIZE()
	ON_WM_ERASEBKGND()
	ON_WM_PAINT()
	ON_WM_QUERYDRAGICON()
	//}}AFX_MSG_MAP
	ON_CBN_SELCHANGE(IDC_PROJECTS, &CBuildManagerDlg::OnCbnSelchangeProjects)
	ON_BN_CLICKED(IDC_PROJECTSREFRESH, &CBuildManagerDlg::OnBnClickedProjectsRefresh)
	ON_CBN_SELCHANGE(IDC_CONNECTION, &CBuildManagerDlg::OnCbnSelchangeConnections)	
	ON_BN_CLICKED(IDC_CONNECTIONSREFRESH, &CBuildManagerDlg::OnBnClickedConnectionsRefresh)
	ON_EN_CHANGE(IDC_UPLOADPATH, &CBuildManagerDlg::OnEnChangeConnectionPath)
	ON_CBN_SELCHANGE(IDC_LEVELS, &CBuildManagerDlg::OnCbnSelchangeConnectionLevels)
	ON_BN_CLICKED(IDC_UPLOAD, &CBuildManagerDlg::OnBnClickedUpload)
	ON_BN_CLICKED(IDC_LAUNCH, &CBuildManagerDlg::OnBnClickedLaunch)
	ON_BN_CLICKED(IDC_FRONTEND, &CBuildManagerDlg::OnBnClickedFlag)
	ON_BN_CLICKED(IDC_NOBINARIES, &CBuildManagerDlg::OnBnClickedFlag)
	ON_BN_CLICKED(IDC_NOLEVELS, &CBuildManagerDlg::OnBnClickedFlag)
	ON_BN_CLICKED(IDC_CLEAN_BUILD, &CBuildManagerDlg::OnBnClickedFlag)
	ON_EN_CHANGE(IDC_LOG, &CBuildManagerDlg::OnEnChangeLog)
	ON_BN_CLICKED(IDC_BUILDSREFRESH, &CBuildManagerDlg::OnBnClickedBuildsRefresh)
	ON_BN_CLICKED(IDC_FILTER_BUILDS, &CBuildManagerDlg::OnBnClickedBuildsRefresh)
END_MESSAGE_MAP()

//

void CBuildManagerDlg::SetAutoUpload(const char *project, const char *target, const char *path, bool bNoBinaries, bool bNoLevels,bool bNoSystemCfg)
{
	m_bAutoUpload = true;
	m_autoUploadProject = project;
	m_autoUploadTarget = target;
	m_autoUploadPath = path;
	m_autoUploadExclude = NULL;
	if (bNoBinaries)
		m_autoUploadExclude |= CProjectManager::eExclude_Binaries;
	if (bNoLevels)
		m_autoUploadExclude |= CProjectManager::eExclude_Levels;
	if (bNoSystemCfg)
		m_autoUploadExclude |= CProjectManager::eExclude_SystemCFG;
}

void CBuildManagerDlg::LogOutput(const char *text)
{
	if (!text)
		return;

	::EnterCriticalSection(&m_logCriticalSection);

	COLORREF color = RGB(196, 196, 196);

	CHARFORMAT2 charFormat2;
	m_log.GetDefaultCharFormat(charFormat2);
	charFormat2.dwMask = CFM_COLOR;
	charFormat2.dwEffects = NULL;
	charFormat2.crTextColor = color;
	charFormat2.crBackColor = NULL;
	m_log.SetSelectionCharFormat(charFormat2);

	char out[5120];

	while (const char *c = ::strchr(text, '$'))
	{
		::strncpy_s(out, text, c - text);
		out[c - text] = '\0';
		m_log.SetSel(INT_MAX, INT_MAX);
		m_log.ReplaceSel(out);
		text = c + 1;
		
		if (*text)
		{
			char colorChar[2] = { *text, '\0' };
			int code = ::atoi(colorChar);
			text++;

			switch (code)
			{
			case 2: color = RGB(128, 128, 255); break;
			case 3: color = RGB(128, 255, 128); break;
			case 4: color = RGB(255, 128, 128); break;
			case 5: color = RGB(128, 255, 255); break;
			case 6: color = RGB(255, 255, 128); break;
			case 7: color = RGB(255, 128, 255); break;
			case 8: color = RGB(255, 128, 128); break;
			case 9: color = RGB(  0, 128, 255); break;
			}

			CHARFORMAT2 charFormat2;
			m_log.GetDefaultCharFormat(charFormat2);
			charFormat2.dwMask = CFM_COLOR;
			charFormat2.dwEffects = NULL;
			charFormat2.crTextColor = color;
			charFormat2.crBackColor = NULL;
			m_log.SetSelectionCharFormat(charFormat2);
		}
	}

	m_log.SetSel(INT_MAX, INT_MAX);
	m_log.ReplaceSel(text);
	if (text[::strlen(text)-1] != '\n')
		m_log.ReplaceSel("\n");

	// Scroll down.

	long cStart, cEnd;
	m_log.GetSel(cStart, cEnd);
	CPoint p = m_log.PosFromChar(cEnd);
	CRect r;
	m_log.GetClientRect(&r);
	for (int i=0; r.bottom < p.y+10 && i<1024; i++)
	{
		m_log.LineScroll(1);
		p = m_log.PosFromChar(cEnd);
	}

	::LeaveCriticalSection(&m_logCriticalSection);
}

inline bool ResolveApplicationSettingPath(string &path)
{
	LPITEMIDLIST pItemIDList = NULL;
	HRESULT hResult = ::SHGetSpecialFolderLocation(NULL, CSIDL_LOCAL_APPDATA, &pItemIDList);
	char appPath[MAX_PATH] = { '\0' };
	::SHGetPathFromIDList(pItemIDList, appPath);
	LPMALLOC pMalloc;
	hResult = ::SHGetMalloc(&pMalloc);
	pMalloc->Free(pItemIDList);
	pMalloc->Release();

	path = appPath;
	::append_if_not_last(path, '\\', "/\\");
	return true;
}

void CBuildManagerDlg::ConfigWrite()
{
	string s;
	if (!::ResolveApplicationSettingPath(s))
		return;
	s += "BuildManager.ini";
	const char *path = s.c_str();

	if (!m_mruProject.empty())
		::WritePrivateProfileString("MRU", "project", m_mruProject.c_str(), path);
	if (!m_mruConnection.empty())
		::WritePrivateProfileString("MRU", "connection", m_mruConnection.c_str(), path);
	if (!m_mruBuild.empty())
		::WritePrivateProfileString("MRU", "build", m_mruBuild.c_str(), path);
	if (!m_mruLevel.empty())
		::WritePrivateProfileString("MRU", "level", m_mruLevel.c_str(), path);
}

void CBuildManagerDlg::ConfigRead()
{
	string s;
	if (!::ResolveApplicationSettingPath(s))
		return;
	s += "BuildManager.ini";
	const char *path = s.c_str();

	char buffer[4096];
	if (::GetPrivateProfileString("MRU", "project", NULL, buffer, sizeof(buffer)-1, path))
		m_mruProject = buffer;
	if (::GetPrivateProfileString("MRU", "connection", NULL, buffer, sizeof(buffer)-1, path))
		m_mruConnection = buffer;
	if (::GetPrivateProfileString("MRU", "build", NULL, buffer, sizeof(buffer)-1, path))
		m_mruBuild = buffer;
	if (::GetPrivateProfileString("MRU", "level", NULL, buffer, sizeof(buffer)-1, path))
		m_mruLevel = buffer;

	if (!m_mruConnection.empty())
		m_connectionPrevious = m_mruConnection;
}

void CBuildManagerDlg::ProjectConfigWrite()
{
	if (!m_pConnection)
		return;

	size_t project = CProjectManager::Instance().GetActiveProject();
	if (CProjectManager::Instance().GetProjectCount() <= project)
		return;
	const CProjectManager::CDesc &desc =
		CProjectManager::Instance().GetProjectDesc(project);

	string s;
	if (!::ResolveApplicationSettingPath(s))
		return;
	s += "BuildManager.ini";
	const char *path = s.c_str();

	string section = desc.name;
	section += m_pConnection->GetType();

	const char *remotePath = CProjectManager::Instance().GetRemotePath();
	if (remotePath)
	{
		if (!*remotePath)
			remotePath = NULL;
	}

	::WritePrivateProfileString(section.c_str(), "remotePath",	remotePath ? remotePath : NULL, path);
	::WritePrivateProfileString(section.c_str(), "NoBinaries",	m_noBinaries.GetCheck() == BST_CHECKED ? "TRUE" : "FALSE", path);
	::WritePrivateProfileString(section.c_str(), "NoLevels",	m_noLevels.GetCheck() == BST_CHECKED ? "TRUE" : "FALSE", path);
	::WritePrivateProfileString(section.c_str(), "NoSystemCFG",	m_noSystemCfg.GetCheck() == BST_CHECKED ? "TRUE" : "FALSE", path);
	::WritePrivateProfileString(section.c_str(), "Frontend",	m_frontend.GetCheck() == BST_CHECKED ? "TRUE" : "FALSE", path);
	::WritePrivateProfileString(section.c_str(), "CleanBuild",	m_cleanBuild.GetCheck() == BST_CHECKED ? "TRUE" : "FALSE", path);
}

void CBuildManagerDlg::ProjectConfigRead()
{
	if (!m_pConnection)
		return;

	size_t project = CProjectManager::Instance().GetActiveProject();
	if (CProjectManager::Instance().GetProjectCount() <= project)
		return;
	const CProjectManager::CDesc &desc =
		CProjectManager::Instance().GetProjectDesc(project);

	string s;
	if (!::ResolveApplicationSettingPath(s))
		return;
	s += "BuildManager.ini";
	const char *path = s.c_str();

	string section = desc.name;
	section += m_pConnection->GetType();

	char buffer[4096];
	if (::GetPrivateProfileString(section.c_str(), "remotePath", NULL, buffer, sizeof(buffer)-1, path))
		CProjectManager::Instance().SetRemotePath(buffer);

	bool bNoSystemCfg = false;
	if (::GetPrivateProfileString(section.c_str(), "NoSystemCFG", NULL, buffer, sizeof(buffer)-1, path))
		bNoSystemCfg = _strnicmp(buffer, "TRUE", 4) == 0 ? true : false;
	
	bool bNoBinaries = false;
	if (::GetPrivateProfileString(section.c_str(), "NoBinaries", NULL, buffer, sizeof(buffer)-1, path))
		bNoBinaries = _strnicmp(buffer, "TRUE", 4) == 0 ? true : false;

	bool bNoLevels = false;
	if (::GetPrivateProfileString(section.c_str(), "NoLevels", NULL, buffer, sizeof(buffer)-1, path))
		bNoLevels = _strnicmp(buffer, "TRUE", 4) == 0 ? true : false;

	bool bFrontend = false;
	if (::GetPrivateProfileString(section.c_str(), "Frontend", NULL, buffer, sizeof(buffer)-1, path))
		bFrontend = _strnicmp(buffer, "TRUE", 4) == 0 ? true : false;

	bool bCleanBuild = false;
	if (::GetPrivateProfileString(section.c_str(), "CleanBuild", NULL, buffer, sizeof(buffer)-1, path))
		bCleanBuild = _strnicmp(buffer, "TRUE", 4) == 0 ? true : false;

	m_noSystemCfg.SetCheck(bNoSystemCfg ? BST_CHECKED : BST_UNCHECKED);
	m_noBinaries.SetCheck(bNoBinaries ? BST_CHECKED : BST_UNCHECKED);
	m_noLevels.SetCheck(bNoLevels ? BST_CHECKED : BST_UNCHECKED);
	m_frontend.SetCheck(bFrontend ? BST_CHECKED : BST_UNCHECKED);
	m_cleanBuild.SetCheck(bCleanBuild ? BST_CHECKED : BST_UNCHECKED);
}

void CBuildManagerDlg::EnableControlsUpload()
{
	m_projects.EnableWindow(FALSE);
	m_projectsRefresh.EnableWindow(FALSE);
	m_connections.EnableWindow(FALSE);
	m_connectionsRefresh.EnableWindow(FALSE);

	m_builds.EnableWindow(FALSE);
	m_buildsRefresh.EnableWindow(FALSE);
	m_connectionBuilds.EnableWindow(FALSE);
	m_connectionPath.EnableWindow(FALSE);
	m_connectionLevels.EnableWindow(FALSE);
	m_upload.EnableWindow(TRUE);
	m_upload.SetWindowText("Cancel");
	m_launch.EnableWindow(FALSE);
	m_command.EnableWindow(FALSE);	

	m_noSystemCfg.EnableWindow(FALSE);
	m_noBinaries.EnableWindow(FALSE);
	m_noLevels.EnableWindow(FALSE);
	m_frontend.EnableWindow(FALSE);
	m_cleanBuild.EnableWindow(FALSE);
	m_filterBuilds.EnableWindow(FALSE);
}

void CBuildManagerDlg::EnableControlsAuto()
{
	BOOL bConnections = m_connections.GetCount() ? TRUE : FALSE;
	BOOL bLevels = m_connectionLevels.GetCount() ? TRUE : FALSE;
	BOOL bBuilds = m_builds.GetCount() ? TRUE : FALSE;
	BOOL bConnected = FALSE;

	BOOL bCommand = FALSE;
	if (m_pConnection)
	{
		if (::strcmp(m_pConnection->GetType(), "XB360") == 0 ||
			::strcmp(m_pConnection->GetType(), "PS3") == 0)
			bCommand = TRUE;

		bConnected = m_pConnection->IsTargetConnected();
	}

	m_projects.EnableWindow(TRUE);
	m_projectsRefresh.EnableWindow(TRUE);
	m_connections.EnableWindow(TRUE);
	m_connectionsRefresh.EnableWindow(TRUE);

	m_builds.EnableWindow(bConnections && bBuilds);
	m_buildsRefresh.EnableWindow(bConnections);
	m_connectionBuilds.EnableWindow(bLevels);
	m_connectionPath.EnableWindow(bConnections && bBuilds);
	m_connectionLevels.EnableWindow(bConnections && bLevels && bConnected);
	m_upload.EnableWindow(bConnections && bBuilds);
	m_launch.EnableWindow(bConnections && bLevels && bConnected);
	m_log.EnableWindow(bConnections);
	m_command.EnableWindow(bConnections && bCommand && bConnected);

	m_noSystemCfg.EnableWindow(bConnections && bBuilds);
	m_noBinaries.EnableWindow(bConnections && bBuilds);
	m_noLevels.EnableWindow(bConnections && bBuilds);
	m_frontend.EnableWindow(bConnections && bBuilds && bConnected);
	m_cleanBuild.EnableWindow(bConnections && bBuilds);
	m_filterBuilds.EnableWindow(bConnections && bBuilds);
}

void CBuildManagerDlg::ProjectsRefresh()
{
	m_projects.ResetContent();

	CProjectManager::Instance().ClearProjects();

	// Retrieve the path of the current process and used it as root to search
	// for Project files.
	char path[MAX_PATH];
	::GetModuleFileName(NULL, path, sizeof(path)-1);
	path[Path::GetLengthWithoutFile(path)] = '\0';
	::strcat(path, "Projects/");
	CProjectManager::Instance().AddProjectFiles(path);
	
	int current = 0;
	for (size_t i=0; i<CProjectManager::Instance().GetProjectCount(); i++)
	{
		const char *name = CProjectManager::Instance().GetProjectDesc(i).name.c_str();
		int r = m_projects.AddString(name);
		if (r < 0)
			continue;

		m_projects.SetItemData(r, i);

		if (::stricmp(name, m_mruProject.c_str()) == 0)
			current = r;
	}

	if ( m_projects.GetCount() > 0 )
	{
		m_projects.SetCurSel(current);
		ProjectActive(m_projects.GetItemData(m_projects.GetCurSel()));
	}
}

void CBuildManagerDlg::ProjectActive(size_t index)
{
	m_builds.ResetContent();
	m_connectionLevels.ResetContent();

	if (!CProjectManager::Instance().GetProjectCount())
		return;

	CProjectManager::Instance().ActiveProject(index);
	m_mruProject = CProjectManager::Instance().GetProjectDesc(index).name;

	m_filterBuilds.SetCheck( CProjectManager::Instance().GetProjectDesc(index).filterIncompleteBuilds ? BST_CHECKED : BST_UNCHECKED );

	ConnectionsRefresh();
}

void CBuildManagerDlg::ConnectionsRefresh()
{
	TaskManager::CTaskManager::Instance().AddTask(ConnectionsRefreshTask::CreateTask(this));
}

void CBuildManagerDlg::ConnectionsRefreshCallback()
{
	if (m_pConnection)
	{
		m_mruConnection = m_pConnection->GetName();
		m_pConnection = NULL;
	}

	m_connections.ResetContent();

	if (!CProjectManager::Instance().GetProjectCount())
		return;

	const CProjectManager::CDesc &desc = CProjectManager::Instance().GetProjectDesc(
		CProjectManager::Instance().GetActiveProject());

	string s;
	int current = 0;
	for (size_t i=0; i<RemoteControl::CConnectionManager::Instance().GetConnectionCount(); i++)
	{
		const char *name = RemoteControl::CConnectionManager::Instance().GetConnection(i).GetName();
		if (!name)
			continue;

		const char *type = RemoteControl::CConnectionManager::Instance().GetConnection(i).GetType();
		if (!type)
			continue;

		// If the Connection is not supported by the current Project, skip it.
		if (!desc.platforms.count(type))
			continue;

		s = name;
		if (::strcmp(type, "XB360") == 0)
			s += " (Xbox 360)";
		else if (::strcmp(type, "PS3") == 0)
			s += " (PlayStation 3)";

		int r = m_connections.AddString(s.c_str());
		if (r < 0)
			continue;

		m_connections.SetItemData(r, i);

		if (::stricmp(name, m_mruConnection.c_str()) == 0)
			current = r;
	}

	if ( m_connections.GetCount() > 0 )
	{
		m_connections.SetCurSel(current);
		ConnectionActive(m_connections.GetItemData(m_connections.GetCurSel()));
	}
}

void CBuildManagerDlg::ConnectionActive(size_t index)
{
	m_connectionLevels.ResetContent();

	RemoteControl::IConnection *pConnectionPrevious = m_pConnection;

	// If a Connection is currently open, close it.
	if (m_pConnection)
		m_pConnection->Close();
	m_pConnection = NULL;

	// Clear the command list.
	CommandListClear();

	// Try to open a Connection with the given index.
	m_pConnection = RemoteControl::CConnectionManager::Instance().OpenConnection(index);
	if (!m_pConnection)
	{
		m_pConnection = &RemoteControl::CConnectionManager::Instance().GetConnection(index);
		EnableControlsAuto();
		return;
	}

	// Active the opened Connection.
	CProjectManager::Instance().ActiveConnection(m_pConnection);

	BuildsRefresh();

	ConnectionRefreshCurrent();
}

void CBuildManagerDlg::ConnectionRefreshCurrent()
{
	m_connectionLevels.ResetContent();
	m_connectionBuilds.SetWindowText("");

	if (!m_pConnection)
		return;

	CProjectManager::CDesc::CPlatform *pPlatform =
		CProjectManager::Instance().GetActivePlatform();
	if (!pPlatform)
		return;

	CProjectManager::Instance().SetRemotePath(pPlatform->remotePath.c_str());
	ProjectConfigRead();

	// NOTE: An explicit call to ConnectionPathRefresh() is not done here since
	// setting the window text of m_uploadPath will call its change function
	// which in turn will call it.
	m_connectionPath.SetWindowText(CProjectManager::Instance().GetRemotePath());
}

void CBuildManagerDlg::ConnectionPathRefresh(const char *path)
{
	m_connectionLevels.ResetContent();
	m_connectionBuilds.SetWindowText("");

	CProjectManager::Instance().SetRemotePath(path);
	ProjectConfigWrite();

	// Set this Connection as MRU.
	m_mruConnection = m_pConnection->GetName();
	m_connectionPrevious = m_mruConnection;

	if (!CProjectManager::Instance().RefreshConnectionCurrent())
	{
		EnableControlsAuto();
		return;
	}

	// Add the list of available levels for the activated Connection.
	int current = 0;
	for (size_t i=0; i<CProjectManager::Instance().GetRemoteLevelCount(); i++)
	{
		const char *name = CProjectManager::Instance().GetRemoteLevelName(i);

		int r = m_connectionLevels.AddString(name);
		if (r < 0)
			continue;

		m_connectionLevels.SetItemData(r, i);

		if (::stricmp(name, m_mruLevel.c_str()) == 0)
			current = r;
	}
	m_connectionLevels.SetCurSel(current);

	// If a build is already present on the activated Connection, append its
	// name to the Builds label.
	if (const char *name = CProjectManager::Instance().GetBuildRemoteVersion())
		m_connectionBuilds.SetWindowText(name);

	EnableControlsAuto();
}

namespace BuildSorting
{
	struct builddetails
	{
		int index;
		string buildname;
	};

	int ExtractBuildNumber(string str)
	{
		int posleft = str.find_first_of(')',0);
		int posright = str.find_first_of('(',0)+1;
		string buildnumber = str.substr(0,posleft);
		buildnumber = buildnumber.substr(posright);
		int buildnumeric = -1;
		int ret = sscanf(buildnumber.c_str(),"%d",&buildnumeric);
		if ( ret > 0 && ret != EOF )
		{
			return buildnumeric;
		}
		return -1;
	}

	bool SortBuilds(const builddetails& d1, const builddetails& d2)
	{
		int buildnumber1 = ExtractBuildNumber(d1.buildname);
		if ( buildnumber1 == -1 )
			return false;

		int buildnumber2 = ExtractBuildNumber(d2.buildname);
		if ( buildnumber2 == -1 )
			return true;

		if ( buildnumber1 == buildnumber2 ) 
			return (d1.buildname.length() > d2.buildname.length());

		return (buildnumber1 > buildnumber2);
	}
}

void CBuildManagerDlg::BuildsRefresh()
{
	std::list<BuildSorting::builddetails> buildsordered;
	m_builds.ResetContent();

	if (!CProjectManager::Instance().RefreshBuilds())
		return;

	CProjectManager::Instance().SetFilterBuilds(m_filterBuilds.GetCheck()==BST_CHECKED?true:false);

	// Add the list of available builds for the activated Connection.
	for (size_t i=0; i<CProjectManager::Instance().GetBuildCount(); i++)
	{
		BuildSorting::builddetails newBuild; 
		newBuild.index = CProjectManager::Instance().GetBuildCount()-1-i;
		newBuild.buildname = CProjectManager::Instance().GetBuildName(newBuild.index);

		if ( !CProjectManager::Instance().FilterBuild(newBuild.buildname) )
		{
			buildsordered.push_back(newBuild);
		}
	}

	buildsordered.sort(BuildSorting::SortBuilds);

	for ( std::list<BuildSorting::builddetails>::iterator item = buildsordered.begin(), end = buildsordered.end(); item != end; ++item )
	{
		int r = m_builds.AddString((*item).buildname.c_str());
		if (r < 0)
			continue;

		m_builds.SetItemData(r, (*item).index);
	}

	buildsordered.clear();
	m_builds.SetCurSel(0);
}

void CBuildManagerDlg::SetFilterBuilds(bool filterbuilds)
{
	m_filterBuilds.SetCheck(filterbuilds ? BST_CHECKED : BST_UNCHECKED);
}

void CBuildManagerDlg::CommandListClear()
{
	*m_commandList = '\0';
	strcpy_s(m_commandList, "map;quit;");
	m_commandListInitialized = false;

	m_commandListLast = m_commandList;
}

bool CBuildManagerDlg::CommandListCreate()
{
	if(!m_pConnection)
		return false;

	CommandListClear();

	char response[2048];
	size_t responseLength = COMMAND_LIST_LENGTH;

	char remoteCommand[1024] = "__getvars__";

	while (m_pConnection->SendCommand(remoteCommand, response, responseLength))
	{
		size_t length = ::strlen(response);
		if (!length)
			break;

		size_t commandsLength = ::strlen(m_commandList);
		if (commandsLength + length >= COMMAND_LIST_LENGTH)
			break;

		::strcpy(&m_commandList[commandsLength], &response[5]); // 5 to skip error code.

		response[length-1] = NULL;
		char *c = ::strrchr(response, ';');
		if (!c)
			break;

		responseLength = COMMAND_LIST_LENGTH;
		::sprintf_s(remoteCommand, "__getvars__%s", c+1);

		m_commandListInitialized = true;
	}

	return m_commandListInitialized;
}

bool CBuildManagerDlg::CommandComplete()
{
	if (!m_commandListInitialized)
	{
		if (!CommandListCreate())
			return false;
	}

	CString s;
	m_command.GetWindowText(s);

	if(!s.GetLength())
		return false;

	if(!m_commandListFind.GetLength())
		m_commandListFind = s;

	char buffer[128]; // Needs to be big enough to store the max command length.
	char *c = buffer;

	for (int i=0; i<2; i++)
	{
		while (*m_commandListLast)
		{
			if (*m_commandListLast == ';')
			{
				*c++ = ' ';
				*c = '\0';
				if (::_strnicmp(buffer, m_commandListFind, m_commandListFind.GetLength()) == 0)
				{
					m_command.SetWindowText(buffer);
					m_command.SetSel((int)::strlen(buffer), -1, FALSE);
					++m_commandListLast;
					return true;
				}
				c = buffer;
			}
			else
			{
				*c++ = *m_commandListLast;
			}
			++m_commandListLast;
		}
		m_commandListLast = m_commandList;
	}

	return false;
}

void CBuildManagerDlg::AutoUpload()
{
	if (!m_bAutoUpload)
		return;

	size_t project;
	for (project=0; project<CProjectManager::Instance().GetProjectCount(); ++project)
	{
		const CProjectManager::CDesc &desc =
			CProjectManager::Instance().GetProjectDesc(project);
		if (::stricmp(desc.name.c_str(), m_autoUploadProject.c_str()))
			continue;
		break;
	}
	if (project == CProjectManager::Instance().GetProjectCount())
		return;

	RemoteControl::IConnection *pConnection = NULL;
	size_t connection;
	for (connection=0; connection<RemoteControl::CConnectionManager::Instance().GetConnectionCount(); ++connection)
	{
		if (::stricmp(RemoteControl::CConnectionManager::Instance().GetConnection(connection).GetName(), m_autoUploadTarget.c_str()))
			continue;
		break;
	}

	if (connection == RemoteControl::CConnectionManager::Instance().GetConnectionCount())
		return;

	if (!CProjectManager::Instance().ActiveProject(project))
		return;

	ProjectActive(project);
	ConnectionActive(connection);
	m_connectionPath.SetWindowText(m_autoUploadPath.c_str());
	m_noSystemCfg.SetCheck(m_autoUploadExclude & CProjectManager::eExclude_SystemCFG ? BST_CHECKED : BST_UNCHECKED);
	m_noBinaries.SetCheck(m_autoUploadExclude & CProjectManager::eExclude_Binaries ? BST_CHECKED : BST_UNCHECKED);
	m_noLevels.SetCheck(m_autoUploadExclude & CProjectManager::eExclude_Levels ? BST_CHECKED : BST_UNCHECKED);
	CProjectManager::Instance().UploadBuildBegin(CProjectManager::Instance().GetBuildCount()-1, m_autoUploadExclude);

	EnableControlsUpload();
	::SetTimer(::AfxGetMainWnd()->GetSafeHwnd(), s_checkBuildUploadFinishedTimer, USER_TIMER_MINIMUM, CheckBuildUploadFinished);
}

BOOL CBuildManagerDlg::OnInitDialog()
{
	// Initialize GUI

	CDialog::OnInitDialog();

	GetWindowRect(&m_rect);
	SetIcon(m_hIcon, TRUE);

	m_controlDescs.push_back(CControlDesc(m_projects, NULL));
	m_controlDescs.push_back(CControlDesc(m_projectsRefresh, NULL));
	m_controlDescs.push_back(CControlDesc(m_connections, NULL));
	m_controlDescs.push_back(CControlDesc(m_connectionsRefresh, NULL));
	m_controlDescs.push_back(CControlDesc(m_builds, NULL));
	m_controlDescs.push_back(CControlDesc(m_buildsRefresh, NULL));
	m_controlDescs.push_back(CControlDesc(m_connectionBuilds, NULL));
	m_controlDescs.push_back(CControlDesc(m_connectionPath,
		CControlDesc::eSF_Right));
	m_controlDescs.push_back(CControlDesc(m_connectionLevels,
		CControlDesc::eSF_Right));

	m_controlDescs.push_back(CControlDesc(m_upload,
		CControlDesc::eSF_Left | CControlDesc::eSF_Right));
	m_controlDescs.push_back(CControlDesc(m_launch,
		CControlDesc::eSF_Right | CControlDesc::eSF_Left));

	m_controlDescs.push_back(CControlDesc(m_log,
		CControlDesc::eSF_Right | CControlDesc::eSF_Bottom));
	m_controlDescs.push_back(CControlDesc(m_command,
		CControlDesc::eSF_Right | CControlDesc::eSF_Top | CControlDesc::eSF_Bottom));

	m_log.SetBackgroundColor(FALSE, ::GetSysColor(COLOR_3DFACE));
	m_log.SetBackgroundColor(FALSE, RGB(32, 32, 32));

	if (CWnd *pWnd = GetDlgItem(IDC_LOG))
	{
		CFont font;
		if (font.CreateFont(
			11, 8, 0, 0, FW_DONTCARE,
			FALSE, FALSE, FALSE,
			ANSI_CHARSET,
			OUT_DEFAULT_PRECIS, CLIP_DEFAULT_PRECIS,
			DEFAULT_QUALITY, DEFAULT_PITCH, "Lucida Console"))
		{
			pWnd->SetFont(&font);
		}
	}

	ShowWindow(SW_SHOW);

	//

	m_noSystemCfg.SetCheck(BST_UNCHECKED);
	m_noBinaries.SetCheck(BST_UNCHECKED);
	m_noLevels.SetCheck(BST_UNCHECKED);
	m_frontend.SetCheck(BST_UNCHECKED);
	m_cleanBuild.SetCheck(BST_UNCHECKED);

	// Initialize logic

	::InitializeCriticalSection(&m_logCriticalSection);
	RemoteControl::CConnectionManager::Instance().Initialize();
	RemoteControl::CConnectionManager::Instance().SetCallback(this);
	RemoteControl::CConnectionManager::Instance().RefreshConnections();

	ConfigRead();
	ProjectsRefresh();

	HICON hIcon = (HICON)::LoadImage(::AfxGetInstanceHandle(), MAKEINTRESOURCE(IDI_ICON1), IMAGE_ICON, 16, 16, 0);
	m_projectsRefresh.SetIcon(hIcon);
	m_connectionsRefresh.SetIcon(hIcon);
	m_buildsRefresh.SetIcon(hIcon);

	AutoUpload();

	return TRUE;  // return TRUE  unless	you set the focus to a control
}

void CBuildManagerDlg::OnSizing(UINT nSide, LPRECT lpRect)
{
	CRect &rect = *(CRect *)lpRect;
	switch (nSide)
	{
	case WMSZ_LEFT:
		if (rect.Width() < m_rect.Width())
			rect.left = rect.right - m_rect.Width();
		break;
	case WMSZ_RIGHT:
		if (rect.Width() < m_rect.Width())
			rect.right = rect.left + m_rect.Width();
		break;
	case WMSZ_TOP:
		if (rect.Height() <  m_rect.Height())
			rect.top = rect.bottom - m_rect.Height();
		break;
	case WMSZ_TOPLEFT:
		if (rect.Width() < m_rect.Width())
			rect.left = rect.right - m_rect.Width();
		if (rect.Height() <  m_rect.Height())
			rect.top = rect.bottom - m_rect.Height();
		break;
	case WMSZ_TOPRIGHT:
		if (rect.Width() < m_rect.Width())
			rect.right = rect.left + m_rect.Width();
		if (rect.Height() <  m_rect.Height())
			rect.top = rect.bottom - m_rect.Height();
		break;
	case WMSZ_BOTTOM:
		if (rect.Height() <  m_rect.Height())
			rect.bottom = rect.top + m_rect.Height();
		break;
	case WMSZ_BOTTOMLEFT:
		if (rect.Width() < m_rect.Width())
			rect.left = rect.right - m_rect.Width();
		if (rect.Height() <  m_rect.Height())
			rect.bottom = rect.top + m_rect.Height();
		break;
	case WMSZ_BOTTOMRIGHT:
		if (rect.Width() < m_rect.Width())
			rect.right = rect.left + m_rect.Width();
		if (rect.Height() <  m_rect.Height())
			rect.bottom = rect.top + m_rect.Height();
		break;
	}
}

void CBuildManagerDlg::OnSize(UINT nType, int cx, int cy)
{
	CRect r;
	GetClientRect(r);

	WINDOWPLACEMENT wp;
	for (size_t i=0; i<m_controlDescs.size(); i++)
	{
		if (!m_controlDescs[i].pWindow)
			continue;
		if (!::IsWindow(m_controlDescs[i].pWindow->m_hWnd))
			continue;

		if (!m_controlDescs[i].sizeFlags)
			continue;

		m_controlDescs[i].pWindow->GetWindowPlacement(&wp);

		if (m_controlDescs[i].sizeFlags & CControlDesc::eSF_Left)
			wp.rcNormalPosition.left = cx - m_controlDescs[i].sizeRect.left;
		if (m_controlDescs[i].sizeFlags & CControlDesc::eSF_Right)
			wp.rcNormalPosition.right = cx - m_controlDescs[i].sizeRect.right;
		if (m_controlDescs[i].sizeFlags & CControlDesc::eSF_Top)
			wp.rcNormalPosition.top = cy - m_controlDescs[i].sizeRect.top;
		if (m_controlDescs[i].sizeFlags & CControlDesc::eSF_Bottom)
			wp.rcNormalPosition.bottom = cy - m_controlDescs[i].sizeRect.bottom;

		m_controlDescs[i].pWindow->SetWindowPlacement(&wp);
	}

	CDialog::OnSize(nType, cx, cy);
}

BOOL CBuildManagerDlg::OnEraseBkgnd(CDC *pDC)
{
	// Background will manually be erased on paint to avoid flickering.
	return FALSE;
	return CDialog::OnEraseBkgnd(pDC);
}

// If you add a minimize button to your dialog, you will need the code below
//  to draw the icon.  For MFC applications using the document/view model,
//  this is automatically done for you by the framework.

void CBuildManagerDlg::OnPaint()
{
	if (IsIconic())
	{
		CPaintDC dc(this); // device context for painting

		SendMessage(WM_ICONERASEBKGND, reinterpret_cast<WPARAM>(dc.GetSafeHdc()), 0);

		// Center icon in client rectangle
		int cxIcon = GetSystemMetrics(SM_CXICON);
		int cyIcon = GetSystemMetrics(SM_CYICON);
		CRect rect;
		GetClientRect(&rect);
		int x = (rect.Width() - cxIcon + 1) / 2;
		int y = (rect.Height() - cyIcon + 1) / 2;

		// Draw the icon
		dc.DrawIcon(x, y, m_hIcon);
	}
	else
	{
		// Manually erase the background to avoid flickering.

		WINDOWPLACEMENT wp;
		for (size_t i=0; i<m_controlDescs.size(); i++)
		{
			if (!m_controlDescs[i].pWindow)
				continue;
			if (!::IsWindow(m_controlDescs[i].pWindow->m_hWnd))
				continue;

			m_controlDescs[i].pWindow->GetWindowPlacement(&wp);
			ValidateRect(&wp.rcNormalPosition);
		}

		CRect r;
		GetClientRect(r);

		CBrush brush;
		brush.CreateSolidBrush(::GetSysColor(COLOR_3DFACE));

		CPaintDC dc(this);
		dc.FillRect(r, &brush);

		InvalidateRect(r, 0);

		CDialog::OnPaint();
	}
}

// The system calls this function to obtain the cursor to display while the user drags
//  the minimized window.
HCURSOR CBuildManagerDlg::OnQueryDragIcon()
{
	return static_cast<HCURSOR>(m_hIcon);
}

//

void CBuildManagerDlg::OnCbnSelchangeProjects()
{
	ProjectActive(m_projects.GetItemData(m_projects.GetCurSel()));
}

void CBuildManagerDlg::OnBnClickedProjectsRefresh()
{
	ProjectsRefresh();
}

void CBuildManagerDlg::OnCbnSelchangeConnections()
{
	ConnectionActive(m_connections.GetItemData(m_connections.GetCurSel()));
}

void CBuildManagerDlg::OnBnClickedConnectionsRefresh()
{
	ConnectionsRefresh();
}

void CBuildManagerDlg::OnBnClickedBuildsRefresh()
{
	BuildsRefresh();
}

void CBuildManagerDlg::OnEnChangeConnectionPath()
{
	char path[MAX_PATH];
	if (!m_connectionPath.GetWindowText(path, sizeof(path)))
		return;

	ConnectionPathRefresh(path);
}

void CBuildManagerDlg::OnCbnSelchangeConnectionLevels()
{
	m_mruLevel = CProjectManager::Instance().GetRemoteLevelName(
		m_connectionLevels.GetItemData(m_connectionLevels.GetCurSel()));
}

void CBuildManagerDlg::OnBnClickedUpload()
{
	if (!m_pConnection)
		return;

	if (CProjectManager::Instance().IsUploadingBuild())
	{
		if (::AfxMessageBox("Cancel the copying process?", MB_YESNO) != IDYES)
			return;

		if (!CProjectManager::Instance().IsUploadingBuild())
			return;

		CProjectManager::Instance().UploadBuildAbort();
		m_upload.SetWindowText("Canceling..");
		m_upload.EnableWindow(FALSE);
	}
	else
	{
		char path[MAX_PATH];
		if (m_connectionPath.GetWindowText(path, sizeof(path)))
			CProjectManager::Instance().SetRemotePath(path);

		CProjectManager::Instance().UploadBuildBegin(
			m_builds.GetItemData(m_builds.GetCurSel()),
			(m_noSystemCfg.GetCheck() == BST_CHECKED ? CProjectManager::eExclude_SystemCFG : NULL)|
			(m_noBinaries.GetCheck() == BST_CHECKED ? CProjectManager::eExclude_Binaries : NULL) |
			(m_noLevels.GetCheck() == BST_CHECKED ? CProjectManager::eExclude_Levels : NULL) |
			(m_cleanBuild.GetCheck() == BST_CHECKED ? CProjectManager::eClean_Build : NULL)
			);

		EnableControlsUpload();
		::SetTimer(::AfxGetMainWnd()->GetSafeHwnd(), s_checkBuildUploadFinishedTimer, USER_TIMER_MINIMUM, CheckBuildUploadFinished);
	}
}

void CBuildManagerDlg::OnBnClickedLaunch()
{
	CommandListClear();

	if (!m_pConnection)
		return;

	LogOutput("Launching...");
	size_t level = ( m_frontend.GetCheck() == BST_UNCHECKED ) ? m_connectionLevels.GetItemData(m_connectionLevels.GetCurSel()) : NO_LEVEL;
	CProjectManager::Instance().Launch( level );
}

// IConnectionManagerCallback

void CBuildManagerDlg::OnConnectionManagerLog(const char *text)
{
	LogOutput(text);
}

void CBuildManagerDlg::OnBnClickedFlag()
{
	ProjectConfigWrite();
}

void CBuildManagerDlg::OnEnChangeLog()
{
	// TODO:  If this is a RICHEDIT control, the control will not
	// send this notification unless you override the __super::OnInitDialog()
	// function and call CRichEditCtrl().SetEventMask()
	// with the ENM_CHANGE flag ORed into the mask.

	// TODO:  Add your control notification handler code here
}
