#pragma once

#include "CryNotificationNetworkClient.h"

namespace RemoteControl {

// Macros

// Macro for registering a new Connection type.
#define ConnectionRegister(name, initialize, shutdown, refresh)\
	CConnectionDesc name##Desc(initialize, shutdown, refresh)

// Structures

struct SFileTime
{
	unsigned short year;
	unsigned short month;
	unsigned short day;
	unsigned short hour;
	unsigned short minute;
	unsigned short second;
	unsigned short milliseconds;
};

// Interfaces

// Interface for Connections to implement.
class IConnection
{
public:
	// Releases the object so that it can be deleted.
	virtual void Release() = 0;

	// Returns the type of this Connection.
	virtual const char *GetType() = 0;

	// Returns the name of this Connection.
	virtual const char *GetName() = 0;

	// Opens/Reopens the Connection.
	virtual bool Open() = 0;

	// Closes the Connection.
	virtual void Close() = 0;

	virtual bool SendCommand(const char *command) = 0;

	// Sends a command to the Connection and optionally returns a response.
	virtual bool SendCommand(const char *command, char *response, size_t &length) = 0;

	// Generates a list of directories that are contained in the given path and
	// stores the result in the given array.
	virtual bool DirectoryList(const char *remotePath, std::vector<string> &directories) = 0;

	// Copies a local file to the Connection while taking care of creating any
	// potentially inexistent directories.
	virtual bool FileCopyTo(const char *localSource, const char *remoteDestination, bool bSilent = false) = 0;

	// Copies a remote file to the given destination while taking care of
	// creating any potentially inexistent directories.
	virtual bool FileCopyFrom(const char *remoteSource, const char *localDestination, bool bSilent = false) = 0;

	// Generates a list of files contained in the given path and stores the
	// result in the given array.
	virtual bool FileList(const char *remotePath, std::vector<string> &files) = 0;

	// Sets the last change time of the file in the given path.
	virtual bool FileSetLastChangeTime(const char *remotePath, const SFileTime &fileTime) = 0;

	// Retrieves the last change time of the file pointed by the given path.
	virtual bool FileGetLastChangeTime(const char *remotePath, SFileTime &fileTime) = 0;

	virtual bool Launch(const char *localBuildsPath, const char *remotePath, const char *projectDirectory, const char *application, const char *level) = 0;

	// Intelligently Cleans the target build
	virtual void CleanBuild(const string &targetdir) = 0;

	// Local target directory
	virtual bool IsLocalTargetBuild() = 0;

	// Is connected 
	virtual bool IsTargetConnected() = 0;

	// TEMP: Should be pure virtual.
public:
	virtual bool Reset() { return false; }
	virtual bool HasActiveProcesses() { return false; }
};

// Callback interface 
class IConnectionManagerCallback
{
public:
	virtual void OnConnectionManagerLog(const char *text) = 0;
//	virtual void OnConnectionManagerMessage(const char *text) = 0;
};

// Classes

class CConnection :
	public IConnection
{
protected:
	CConnection();
	virtual ~CConnection();

protected:
	virtual void Release() = 0;

protected:
	void SetCNNAddress(const char *address) { m_cnnAddress = address; }
	bool CNNSend(const char *channel, const void *pBuffer, size_t length);

	// IConnection
public:
	bool SendCommand(const char *command);

protected:
	string m_cnnAddress;
	CCryNotificationNetworkClient *m_pCNNClient;
};

// Singleton class used to manage Connections.
class CConnectionManager
{
public:
	typedef IConnection *(*CreateConnection)(void);

public:
	static CConnectionManager &Instance();

private:
	CConnectionManager();
	~CConnectionManager();

public:
	void ReleaseConnections();

	bool Initialize();
	void Shutdown();

	void SetCallback(IConnectionManagerCallback *pCallback) { m_pCallback = pCallback; }

	// Adds a Connection to the Manager.  Should only be called by Connections' refresh methods.
	void AddConnection(IConnection *pConnection) { m_connections.push_back(pConnection); }

	// Returns the count of Connections.
	size_t GetConnectionCount() { return m_connections.size(); }

	// Returns the Connection of the given index.
	IConnection &GetConnection(size_t index) { return *m_connections[index]; }

	// Tries to open the given Connection, if it succeeds the Connection object is returned.
	IConnection *OpenConnection(size_t index);

	// Refreshes the list of available Connections.
	void RefreshConnections();

	// Logs text from the calling Connection.
	void Log(IConnection *pConnection, const char *text);

private:
	std::vector<IConnection *> m_connections;
	size_t m_connectionLast;

	IConnectionManagerCallback *m_pCallback;
};

// Class used to build the implemented Connection descriptions linked list.
// NOTE: Never use this class explicitly.
class CConnectionDesc
{
public:
	static CConnectionDesc *pDescs;

public:
	void *pInitialize;
	void *pShutdown;
	void *pRefresh;

	CConnectionDesc *pNext;

public:
	CConnectionDesc(void *pInitialize, void *pShutdown, void *pRefresh);
};

} // namespace RemoteControl
