#include "stdafx.h"
#include "ConnectionXB360.h"
#include "FileUtil.h"
#include "TaskManager.h"

using namespace RemoteControl;

// 360 refresh task

class X360ConnectionTask : public TaskManager::ITaskObject
{
	public:
		static ITaskObject * CreateTask();
		virtual ~X360ConnectionTask(){}
		virtual void Process();
		virtual void PostProcess();
	private:
		X360ConnectionTask();
		bool m_connected;
};


TaskManager::ITaskObject * X360ConnectionTask::CreateTask()
{
	CConnectionManager::Instance().Log(NULL, "Checking XBOX360 Targets for Connectivity...");
	return new X360ConnectionTask();
}

X360ConnectionTask::X360ConnectionTask()
{ 
	m_connected = false; 
}

void X360ConnectionTask::Process()
{
	DM_SYSTEM_INFO dmSystemInfo;
	::ZeroMemory(&dmSystemInfo, sizeof(DM_SYSTEM_INFO));
	dmSystemInfo.SizeOfStruct = sizeof(DM_SYSTEM_INFO);
	HRESULT hResult = ::DmGetSystemInfo(&dmSystemInfo);
	if (hResult == XBDM_NOERR)
	{
		m_connected = true;
	}
}

void X360ConnectionTask::PostProcess()
{
	if ( m_connected )
	{
		CConnectionManager::Instance().Log(NULL, "XBOX360 Target found refreshing connections..");
		CConnectionManager::Instance().AddConnection(new CConnectionXB360());
	}
	CConnectionManager::Instance().Log(NULL, "Finished Checking XBOX360 Targets for Connectivity.");
}
///////////////////////////////////////////////////////////////////////////////////

/*

  CConnectionXB360

*/

ConnectionRegister(XB360,
	NULL,
	NULL,
	&CConnectionXB360::RefreshConnections);

//

bool CConnectionXB360::RefreshConnections()
{
	TaskManager::CTaskManager::Instance().AddTask(X360ConnectionTask::CreateTask());
	return true;
}

//

bool g_bXB360Log = false;
DWORD __stdcall CConnectionXB360::DmNotificationProcess(const CHAR *notification)
{
	if (!g_bXB360Log)
		return S_OK;

	while (*notification && *notification != '!')
		++notification;
	if (!*notification)
		return S_OK;

	// TODO: Should retrieve the right connection instead of passing NULL.
	CConnectionManager::Instance().Log(NULL, ++notification);
	return S_OK;
}

//

CConnectionXB360::CConnectionXB360()
{
	m_pdmConnection = NULL;
	m_pdmnSession = NULL;
	m_bConnected = false;
}

CConnectionXB360::~CConnectionXB360()
{
}

//

bool CConnectionXB360::ApplicationConnection()
{
	DWORD responseLength = MAX_PATH;
	CHAR response[MAX_PATH];
	HRESULT hR = ::DmSendCommand(m_pdmConnection, "XCON!__connect__",
		response, &responseLength);
	if (hR != XBDM_NOERR)
	{
		if (!Open())
			return false;
	}

	return true;
}

// IConnection

const char *CConnectionXB360::GetName()
{
	// NOTE: This is not thread safe.
	static char name[1024] = { '\0' };
	DWORD length = sizeof(name);
	::DmGetNameOfXbox(name, &length, TRUE);
	return name;
}

bool CConnectionXB360::Open()
{
	m_bConnected = false;
	if (m_pdmnSession || m_pdmConnection)
		Close();

	HRESULT hR = ::DmOpenConnection(&m_pdmConnection);
	if(FAILED(hR))
	{
		CConnectionManager::Instance().Log(this,
			"DmOpenConnection failed.");
		return false;
	}

	hR = ::DmOpenNotificationSession(0, &m_pdmnSession);
	if(FAILED(hR))
	{
		CConnectionManager::Instance().Log(this,
			"DmOpenNotificationSession failed.");
		return false;
	}

	hR = ::DmRegisterNotificationProcessor(m_pdmnSession, "FXEN",
		CConnectionXB360::DmNotificationProcess);
	if(FAILED(hR))
	{
		CConnectionManager::Instance().Log(this,
			"DmRegisterNotificationProcessor");
		return false;
	}

	DWORD address = 0;
	
	if (::DmGetAltAddress(&address) == XBDM_NOERR)
	{
		static char ip[16];
		sprintf_s(ip, 16, "%d.%d.%d.%d", 
			((unsigned char *)&address)[3],
			((unsigned char *)&address)[2],
			((unsigned char *)&address)[1],
			((unsigned char *)&address)[0]);
		
		SetCNNAddress(ip);
	}

	g_bXB360Log = true;
	m_bConnected = true;
	return true;
}

void CConnectionXB360::Close()
{
	::DmRegisterNotificationProcessor(m_pdmnSession, "FXEN", NULL);

	if(m_pdmnSession)
		::DmCloseNotificationSession(m_pdmnSession);
	m_pdmnSession = NULL;

	if(m_pdmConnection)
		::DmCloseConnection(m_pdmConnection);
	m_pdmConnection = NULL;

	g_bXB360Log = false;
}

bool CConnectionXB360::SendCommand(const char *command, char *response, size_t &length)
{
	char remoteCommand[1024];
	::strcpy_s(remoteCommand, "XCON!");
	::strcat_s(remoteCommand, command);

	DWORD responseLength = (DWORD)length;
	HRESULT hR = ::DmSendCommand(m_pdmConnection, remoteCommand,
		response, &responseLength);
	length = responseLength;
	if (hR == XBDM_NOERR)
		return true;

	if (!ApplicationConnection())
		return false;

	hR = ::DmSendCommand(m_pdmConnection, remoteCommand,
		response, &responseLength);
	length = responseLength;
	if (hR != XBDM_NOERR)
	{
		// TODO: Handle error.
		return false;
	}

	return true;
}

bool CConnectionXB360::DirectoryCreate(const char *remotePath)
{
	HRESULT hResult = ::DmMkdir(remotePath);
	if (!hResult != XBDM_NOERR)
		return false;

	return true;
}

bool CConnectionXB360::DirectoryList(const char *remotePath, std::vector<string> &directories)
{
	string s = remotePath;
	::replace_chars(s, '/', '\\');

	PDM_WALK_DIR pdmWalkDir = NULL;
	HRESULT hR = XBDM_NOERR;
	while (hR == XBDM_NOERR)
	{
		DM_FILE_ATTRIBUTES dmFileAttributes;
		hR = ::DmWalkDir(&pdmWalkDir, s.c_str(), &dmFileAttributes);
		if (hR != XBDM_NOERR)
			break;

		if (dmFileAttributes.Attributes & FILE_ATTRIBUTE_DIRECTORY)
			directories.push_back(dmFileAttributes.Name);
	}

	::DmCloseDir(pdmWalkDir);

	if (hR != XBDM_ENDOFLIST)
	{
		// TODO: Handle specific error.
		return false;
	}

	return true;
}

void CConnectionXB360::LogError(HRESULT hResult)
{
	char buffer[1024];
	buffer[0] = '$';
	buffer[1] = '4';
	hResult = ::DmTranslateError(hResult, &buffer[2], sizeof(buffer)-2);
	if (hResult != XBDM_NOERR)
		return;

	CConnectionManager::Instance().Log(this, buffer);
}

bool CConnectionXB360::FileCopyTo(const char *localSource, const char *remoteDestination, bool bSilent)
{
	string destination = remoteDestination;
	::replace_chars(destination, '/', '\\');

	string s;

	size_t offset = 0;
	while (true)
	{
		s = destination;
		size_t length = Path::GetDirectoryLength(&s[offset]);
		if (!length)
			break;

		offset += length;
		s.clear();
		s.append(destination.c_str(), offset);
		DirectoryCreate(s.c_str());
	}

	HRESULT hResult = ::DmSendFile(localSource, destination.c_str());
	if (hResult != XBDM_NOERR)
	{
		if (!bSilent)
			LogError(hResult);
		return false;
	}

	return true;
}

bool CConnectionXB360::FileCopyFrom(const char *remoteSource, const char *localDestination, bool bSilent)
{
	string s = remoteSource;
	::replace_chars(s, '/', '\\');

	HRESULT hResult = ::DmReceiveFile(localDestination, s.c_str());
	if (hResult != XBDM_NOERR)
	{
		if (!bSilent)
			LogError(hResult);
		return false;
	}

	return true;
}

bool CConnectionXB360::FileDelete(const char *remotePath)
{
	HRESULT hResult = ::DmDeleteFile(remotePath, FALSE);
	if (hResult != XBDM_NOERR)
		return false;

	return true;
}

bool CConnectionXB360::FileList(const char *remotePath, std::vector<string> &files)
{
	string s = remotePath;
	::replace_chars(s, '/', '\\');

	PDM_WALK_DIR pdmWalkDir = NULL;
	HRESULT hR = XBDM_NOERR;
	while (hR == XBDM_NOERR)
	{
		DM_FILE_ATTRIBUTES dmFileAttributes;
		hR = ::DmWalkDir(&pdmWalkDir, s.c_str(), &dmFileAttributes);
		if (hR != XBDM_NOERR)
			break;

		if (dmFileAttributes.Attributes & FILE_ATTRIBUTE_DIRECTORY)
			continue;

		files.push_back(dmFileAttributes.Name);
	}

	::DmCloseDir(pdmWalkDir);

	if (hR != XBDM_ENDOFLIST)
	{
		// TODO: Handle specific error.
		return false;
	}

	return true;
}

bool CConnectionXB360::DirectoryDelete(const char *remotePath)
{
	HRESULT hResult = ::DmDeleteFile(remotePath, TRUE);
	if (hResult != XBDM_NOERR)
		return false;

	return true;
}

bool CConnectionXB360::FileSetLastChangeTime(const char *remotePath, const SFileTime &fileTime)
{
	string s(remotePath);
	::replace_chars(s, '/', '\\');

	DM_FILE_ATTRIBUTES dmFileAttributes;
	HRESULT hResult = ::DmGetFileAttributes(s.c_str(), &dmFileAttributes);
	if (hResult != XBDM_NOERR)
		return false;

	SYSTEMTIME systemTime;
	systemTime.wYear = fileTime.year;
	systemTime.wMonth = fileTime.month;
	systemTime.wDay = fileTime.day;
	systemTime.wHour = fileTime.hour;
	systemTime.wMinute = fileTime.minute;
	systemTime.wSecond = fileTime.second;
	systemTime.wMilliseconds = fileTime.milliseconds;
	if (!::SystemTimeToFileTime(&systemTime, &dmFileAttributes.ChangeTime))
		return false;

	hResult = ::DmSetFileAttributes(s.c_str(), &dmFileAttributes);
	if (hResult != XBDM_NOERR)
		return false;

	return true;
}

bool CConnectionXB360::FileGetLastChangeTime(const char *remotePath, SFileTime &fileTime)
{
	string s(remotePath);
	::replace_chars(s, '/', '\\');

	DM_FILE_ATTRIBUTES dmFileAttributes;
	HRESULT hResult = ::DmGetFileAttributes(s.c_str(), &dmFileAttributes);
	if (hResult != XBDM_NOERR)
		return false;

	SYSTEMTIME systemTime;
	if (!::FileTimeToSystemTime(&dmFileAttributes.ChangeTime, &systemTime))
		return false;

	fileTime.year = systemTime.wYear;
	fileTime.month = systemTime.wMonth;
	fileTime.day = systemTime.wDay;
	fileTime.hour = systemTime.wHour;
	fileTime.minute = systemTime.wMinute;
	fileTime.second = systemTime.wSecond;
	fileTime.milliseconds = systemTime.wMilliseconds;
	return true;
}

bool CConnectionXB360::Launch(const char *localBuildsPath, const char *remotePath, const char *projectDirectory, const char *application, const char *level)
{
	string s = remotePath;
	::replace_chars(s, '/', '\\');
	string command = "+map ";
	
	if(level)
	{
		command += level;
	}

	::DmReboot(DMBOOT_WAIT);
	::DmSetTitleEx(s.c_str(), application, command.c_str(), NULL);
	return true;
}


static void recursedeletexbox(const string & path)
{
	HRESULT error;
	PDM_WALK_DIR pWalkDir = NULL;
	DM_FILE_ATTRIBUTES fileAttr;

	do
	{
		error = DmWalkDir(&pWalkDir, path.c_str(), &fileAttr);
		string itempath = FolderParseUtil::Instance().Join(path,fileAttr.Name);;
		if ( fileAttr.Attributes & FILE_ATTRIBUTE_DIRECTORY )
		{		
			if ( !FolderParseUtil::Instance().IgnoreBinaryFileOrFolder(itempath,false) )
			{
				recursedeletexbox(itempath);
				::DmDeleteFile(itempath.c_str(),TRUE);
			}
		}
		else
		{
			if ( !FolderParseUtil::Instance().IgnoreBinaryFileOrFolder(itempath,true) )
			{
				::DmDeleteFile(itempath.c_str(), FALSE);
			}
		}
		
	}
	while (error == XBDM_NOERR);
	DmCloseDir(pWalkDir);
}

void CConnectionXB360::CleanBuild(const string &targetdir)
{
	CConnectionManager::Instance().Log(this, "Cleaning build folder...");
	string message = "Removing folder: '" + targetdir + "'...";
	recursedeletexbox(targetdir);
	::DmDeleteFile(targetdir.c_str(),TRUE);
	CConnectionManager::Instance().Log(this, "Build Destination has been cleaned.");
}
