#include "stdafx.h"
#include "ProjectManager.h"
#include "BuildManager.h"
#include "BuildManagerDlg.h"
#include "FileUtil.h"

#include <iostream>
#include <fstream>
#include <string>
#include <algorithm>

/*

  CProjectManager

*/

CProjectManager &CProjectManager::Instance()
{
	static CProjectManager instance;
	return instance;
}

//

CProjectManager::CProjectManager()
{
	m_projectActive = NULL;

	m_pConnection = NULL;

	m_uploadBuildIndex = NULL;
	m_uploadBuildExecuting = false;
	m_uploadBuildAbort = false;
	m_uploadBuildSuccessful = false;

	m_pThread = ThreadManager::CThread::Create();
}

CProjectManager::~CProjectManager()
{
	if (m_pThread)
		m_pThread->Release();
}

//

bool CProjectManager::AddProjectFile(const char *path)
{
	char buffer[4095];
	memset(buffer, '\0', sizeof(buffer)/sizeof(char));
	CDesc desc;

	if (!::GetPrivateProfileString("Project", "name", NULL, buffer, sizeof(buffer)-1, path))
		return false;
	desc.name = buffer;

	if (!::GetPrivateProfileString("Project", "buildsPath", NULL, buffer, sizeof(buffer)-1, path))
		return false;
	desc.localBuildsPath = buffer;

	::GetPrivateProfileString("Project", "cleanTargetFolders", NULL, buffer, sizeof(buffer)-1, path);
	FolderParseUtil::Instance().ParseFoldersToClean(buffer);

	::GetPrivateProfileString("Project", "ignoreBinaries", NULL, buffer, sizeof(buffer)-1, path);
	FolderParseUtil::Instance().ParseBinariesToIgnore(buffer);

	desc.filterIncompleteBuilds = (bool)::GetPrivateProfileInt("Project","filterIncompleteBuilds", 0, path);


	size_t length = ::GetPrivateProfileSectionNames(buffer, sizeof(buffer)-1, path);
	if (!length)
		return false;

	CDesc::CPlatform platform;
	size_t offset = 0;
	size_t current = 0;

	char subBuffer[1024];
	memset(subBuffer, '\0', sizeof(subBuffer)/sizeof(char));
	while (offset < length)
	{
		const char *platformName = &buffer[offset];
		offset += ::strlen(&buffer[offset]) + 1;

		if (!::GetPrivateProfileString(platformName, "buildsDirectory", NULL, subBuffer, sizeof(subBuffer)-1, path))
			continue;
		platform.buildsDirectory = subBuffer;
		if (!::GetPrivateProfileString(platformName, "remotePath", NULL, subBuffer, sizeof(subBuffer)-1, path))
			continue;
		platform.remotePath = subBuffer;
		if (!::GetPrivateProfileString(platformName, "application", NULL, subBuffer, sizeof(subBuffer)-1, path))
			continue;
		platform.application = subBuffer;
		if (!::GetPrivateProfileString(platformName, "project", NULL, subBuffer, sizeof(subBuffer)-1, path))
			continue;
		platform.project = subBuffer;

		desc.platforms[platformName] = platform;
	}

	if (desc.platforms.empty())
		return false;

	AddProject(desc);
	return true;
}

void CProjectManager::AddProjectFiles(const char *path)
{
	WIN32_FIND_DATA findData;
	string s = path;
	::append_if_not_last(s, '/', "/\\");
	s += "*.ini";
	HANDLE hR = ::FindFirstFile(s.c_str(), &findData);
	while (true)
	{
		s = path;
		::append_if_not_last(s, '/', "/\\");
		s += findData.cFileName;

		AddProjectFile(s.c_str());
		if (!::FindNextFile(hR, &findData))
			break;
	}
	if (hR)
		::FindClose(hR);
}

void CProjectManager::ClearProjects()
{
	m_projects.clear();
	m_projectActive = 0;

	m_builds.clear();

	m_remotePath.clear();
	m_remoteVersion.clear();
	m_remoteLevels.clear();
}

bool CProjectManager::ActiveProject(size_t index)
{
	if (index >= m_projects.size())
		return false;

	m_projectActive = index;

	return true;
}

bool CProjectManager::ActiveConnection(RemoteControl::IConnection *pConnection)
{
	m_pConnection = pConnection;

	CDesc::CPlatform *pPlatform = GetActivePlatform();
	if (!pPlatform)
		return false;

	m_remotePath = pPlatform->remotePath;

	RetrieveLocalBuilds();

	if (!RefreshConnectionCurrent())
		return false;

	return true;
}

bool CProjectManager::RefreshConnectionCurrent()
{
	if (!m_pConnection)
		return false;

	if (!RetrieveRemoteBuildVersion())
		return false;

	RetrieveRemoteLevels();
	return true;
}

CProjectManager::CDesc::CPlatform *CProjectManager::GetActivePlatform()
{
	if (!m_pConnection)
		return NULL;

	if (m_projects.size() <= m_projectActive)
		return NULL;

	std::map<string, CDesc::CPlatform>::iterator platform =
		m_projects[m_projectActive].platforms.find(m_pConnection->GetType());
	if (platform == m_projects[m_projectActive].platforms.end())
		return NULL;

	return &m_projects[m_projectActive].platforms[m_pConnection->GetType()];
}

bool CProjectManager::SetRemotePath(const char *path)
{
	if (!path)
		return false;

	if (!::strlen(path))
		return false;

	m_remotePath = path;
	return false;
}

const char *CProjectManager::GetRemotePath()
{
	return m_remotePath.c_str();
}

//

bool CProjectManager::RetrieveLocalBuilds()
{
	m_builds.clear();

	CDesc::CPlatform *pPlatform = GetActivePlatform();
	if (!pPlatform)
		return false;

	string s = m_projects[m_projectActive].localBuildsPath;
	::append_if_not_last(s, '/', "/\\");
	s += pPlatform->buildsDirectory;
	::append_if_not_last(s, '/', "/\\");
	s += "*.*";

	WIN32_FIND_DATA findData;
	HANDLE hR = ::FindFirstFile(s.c_str(), &findData);
	while (true)
	{
		if ((findData.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY) == FILE_ATTRIBUTE_DIRECTORY)
		{
			if (::strcmp(findData.cFileName, ".") != 0 &&
				::strcmp(findData.cFileName, "..") != 0)
			{
				m_builds.push_back(findData.cFileName);
			}
		}

		if (!::FindNextFile(hR, &findData))
			break;
	}
	if (hR)
		::FindClose(hR);

	std::sort(m_builds.begin(), m_builds.end());

	return true;
}

bool CProjectManager::RetrieveRemoteBuildVersion()
{
	m_remoteVersion.clear();

	if (!m_pConnection)
		return false;

	CDesc::CPlatform *pPlatform = GetActivePlatform();
	if (!pPlatform)
		return false;

	string s = m_remotePath;
	::append_if_not_last(s, '\\', "/\\");
	s += "buildversion";
	if (!m_pConnection->FileCopyFrom(s.c_str(), "C:/buildversion", true))
		return false;

	std::ifstream inFile;
	inFile.open("C:/buildversion");
	if (!inFile.is_open())
		return false;

	char version[1024];
	memset(version,'\0',sizeof(version)/sizeof(char));
	inFile.get(version, sizeof(version));
	inFile.close();
	::remove("C:/buildversion");

	m_remoteVersion = version;
	return true;
}

bool CProjectManager::IsPathLevel(const char *path)
{
	std::vector<string> files;
	if (!m_pConnection->FileList(path, files))
		return false;

	for (size_t i=0; i<files.size(); ++i)
	{
		if (::_strnicmp(files[i].c_str(), "level.pak", 9) == 0)
			return true;
	}

	return false;
}

bool CProjectManager::RetrieveRemoteLevelsRecursion(const char *path, size_t rootLength)
{
	string s;
	if (IsPathLevel(path))
	{
		s = &path[rootLength+1];
		::replace_chars(s, '\\', '/');
		m_remoteLevels.push_back(s);
		return true;
	}

	std::vector<string> directories;
	if (!m_pConnection->DirectoryList(path, directories))
		return false;

	for (size_t i=0; i<directories.size(); ++i)
	{
		s = path;
		::append_if_not_last(s, '/', "/\\");
		s += directories[i];
		RetrieveRemoteLevelsRecursion(s.c_str(), rootLength);
	}

	return true;
}

bool CProjectManager::RetrieveRemoteLevels()
{
	if (m_remoteVersion.empty())
	{
		if (!RetrieveRemoteBuildVersion())
			return false;
	}

	if (!m_pConnection)
		return false;

	m_remoteLevels.clear();

	CDesc::CPlatform *pPlatform = GetActivePlatform();
	if (!pPlatform)
		return false;

	// If we are connected to a PS3 then force lowercase.
	if (::_strnicmp(m_pConnection->GetType(), "PS3", 3) == 0)
		::_strlwr(&pPlatform->project[0]);

	string s = m_remotePath;
	::append_if_not_last(s, '/', "/\\");
	s += pPlatform->project;
	::append_if_not_last(s, '/', "/\\");
	s += "levels"; // NOTE: Keep it lowercase for PS3 compatibility.

	return RetrieveRemoteLevelsRecursion(s.c_str(), s.length());
}

bool CProjectManager::UploadBuildRecursion(const std::string &path, const std::string &pathSub, bool bForceLowercase)
{
	if (!pathSub.empty())
	{
		if (m_uploadBuildExclude & eExclude_Binaries)
		{
			if ( FolderParseUtil::Instance().IgnoreBinaryFileOrFolder(pathSub,false) )
			{
				return true;
			}
		}
		if (m_uploadBuildExclude & eExclude_Levels)
		{
			std::string levels("levels");
			if (pathSub.length() >= levels.length()+1)
			{
				std::string endsubpath = pathSub.substr(pathSub.length()-levels.length());
				std::transform(endsubpath.begin(), endsubpath.end(), endsubpath.begin(), tolower);
				if ( levels == endsubpath )
				{
					return true;
				}
			}
		}
		if (m_uploadBuildExclude & eExclude_SystemCFG)
		{
			std::string systemcfg("system.cfg");
			if (pathSub.length() >= systemcfg.length()+1)
			{
				if ( pathSub.compare(pathSub.length()-systemcfg.length(),systemcfg.length(),systemcfg) == 0)
				{
					return true;
				}
			}
		}
	}

	string remoteDetinationPath = m_remotePath;
	string s = path;
	::append_if_not_last(s, '/', "/\\");
	if (!pathSub.empty()) 
		s += pathSub;
	::append_if_not_last(s, '/', "/\\");
	s += "*.*";

	string fileSource, fileDestination;

	WIN32_FIND_DATA findData;
	SYSTEMTIME systemTime;
	RemoteControl::SFileTime fileTime;
	HANDLE hR = ::FindFirstFile(s.c_str(), &findData);
	if (hR == INVALID_HANDLE_VALUE)
		return false;

	bool bCopied = false;
	while (true)
	{
		if (m_uploadBuildAbort)
			return false;

		if ((findData.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY) == FILE_ATTRIBUTE_DIRECTORY)
		{
			if (::strcmp(findData.cFileName, ".") != 0 &&
				::strcmp(findData.cFileName, "..") != 0)
			{
				s.clear();
				if (!pathSub.empty()) s = pathSub;
				::append_if_not_last(s, '/', "/\\");
				s += findData.cFileName;
				
				if (bForceLowercase)
					::_strlwr((char *)s.c_str());
				if (!UploadBuildRecursion(path, s.c_str(), bForceLowercase))
					return false;
			}
		}
		else
		{
			fileDestination = remoteDetinationPath;
			::append_if_not_last(fileDestination, '/', "/\\");
			if (!pathSub.empty()) fileDestination += pathSub;
			::append_if_not_last(fileDestination, '/', "/\\");
			size_t forceLowercaseOffset = fileDestination.length();
			fileDestination += findData.cFileName;

			bool bSkip = false;
			bool bExists = false;

			if (m_uploadBuildExclude & eExclude_Binaries)
			{
				std::string filestr = findData.cFileName;
				if ( filestr.length() > 3 && FolderParseUtil::Instance().IgnoreBinaryFileOrFolder(filestr,true) )
				{
					bSkip = true;
				}
			}
			
			if (!bSkip)
			{
				::FileTimeToSystemTime(&findData.ftLastWriteTime, &systemTime);
				bExists = m_pConnection->FileGetLastChangeTime(fileDestination.c_str(), fileTime);
				if (bExists)
				{
					if (systemTime.wYear > fileTime.year)
						bExists = false;
					else if (systemTime.wMonth > fileTime.month)
						bExists = false;
					else if (systemTime.wDay > fileTime.day)
						bExists = false;
					else if (systemTime.wMinute > fileTime.minute)
						bExists = false;
					else
						bExists = true;
				}
			}

			if (bSkip)
			{
				s.clear();
				s += "File \"";
				s += fileDestination;
				s += "\" skipped.";
				bCopied = true;
			}
			else if (bExists)
			{
				s.clear();
				s += "File \"";
				s += fileDestination;
				s += "\" already up to date.";
				bCopied = true;
			}
			else
			{
				fileSource = path;
				::append_if_not_last(fileSource, '/', "/\\");
				if (!pathSub.empty()) fileSource += pathSub;
				::append_if_not_last(fileSource, '/', "/\\");
				fileSource += findData.cFileName;

				fileTime.year = systemTime.wYear;
				fileTime.month = systemTime.wMonth;
				fileTime.day = systemTime.wDay;
				fileTime.hour = systemTime.wHour;
				fileTime.minute = systemTime.wMinute;
				fileTime.second = systemTime.wSecond;
				fileTime.milliseconds = systemTime.wMilliseconds;

				if (bForceLowercase)
					::_strlwr((char *)&fileDestination[forceLowercaseOffset]);
				if (bCopied = m_pConnection->FileCopyTo(fileSource.c_str(), fileDestination.c_str()))
					m_pConnection->FileSetLastChangeTime(fileDestination.c_str(), fileTime);

				s.clear();
				s += bCopied ? "Copying \"" : "$4Copying \"";
				s += fileSource;
				s += "\" to \"";
				s += fileDestination;
				s += bCopied ? "\"." : "\" failed.";
			}

			// TEMP
			((CBuildManagerDlg *)::AfxGetApp()->GetMainWnd())->LogOutput(s.c_str());

			if (!bCopied)
				return false;
			bCopied = false;
		}

		if (!::FindNextFile(hR, &findData))
			break;
	}
	if (hR)
		::FindClose(hR);

	return true;
}

bool CProjectManager::UploadBuild()
{
	CDesc::CPlatform *pPlatform = GetActivePlatform();
	if (!pPlatform)
		return false;

	if (m_uploadBuildIndex >= m_builds.size())
		return false;

	FolderParseUtil::Instance().SetIgnoreBinaries(m_uploadBuildExclude & eExclude_Binaries);

	if ( m_uploadBuildExclude & eClean_Build )
		m_pConnection->CleanBuild(m_remotePath);

	string s = m_projects[m_projectActive].localBuildsPath;
	::append_if_not_last(s, '/', "/\\");
	s += pPlatform->buildsDirectory;
	::append_if_not_last(s, '/', "/\\");
	s += m_builds[m_uploadBuildIndex];

	bool bForceLowercase = ::_strnicmp(m_pConnection->GetType(), "PS3", 3) == 0;
	if (!UploadBuildRecursion(s,"", bForceLowercase))
		return false;

	// Create and upload the buildversion file.  It is always copied last to
	// validated a successful upload.

	std::ofstream file;
	file.open("C:/buildversion");
	if (!file.is_open())
		return false;

	file << m_builds[m_uploadBuildIndex].c_str();
	file.flush();
	file.close();

	s = m_remotePath;
	::append_if_not_last(s, '/', "/\\");
	s += "buildversion";
	if (!m_pConnection->FileCopyTo("C:/buildversion", s.c_str()))
		return false;

	if (!RefreshConnectionCurrent())
		return false;

	return true;
}

//

void CProjectManager::RoutineUploadBuild(void *pParam)
{
	((CProjectManager *)pParam)->m_uploadBuildSuccessful =
		((CProjectManager *)pParam)->UploadBuild();
	((CProjectManager *)pParam)->m_uploadBuildExecuting = false;
	((CProjectManager *)pParam)->m_uploadBuildAbort = false;
}

bool CProjectManager::UploadBuildBegin(size_t index, unsigned int exclude)
{
	m_uploadBuildIndex = index;
	m_uploadBuildExclude = exclude;
	m_uploadBuildExecuting = true;
	m_uploadBuildAbort = false;

	ThreadManager::CTask *pTask =
		m_pThread->CreateTask(CProjectManager::RoutineUploadBuild, this);
	if (!pTask)
		return false;

	return true;
}

bool CProjectManager::IsUploadingBuild(bool *pWasSuccessful)
{
	if (pWasSuccessful)
	{
		*pWasSuccessful = m_uploadBuildSuccessful;
		m_uploadBuildSuccessful = false;
	}

	return m_uploadBuildExecuting;
}

bool CProjectManager::CalculateBuildServerPath(const string& buildname, string & buildpath)
{
	CDesc::CPlatform *pPlatform = GetActivePlatform();
	if (!pPlatform)
		return false;

	buildpath = m_projects[m_projectActive].localBuildsPath;
	::append_if_not_last(buildpath, '/', "/\\");

	buildpath += pPlatform->buildsDirectory;
	::append_if_not_last(buildpath, '/', "/\\");

	buildpath += buildname;
	return true;
}

void CProjectManager::Launch(size_t level)
{
	if (!m_pConnection)
		return;

	CDesc::CPlatform *pPlatform = GetActivePlatform();
	if (!pPlatform)
		return;

	if ( ( level != NO_LEVEL ) && (level >= m_remoteLevels.size()) )
		return;

	string s = m_projects[m_projectActive].localBuildsPath.c_str();
	const char* const levelName = ( level != NO_LEVEL ) ? m_remoteLevels[level].c_str() : NULL;
	::append_if_not_last(s, '/', "/\\");
	s += pPlatform->buildsDirectory;
	::append_if_not_last(s, '/', "/\\");
	m_pConnection->Launch(
		s.c_str(),
		m_remotePath.c_str(),
		pPlatform->project.c_str(),
		pPlatform->application.c_str(),
		levelName );
}

void CProjectManager::SetFilterBuilds(bool filter)
{
	m_projects[m_projectActive].filterIncompleteBuilds = filter;
}

bool CProjectManager::FilterBuild(const string& buildname)
{
	if ( m_projects[m_projectActive].filterIncompleteBuilds )
	{
		string buildpath;
		if ( CalculateBuildServerPath(buildname, buildpath) )
		{
			string completedFile = buildpath + "\\completed.txt";
			if ( !FolderParseUtil::Instance().FileExists(completedFile) )
			{
				return true;
			}
		}
	}
	return false;
}