﻿using System;
using System.Collections.Generic;
using System.Text;
using System.Text.RegularExpressions;

namespace BuildTools
{
	public class BaseBuildInfoList : IEnumerable<BuildInfo>
	{
		private Regex versionRegEx;
		private Platform platform;
		private List<BuildInfo> buildInfoList;

		protected BaseBuildInfoList(string project, Platform platform)
		{
			this.platform = platform;
			this.versionRegEx = new Regex(project + BuildInfo.GetPlatformString(platform) + @"\((?<number>\d+)\)_(?<month>\d+)_(?<day>\d+)");
			this.buildInfoList = new List<BuildInfo>();
		}

		protected void OnUpdateDirectory(string directoryName, string fullDirectoryName)
		{
			Match match = versionRegEx.Match(directoryName);

			if (match.Success)
			{
				int month = Int32.Parse(match.Groups["month"].Value);
				int day = Int32.Parse(match.Groups["day"].Value);
				int number = Int32.Parse(match.Groups["number"].Value);

				buildInfoList.Add(new BuildInfo(number, platform, new DateTime(2009, month, day), fullDirectoryName));
			}
		}

		protected void OnUpdateStart()
		{
			buildInfoList.Clear();
		}

		protected void OnUpdateDone()
		{
			buildInfoList.Sort();
		}

		public Platform Platform
		{
			get { return platform; }
		}

		public BuildInfo this[int idx]
		{
			get { return buildInfoList[idx]; }
		}

		public int Count
		{
			get { return buildInfoList.Count; }
		}

		public IEnumerator<BuildInfo> GetEnumerator()
		{
			return buildInfoList.GetEnumerator();
		}

		System.Collections.IEnumerator System.Collections.IEnumerable.GetEnumerator()
		{
			return GetEnumerator();
		}

		public IEnumerable<BuildInfo> Reverse
		{
			get
			{
				for (int idx = buildInfoList.Count - 1; idx >= 0; idx--)
					yield return buildInfoList[idx];
			}
		}

		public BuildInfo GetLastVersion()
		{
			BuildInfo res = null;

			int size = buildInfoList.Count;
			if (size > 0)
				res = buildInfoList[size - 1];

			return res;
		}

		public IList<BuildInfo> GetVersions(DateTime date)
		{
			List<BuildInfo> res = new List<BuildInfo>();

			foreach (BuildInfo buildInfo in buildInfoList)
			{
				if (buildInfo.DateTime.Month == date.Month && buildInfo.DateTime.Day == date.Day)
					res.Add(buildInfo);
			}

			return res;
		}

		public BuildInfo GetFirstVersion(DateTime date)
		{
			BuildInfo res = null;
			IList<BuildInfo> versions = GetVersions(date);

			if (versions.Count > 0)
				res = versions[0];

			return res;
		}

		public BuildInfo GetLastVersion(DateTime date)
		{
			BuildInfo res = null;
			IList<BuildInfo> versions = GetVersions(date);
			int size = versions.Count;

			if (size > 0)
				res = versions[size - 1];

			return res;
		}
	}
}
