using System;
using System.IO;
using System.Diagnostics;

namespace BuildTools
{
	public class Xbox360Exception : Exception
	{
		public Xbox360Exception(string message)
			: base(message)
		{ 
		
		}
	}

	public class Xbox360
	{
		private string _copyCmd;
		private string _delCmd;
		private bool _createNoWindow;
		private bool _minimized;
		private bool _useShellExecute;

		public Xbox360()
		{
			string xbcpFileName = "xbcp.exe";
			string xbdelFileName = "xbdel.exe";
			string _sdkEnvPath = "XEDK";

			string xedk = Environment.GetEnvironmentVariable(_sdkEnvPath);

			if (xedk == null || xedk.Length == 0)
				throw new Xbox360Exception("Xbox360 SDK not available");

			string win32Folder = FileUtils.CombinePath(xedk, "bin", "win32");

			_copyCmd = Path.Combine(win32Folder, xbcpFileName);
			_delCmd = Path.Combine(win32Folder, xbdelFileName);

			if (!File.Exists(_copyCmd))
				throw new Xbox360Exception(xbcpFileName + " command not found");

			if (!File.Exists(_delCmd))
				throw new Xbox360Exception(xbdelFileName + " command not found");

			_createNoWindow = false;
			_minimized = false;
			_useShellExecute = false;
		}

		public bool Copy(string xboxAddress, string src, string dst)
		{
			CmdLineArgsBuilder args = new CmdLineArgsBuilder();
			
			if (xboxAddress != null)
				args.AppendFlag("X", xboxAddress);

			args.AppendFlag("Y");
			args.AppendFlag("C");
			args.AppendFlag("T");
			args.AppendFlag("R");
			args.AppendFlag("H");
			args.AppendFlag("D");
			if (File.Exists(src))
				args.Append(src);
			else
				args.Append(Path.Combine(src, "*.*"));
			args.Append(string.Format(@"xE:\{0}", dst));

			return RunProcess(_copyCmd, args);
		}

		public bool Copy(string src, string dst)
		{
			return Copy(null, src, dst);
		}

		public bool Del(string xboxAddress, string dir)
		{
			CmdLineArgsBuilder args = new CmdLineArgsBuilder();

			if (xboxAddress != null)
				args.AppendFlag("X", xboxAddress);

			args.AppendFlag("R");
			args.AppendFlag("H");
			args.AppendFlag("F");
			args.Append(string.Format(@"xE:\{0}", dir));

			return RunProcess(_delCmd, args);
		}

		public bool Del(string dir)
		{
			return Del(null, dir);
		}

		public bool CreateNoWindow
		{
			set { _createNoWindow = value; }
		}

		public bool Minimized
		{
			set { _minimized = value; }
		}

		public bool UseShellExecute
		{
			set { _useShellExecute = value; }
		}
		
		private bool RunProcess(string fileName, CmdLineArgsBuilder args)
		{
			Process process = new Process();
			process.StartInfo.CreateNoWindow = _createNoWindow;

			if (_minimized)
				process.StartInfo.WindowStyle = ProcessWindowStyle.Minimized;

			process.StartInfo.FileName = fileName;
			process.StartInfo.UseShellExecute = _useShellExecute;
			process.StartInfo.Arguments = args.ToWin();

			process.Start();
			process.WaitForExit();

			return process.ExitCode == 0;
		}
	}
}
