using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Text;
using System.Windows.Forms;
using BuildTools;
using System.Diagnostics;
using System.IO;

namespace SystemTray
{
	public partial class SystemTrayForm : Form
	{
        private Settings m_settings;
		BuildWatchers m_buildWatchers;

		public SystemTrayForm()
		{
			InitializeComponent();

			m_settings = new Settings();
			m_buildWatchers = CreateWatchers();
		}

		private BuildWatchers CreateWatchers()
		{
			BuildWatchers buildWatchers = null;
			if (!m_settings.DisableNotifier)
			{
				buildWatchers = new BuildWatchers(m_settings.BuildsFolder, m_settings.Project, m_settings.TimePeriod);
				buildWatchers.OnNewBuild += OnNewBuild;
			}
			return buildWatchers;
		}

		public void ShowBaloonTipInfoAsync(int lastBuildNumber, Platform platform)
		{
			BeginInvoke((MethodInvoker)delegate()
			{
				string title = string.Format("New {0} build available", platform.ToString());
				string text = string.Format("Last {0} build is {1}", platform.ToString(), lastBuildNumber);
				notifyIcon.ShowBalloonTip(0, title, text, ToolTipIcon.Info);
				textBoxLogs.Text += DateTime.Now.ToString() + " " + title + ". " + text  + "\r\n";
			});
		}

		private void OnNewBuild(BuildInfo buildInfo) // from watcher -> not on main thread
		{
			ShowBaloonTipInfoAsync(buildInfo.Number, buildInfo.Platform);

			if (buildInfo.Platform == Platform.Xbox360)
			{
				OnNewXbox360Build(buildInfo);
			}

			if (buildInfo.Platform == Platform.Pc)
			{
				OnNewPcBuild(buildInfo);
			}
		}

		private void OnNewPcBuild(BuildInfo buildInfo) // from watcher -> not on main thread
		{
			if (m_settings.PcAutoCopy)
			{
				BeginInvoke((MethodInvoker)delegate()
				{
					CopyPcBuild(buildInfo);
				});
			}

			if (m_settings.PcAutoCopyData)
			{
				BeginInvoke((MethodInvoker)delegate()
				{
					CopyPcData(buildInfo);

				});
			}
		}

		private void OnNewXbox360Build(BuildInfo buildInfo) // from watcher -> not on main thread
		{
			if (m_settings.XboxAutoCopy)
			{
				BeginInvoke((MethodInvoker)delegate()
				{
					CopyXboxBuild(buildInfo);
				});
			}

			if (m_settings.XboxAutoCopyData)
			{
				BeginInvoke((MethodInvoker)delegate()
				{
					CopyXboxData(buildInfo);
				});
			}
		}

		#region Copy Build and Data

		private void CopyPcBuild(BuildInfo buildInfo)
		{
			string destination = Path.Combine(m_settings.PcDestFolder, buildInfo.DirectoryName);
			CopyForm copyForm = new CopyForm(new CopyPcBuild(buildInfo, destination));
			copyForm.Show();
		}

		private void CopyPcData(BuildInfo buildInfo)
		{
			CopyForm copyForm = new CopyForm(new CopyPcData(buildInfo, m_settings.PcLatestDataDestFolder));
			copyForm.Show();
		}

		private void CopyXboxBuild(BuildInfo buildInfo)
		{
			CopyXBox360Build copyFiles = new CopyXBox360Build(buildInfo, m_settings.XboxUseDefault, m_settings.XboxAddress);
			if (copyFiles.ConnectionEsablished())
			{
				CopyForm copyForm = new CopyForm(copyFiles);
				copyForm.Show();
			}
			else
			{
				MessageBox.Show("Can't connect to selected Xbox360");
			}
		}

		private void CopyXboxData(BuildInfo buildInfo)
		{
			CopyXbox360Data copyFiles = new CopyXbox360Data(buildInfo, m_settings.XboxLatestDataDestFolder, m_settings.XboxUseDefault, m_settings.XboxAddress);
			if (copyFiles.ConnectionEsablished())
			{
				CopyForm copyForm = new CopyForm(copyFiles);
				copyForm.Show();
			}
			else
			{
				MessageBox.Show("Can't connect to selected Xbox360");
			}
		}

		#endregion

		#region ToolStrip Menu

		private void exitToolStripMenuItem_Click(object sender, EventArgs e)
		{
			Application.Exit();
		}

		private void settingsToolStripMenuItem_Click(object sender, EventArgs e)
		{
			SettingsForm settingsForm = new SettingsForm();
			if (DialogResult.OK == settingsForm.ShowDialog())
			{
				Settings newSettings = settingsForm.Settings;
				if (!newSettings.Equals(m_settings))
				{
					m_settings = newSettings;
					m_settings.Save();
					m_buildWatchers.Stop();
					m_buildWatchers = CreateWatchers();
					MessageBox.Show("Changes applied.");
				}
				else
				{
					MessageBox.Show("Nothing changed.");
				}
			}
		}

		private BuildInfo GetLastBuildInfo(Platform platform)
		{
			BuildInfo buildInfo = null;
			
			try
			{
				BuildInfoList buildInfoList = BuildInfoList.Create(m_settings.BuildsFolder, m_settings.Project, platform);

				buildInfoList.Update();
				buildInfo = buildInfoList.GetLastVersion();
				return buildInfo;
			}
			catch (Exception) {}

			return buildInfo;
		}

		private void pcCopyBuildToolStripMenuItem_Click(object sender, EventArgs e)
		{
			BuildInfo buildInfo = GetLastBuildInfo(Platform.Pc);
			if (buildInfo != null)
			{
				CopyPcBuild(buildInfo);

			}
			else
			{
				MessageBox.Show("Can't find any build");
			}
		}

		private void pcCopyDataToolStripMenuItem_Click(object sender, EventArgs e)
		{
			if (!Directory.Exists(m_settings.PcLatestDataDestFolder))
			{
				MessageBox.Show("Destination folder does not exist", "Error", MessageBoxButtons.OK, MessageBoxIcon.Error);
				return;
			}

			BuildInfo buildInfo = GetLastBuildInfo(Platform.Pc);
			if (buildInfo != null)
			{
				CopyPcData(buildInfo);
			}
			else
			{
				MessageBox.Show("Can't find any build");
			}
		}

		private void xbox360CopyBuildToolStripMenuItem_Click(object sender, EventArgs e)
		{
			BuildInfo buildInfo = GetLastBuildInfo(Platform.Xbox360);
			if (buildInfo != null)
			{
				CopyXboxBuild(buildInfo);
			}
			else
			{
				MessageBox.Show("Can't find any build");
			}
		}

		private void xbox360CopyDataToolStripMenuItem_Click(object sender, EventArgs e)
		{
			if (m_settings.XboxLatestDataDestFolder.Length == 0)
			{
				MessageBox.Show("Destination Xbox360 folder has not been set", "Error", MessageBoxButtons.OK, MessageBoxIcon.Error);
				return;
			}

			BuildInfo buildInfo = GetLastBuildInfo(Platform.Xbox360);
			if (buildInfo != null)
			{
				CopyXboxData(buildInfo);
			}
			else
			{
				MessageBox.Show("Can't find any build");
			}
		}

		#endregion

		#region System Tray Management

		private void SystemTrayForm_Shown(object sender, EventArgs e)
		{
			Hide();
		}

		private void SystemTrayForm_Resize(object sender, EventArgs e)
		{
			if (WindowState == FormWindowState.Minimized)
				Hide();
		}

		private void SystemTrayForm_FormClosing(object sender, FormClosingEventArgs e)
		{
			if (e.CloseReason == CloseReason.UserClosing)
			{
				e.Cancel = true;
				Hide();
			}
		}

		private void notifyIcon_DoubleClick(object sender, EventArgs e)
		{
			Show();
			WindowState = FormWindowState.Normal;
		}


		#endregion

		private void RunLastPcBuild(string exe, string args)
		{
			RunLastPcBuild runLastPcBuild = new RunLastPcBuild(m_settings.PcDestFolder, m_settings.Project, exe, args);
			BuildInfo buildInfo = runLastPcBuild.Find();
			if (buildInfo != null)
			{
				try
				{
					runLastPcBuild.Run(buildInfo);
				}
				catch (Exception execption)
				{
					MessageBox.Show(execption.Message);
				}
			}
			else
			{
				MessageBox.Show("No build available on your drive");
			}
		}

		private void editorToolStripMenuItem_Click(object sender, EventArgs e)
		{
			RunLastPcBuild(m_settings.PcExecFileEditor, m_settings.PcExecFileEditorArguments);
		}

		private void gameToolStripMenuItem_Click(object sender, EventArgs e)
		{
			RunLastPcBuild(m_settings.PcExecFileGame, m_settings.PcExecFileGameArguments);
		}

		private void playLevelToolStripMenuItem_Click(object sender, EventArgs e)
		{
			if (PlayLevelForm.IsRunning())
				return;

			LevelInfo levelInfo = new LevelInfo(m_settings.PcDestFolder, m_settings.Project, m_settings.PcExecFileGame, m_settings.PcExecFileEditorArguments);
			if (levelInfo.Load())
			{
				PlayLevelForm playLevelForm = new PlayLevelForm(levelInfo);
				playLevelForm.Show();
			}
			else
			{
				MessageBox.Show("No build available on your drive");
			}
		}
	}
}