using System;
using System.Collections.Generic;
using System.Text;
using BuildTools;
using NUnit.Framework;

namespace UnitTest
{
	[TestFixture]
	public class BuildInfoListTest
	{
		private Config _config;
		private BuildInfoList _buildInfoList;

		public BuildInfoListTest()
		{
			_config = new Config();
			_buildInfoList = null;
		}

		private void CreateBuildInfoList(Platform platform)
		{
			_buildInfoList = BuildInfoList.Create(_config.Storage, _config.Game, platform);
			_buildInfoList.Update();
		}

		private BuildInfo GetVersion(ICollection<BuildInfo> buildInfoList, int version)
		{
			foreach (BuildInfo buildInfo in buildInfoList)
			{
				if (buildInfo.Number == version)
					return buildInfo;
			}

			return null;
		}

		[SetUp]
		public void SetUp()
		{
			CreateBuildInfoList(Platform.Pc);
		}

		[Test]
		public void IsSorted()
		{
			IEnumerator<BuildInfo> buildInfoListEnumerator = _buildInfoList.GetEnumerator();

			Assert.IsTrue(buildInfoListEnumerator.MoveNext());
			Assert.AreEqual(2, buildInfoListEnumerator.Current.Number);
			Assert.IsTrue(buildInfoListEnumerator.MoveNext());
			Assert.AreEqual(5, buildInfoListEnumerator.Current.Number);
			Assert.IsTrue(buildInfoListEnumerator.MoveNext());
			Assert.AreEqual(6, buildInfoListEnumerator.Current.Number);
			Assert.IsTrue(buildInfoListEnumerator.MoveNext());
			Assert.AreEqual(7, buildInfoListEnumerator.Current.Number);
			Assert.IsTrue(buildInfoListEnumerator.MoveNext());
			Assert.AreEqual(9, buildInfoListEnumerator.Current.Number);
			Assert.IsFalse(buildInfoListEnumerator.MoveNext());
		}

		[Test]
		public void RevertEnumerator()
		{
			IEnumerator<BuildInfo> buildInfoListEnumerator = _buildInfoList.Reverse.GetEnumerator();

			Assert.IsTrue(buildInfoListEnumerator.MoveNext());
			Assert.AreEqual(9, buildInfoListEnumerator.Current.Number);
			Assert.IsTrue(buildInfoListEnumerator.MoveNext());
			Assert.AreEqual(7, buildInfoListEnumerator.Current.Number);
			Assert.IsTrue(buildInfoListEnumerator.MoveNext());
			Assert.AreEqual(6, buildInfoListEnumerator.Current.Number);
			Assert.IsTrue(buildInfoListEnumerator.MoveNext());
			Assert.AreEqual(5, buildInfoListEnumerator.Current.Number);
			Assert.IsTrue(buildInfoListEnumerator.MoveNext());
			Assert.AreEqual(2, buildInfoListEnumerator.Current.Number);
			Assert.IsFalse(buildInfoListEnumerator.MoveNext());
		}

		[Test]
		public void GetVersionsByDateEmpty()
		{
			IList<BuildInfo> versions = _buildInfoList.GetVersions(new DateTime(2008, 11, 5));

			Assert.AreEqual(0, versions.Count);
		}

		[Test]
		public void GetVersionsByDate()
		{
			IList<BuildInfo> versions = _buildInfoList.GetVersions(new DateTime(2008, 11, 8));

			Assert.AreEqual(4, versions.Count);
			Assert.IsNotNull(GetVersion(versions, 2));
			Assert.IsNotNull(GetVersion(versions, 5));
			Assert.IsNotNull(GetVersion(versions, 6));
			Assert.IsNotNull(GetVersion(versions, 7));
		}

		[Test]
		public void GetVersionsByDateXbox360()
		{
			CreateBuildInfoList(Platform.Xbox360);
			IList<BuildInfo> versions = _buildInfoList.GetVersions(new DateTime(2008, 11, 7));

			Assert.AreEqual(1, versions.Count);
			Assert.IsNotNull(GetVersion(versions, 3));
		}

		[Test]
		public void GetVersionsByDatePS3()
		{
			CreateBuildInfoList(Platform.PS3);
			IList<BuildInfo> versions = _buildInfoList.GetVersions(new DateTime(2008, 11, 7));

			Assert.AreEqual(1, versions.Count);
			Assert.IsNotNull(GetVersion(versions, 1));
		}

		[Test]
		public void GetFirstByDateVersion()
		{
			BuildInfo version = _buildInfoList.GetFirstVersion(new DateTime(2008, 11, 8));

			Assert.AreEqual(2, version.Number);
		}

		[Test]
		public void GetFirstVersionByDateNone()
		{
			BuildInfo version = _buildInfoList.GetFirstVersion(new DateTime(2008, 11, 1));

			Assert.IsNull(version);
		}

		[Test]
		public void GetLastVersionByDate()
		{
			BuildInfo version = _buildInfoList.GetLastVersion(new DateTime(2008, 11, 8));

			Assert.AreEqual(7, version.Number);
		}

		[Test]
		public void GetLastVersionByDateNone()
		{
			BuildInfo version = _buildInfoList.GetLastVersion(new DateTime(2008, 11, 1));

			Assert.IsNull(version);
		}

		[Test]
		public void GetLastVersions()
		{
			BuildInfo version = _buildInfoList.GetLastVersion(new DateTime(2008, 11, 8));

			Assert.AreEqual(7, version.Number);
		}

		[Test]
		public void GetLastVersion()
		{
			BuildInfo version = _buildInfoList.GetLastVersion();

			Assert.AreEqual(9, version.Number);
		}

		[Test]
		public void PlatformFolder()
		{
			Assert.AreEqual(FileUtils.CombinePath(_config.Storage, _config.Game, Platform.Pc.ToString()), _buildInfoList.PlatformFolder);
		}

	}
}
