using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Diagnostics;
using System.IO;
using System.Net.Mail;
using System.Text;
using System.Text.RegularExpressions;
using System.Threading;
using System.Windows.Forms;
using System.Management;

// http://confluence.atlassian.com/display/JIRA/Creating+a+SOAP+Client

// JiraSoapService.cs was created automatically by installing the .NET SDK,
// and then passing http://jira/rpc/soap/jirasoapservice-v2?wsdl into
// C:\Program Files\Microsoft.NET\SDK\v2.0 64bit\Bin\wsdl.exe /appsettingurlkey:JiraServer /appsettingbaseurl:http://jira/ <address>

// Similarly, CrashHandlerSoapService.cs was created automatically from
// http://jira/rpc/soap/crashhandlersoapservice?wsdl

namespace CrashHandler
{
    public class CrashHandler
    {
        #region Private Fields

        CrashProgressListener m_crashProgressListener;
        BackgroundWorker m_worker;
        protected readonly ExceptionHandler m_exceptionHandler;

        protected readonly Settings m_settings;
        protected readonly CrashInfo m_crashInfo;
        protected readonly Log m_log;
        string m_reporterWindowsAccount;
        JiraConnector m_jiraConnector;

        const String buildFolderArgument = "-buildFolder";
        const String logFileNameArgument = "-logFileName";
        public const String errorLogNameArgument = "-errorLog";

        #endregion

        public CrashHandler(String buildFolder, String gameLogFilename, String errorLogName)
        {
            String errorLog = errorLogName;
            if (String.IsNullOrEmpty(errorLog))
            {
                errorLog = "error.log";
            }

            try
            {
                m_settings = new Settings(errorLog);
            }
            catch (Exception exception)
            {
                ExceptionHandler.StartupException("CrashHandler encountered an exception loading settings: ", exception);
                Environment.Exit(1);
            }


            m_log = new Log(buildFolder, m_settings);
            m_crashInfo = new CrashInfo(buildFolder, gameLogFilename, m_settings, m_log);
            m_exceptionHandler = new ExceptionHandler(m_log, m_crashInfo, m_settings);
        }

        static void Main(string[] args)
        {
            
            String buildFolder = null;
            String logFileName = null;
            String errorLogName = null; 
            try
            {
                IDictionary<String, String> arguments = Util.GetNamedArgumentsWithProperties(args);
                buildFolder = arguments[buildFolderArgument];
                logFileName = arguments[logFileNameArgument];
                errorLogName = arguments.ContainsKey(errorLogNameArgument)? arguments[errorLogNameArgument] : String.Empty;
            }
            catch (Exception)
            {
                ShowHelpAndExit();
            }
           if(IsApplicationRunning(buildFolder)) {
               MessageBox.Show("The crash handler is already running for this crash");
               Environment.Exit(0);
           }

            CrashHandler crashHandler = new CrashHandler(buildFolder, logFileName, errorLogName);
            crashHandler.HandleGameCrash();

            Environment.Exit(0);
        }

        static bool IsApplicationRunning(String buildFolder)
        {
            String processName = Process.GetCurrentProcess().ProcessName;
            processName = Path.GetFileNameWithoutExtension(processName);

            //http://stackoverflow.com/questions/504208/how-to-read-command-line-arguments-of-another-process-in-c
            string wmiQuery = string.Format("select CommandLine from Win32_Process where Name like '{0}.%'", processName);
            ManagementObjectSearcher searcher = new ManagementObjectSearcher(wmiQuery);
            ManagementObjectCollection retObjectCollection = searcher.Get();

            int count = 0;
            foreach (ManagementObject retObject in retObjectCollection)
            {
                bool isSameFolder = CompareProcessFolders(buildFolder, retObject);
                if(isSameFolder){
                    count++;
                }

            }
            return count > 1;
        }

        static bool CompareProcessFolders(String buildFolder, ManagementObject otherProcess){
            String quotedPath = "\".*?\"";
            String unquotedPath = "[^\\s]*";
            String commandLineArgument = "(" + quotedPath + "|" + unquotedPath + ")";

            Match match = Regex.Match(otherProcess["CommandLine"].ToString(), buildFolderArgument + "=" + commandLineArgument);
            if(!match.Success){
                return false;
            }
            String otherProcessFolder = match.Groups[1].Value.Replace("\"", "");
            return otherProcessFolder.ToLower().Equals(buildFolder.ToLower());
        }

        private static void ShowHelpAndExit()
        {
            MessageBox.Show("Usage: JiraCrashHandler.exe [arguments]" + Environment.NewLine + Environment.NewLine +
                            "Arguments are the following:" + Environment.NewLine + Environment.NewLine +
                            buildFolderArgument + "=[build folder]: Define the build folder" + Environment.NewLine + 
                            logFileNameArgument + "=[log file name]: Define the logfile this application will use"+ Environment.NewLine +
                            errorLogNameArgument + "=[error log file name]: Define the error.log name the application will use");
            Environment.Exit(0);
        }

        protected bool LoadAttachment(string folder, string fileName)
        {
            string filePath = Path.Combine(folder, fileName);
            if (!File.Exists(filePath))
            {
                m_log.Info("Couldn't find file " + fileName);
                return false;
            }

            if (fileName.EndsWith(".bmp"))
            {
                m_log.Info("Converting " + fileName + " to jpg");
                try
                {
                    fileName = Util.ConvertToJpg(folder, fileName, m_settings.JpgCompressionQuality);
                    return LoadAttachment(folder, fileName);
                }
                catch (System.Exception exception)
                {
                    m_log.Info("Couldn't convert to jpg: " + exception.Message);
                    return false;
                }
            }

            long fileSizeInKB = GetFileSizeInKB(filePath);
            if(fileSizeInKB >= m_settings.MinSizeInKBToZipAttachments)
            {
                m_log.Info("Attachment " + fileName + " is larger than "
                    + m_settings.MinSizeInKBToZipAttachments + "KB. Compressing...");
                fileName = Util.CreateCompressedFile(m_crashInfo.BuildFolder, fileName);
            }

            fileSizeInKB = GetFileSizeInKB(Path.Combine(m_crashInfo.BuildFolder, fileName));
            if (fileSizeInKB >= m_settings.MaximumAttachmentSizeInKb)
            {
                m_log.Info("Attachment" + fileName + " has been excluded, even compressed is larger than " + m_settings.MaximumAttachmentSizeInKb);
                return false;
            }
            
            m_crashInfo.LoadAttachment(fileName);

            return true;
        }

        private static long GetFileSizeInKB(string filePath)
        {
            FileInfo fileInfo = new FileInfo(filePath);
            long fileSizeInKB = fileInfo.Length / 1024L;
            return fileSizeInKB;
        }

        private void ReadCrashInfo()
        {
            m_log.Info("Loading crash info...");
            RetrieveLogFiles();
            m_reporterWindowsAccount = Environment.UserName;
            m_log.Info("Crash reporter: " + m_reporterWindowsAccount);
            m_log.Info("Machine name: " + System.Environment.MachineName);

            try
            {
                m_log.Info("Opening attachments...");
                DataExtractor extractor = new DataExtractor(m_crashInfo, m_exceptionHandler, m_log);
                if(LoadAttachment(m_crashInfo.BuildFolder, m_crashInfo.ErrorLogFile))
                {
                    extractor.ProcessErrorLog();
                    ShowProgressToUser();
                }
                else
                {
                    m_exceptionHandler.HandleFailedSubmit("Couldn't find " + m_crashInfo.ErrorLogFile);
                }

                if (LoadAttachment(m_crashInfo.BuildFolder, m_crashInfo.GameLogFile))
                {
                    extractor.ProcessGameLog(); ;
                }

                foreach(string attachment in m_settings.ExtraAttachments)
                {
                    LoadAttachment(m_crashInfo.BuildFolder, attachment);
                }
                RetrievePlatformAttachments();
                 
            }
            catch (Exception exception)
            {
                m_exceptionHandler.HandleFailedSubmit("Couldn't extract crash info: " + exception.ToString());
            }
        }

        protected virtual void RetrievePlatformAttachments(){
            LoadDirectXDiagInfo();
        }

        protected virtual void RetrieveLogFiles(){
            //used by subclasses ie. load the error.log and Game.log
        }

        private void ShowProgressToUser()
        {
            m_worker.ReportProgress(1);
        }


        #region Finding child processes
        Process FindChildProcess(int parentProcessId)
        {
            foreach (Process childProcess in Process.GetProcesses())
            {
                try
                {
                    if (GetParentProcessId(childProcess.ProcessName) == parentProcessId)
                    {
                        return childProcess;
                    }
                }
                catch (Exception exception)
                {
                    m_log.Info("Exception finding child process: " + exception.Message);
                }
            }

            return null;
        }

        // http://stackoverflow.com/questions/394816/how-to-get-parent-process-in-net-in-managed-way
        static int GetParentProcessId(string processName)
        {
            PerformanceCounter performanceCounter = new PerformanceCounter("Process",
                "Creating Process ID", processName);
            return (int)performanceCounter.NextValue();
        }

        #endregion

        void MoveFileAndOverwrite(string sourceFile, string destinationFile)
        {
            if (File.Exists(destinationFile))
            {
                File.Delete(destinationFile);
            }

            File.Move(sourceFile, destinationFile);
        }

        void WaitForUrgentProcess(Process process)
        {
            process.PriorityClass = ProcessPriorityClass.High;
            process.WaitForExit();
        }

        private void RunDxDiag(string outputFolder, string outputFile)
        {
            // arguments: enable WHQL digital signature checks, and silently output to a text file
            // see dxdiag /?
            ProcessStartInfo processStartInfo = new ProcessStartInfo("dxdiag",
                "/whql:on /t " + outputFolder + outputFile);
            processStartInfo.UseShellExecute = false;
            Process dxDiag = Process.Start(processStartInfo);
            int processId = dxDiag.Id;
            WaitForUrgentProcess(dxDiag);

            // dxDiag might spawn a new child process and exit, so we have wait for that child
            dxDiag = FindChildProcess(processId);
            if (dxDiag != null)
            {
                WaitForUrgentProcess(dxDiag);
            }
        }

        private void LoadDirectXDiagInfo()
        {
            try
            {
                m_log.Info("Running DirectX Diagnostic Tool...");
                // I can't get dxDiag to output to any folder other than C:/ or E:/
                // As a workaround, write to C:/ (since that will hopefully always exist),
                // then copy the result to the build folder
                string outputFolder = "C:\\";
                string outputFile = "dxdiag.txt";

                RunDxDiag(outputFolder, outputFile);

                string sourceFile = outputFolder + outputFile;
                string destinationFile = m_crashInfo.BuildFolder + outputFile;
                MoveFileAndOverwrite(sourceFile, destinationFile);
                LoadAttachment(m_crashInfo.BuildFolder, outputFile);

                m_log.Info("DirectX Diagnostic Tool Finished.");
            }
            catch (Exception exception)
            {
                m_log.Info("Error running DirectX Diagnostic Tool: " + exception.Message);
            }
        }


        private void PrepareCrash(object sender, DoWorkEventArgs eventArgs)
        {
            ReadCrashInfo();
        }

        private void OnCrashPrepared(object sender, RunWorkerCompletedEventArgs eventArgs)
        {
            m_crashInfo.IsFullyLoaded = true;
            if(m_crashProgressListener != null)
            {
                m_crashProgressListener.NotifyCrashLoaded();
            }
        }

        private void OnCrashProgress(object sender, ProgressChangedEventArgs eventArgs)
        {
            if (m_crashProgressListener != null)
            {
                m_crashProgressListener.DisplayCallstack();
            }
        }

        private void CreateBackgroundWorker()
        {
            m_worker = new BackgroundWorker();
            m_worker.WorkerReportsProgress = true;
            m_worker.DoWork
                += new DoWorkEventHandler(PrepareCrash);
            m_worker.RunWorkerCompleted
                += new RunWorkerCompletedEventHandler(OnCrashPrepared);
            m_worker.ProgressChanged
                += new ProgressChangedEventHandler(OnCrashProgress);

            m_worker.RunWorkerAsync();
        }

        public void HandleGameCrash()
        {
            UI.Common window = new UI.Common();
            m_log.SetListener(window);
            UI.EditCrashDetail editCrashDetail = new UI.EditCrashDetail(this, window);
            m_crashProgressListener = editCrashDetail;

            CreateBackgroundWorker();
            Application.Run(window);
        }

        public virtual JiraIssueData ReportIssue()
        {
            m_jiraConnector = new JiraConnector(m_log, m_exceptionHandler, m_settings, m_crashInfo, m_reporterWindowsAccount);
            return m_jiraConnector.ReportIssue();
            
        }

        public CrashInfo crashInfo
        {
            get { return m_crashInfo; }
        }
    }
}
