using System;
using System.Collections.Generic;
using System.Text;
using System.Text.RegularExpressions;
using System.IO;

namespace CrashHandler
{
    public class CrashInfo
    {
        public enum IssueAction { Create, Comment, Reopen, Ignore, FixedInNewerBuild };

        public const string CallStackHeader = "Call Stack Trace:";

        public CrashInfo(String buildFolder, String logFileName, Settings setting, Log log)
        {
            buildFolder += Path.DirectorySeparatorChar;

            m_attachmentNames = new List<string>();
            m_attachmentData = new List<string>();
            m_buildFolder = buildFolder;
            m_errorLogFile = setting.ErrorLogName;
            m_platform = setting.PlatformPC;
            m_gameLogFile = logFileName;
            m_isEditorIssue = logFileName.StartsWith("Editor");
            m_buildName = "<no info>";
            m_gameMode = "<no info>";

            m_log = log;
            m_fullyLoaded = false;
        }

        public CrashInfo(Log log)
        {
            m_attachmentNames = new List<string>();
            m_attachmentData = new List<string>();

            m_log = log;
        }

        public void LoadAttachment(string attachmentName)
        {
            m_log.Info("Loading attachment " + attachmentName);

            byte[] rawAttachmentData = Util.GetFileData(BuildFolder + attachmentName);

            m_attachmentNames.Add(attachmentName);
            m_attachmentData.Add(System.Convert.ToBase64String(rawAttachmentData));
        }

        public bool IsLineAtEndOfStack(string line)
        {
            if (line == "__tmainCRTStartup")
                return true;

            if (line == "CrySimpleThread<CryRunnable>::RunThis")
                return true;

            return false;
        }

        private static string GetStrippedCallStackEntry(string callStackLine)
        {
            string strippedLine = callStackLine.Trim();

            // strip out stack line number
            int index = strippedLine.IndexOf(')');
            if (index >= 0)
            {
                strippedLine = strippedLine.Substring(index + 1).Trim();
            }

            // ignore functions with no filename or line number, since they're not from our code
            Match match = Regex.Match(strippedLine, ".+\\(\\) *\\[.+\\:\\d+\\]");
            if(!match.Success)
            {
                strippedLine = "";
            }

            // strip out filename & line number and everything past the last brackets
            index = strippedLine.LastIndexOf("()");
            if(index >= 0)
            {
                strippedLine = strippedLine.Substring(0, index).Trim();
            }

            return strippedLine.Trim();
        }

        private string FirstUsefulLine
        {
            get
            {
                foreach(string line in UsefulCallStack)
                {
                    // return the first useful line (which may not exist)
                    return line;
                }

                return CallStackTrace[0];
            }
        }

        public string Summary
        {
            get
            {
                string summary = IsEditorIssue ? "EDITOR: " : "";
                summary += "CRASH: " + FirstUsefulLine;
                if(summary.Length > 100){
                    summary = summary.Substring(0, 95);
                    summary += "[...]";
                }
                return summary;
            }
        }

        public string[] RawAttachmentNames
        {
            get { return m_attachmentNames.ToArray(); }
        }

        public string[] AttachmentNames
        {
            get
            {
                string[] modifiedAttachmentNames = m_attachmentNames.ToArray();
                for (int i = 0; i < modifiedAttachmentNames.Length; ++i)
                {
                    modifiedAttachmentNames[i] = Platform + " Build " + ProductVersion.ToString()
                        + " (Crash " + String.Format("{0:00}", DuplicateNumber) + ") - " + m_attachmentNames[i];
                }
                return modifiedAttachmentNames;
            }
        }

        public string[] AttachmentData
        {
            get { return m_attachmentData.ToArray(); }
        }

        public string BuildFolder
        {
            get { return m_buildFolder; }
        }

        public string ErrorLogFile
        {
            get { return m_errorLogFile; }
        }

        public string ErrorText
        {
            set{ m_errorText = value;}
            get { return m_errorText; }
        }

        public string BuildName
        {
            set{m_buildName = value;}
            get { return m_buildName; }
        }
        public string GameLogFile
        {
            get { return m_gameLogFile; }
        }

        public string ProductVersion
        {
            get { return m_productVersion; }
            set { m_productVersion = value; }
        }

        public DateTime BuildTime
        {
            get { return m_buildTime; }
            set { m_buildTime = value; }
        }

        public String LastLevelLoaded
        {
            get { return m_lastLevelLoaded; }
            set { m_lastLevelLoaded = value;}
        }

        public int DuplicateNumber
        {
            get { return m_duplicateNumber; }
            set { m_duplicateNumber = value; }
        }

        public IssueAction IssueActionTaken
        {
            get { return m_issueAction; }
            set { m_issueAction = value; }
        }

        public String GameMode{
            get { return m_gameMode;  }
            set { m_gameMode = value;}
        }

        public void SetIssueActionFromString(String value){
            if(value != null && Enum.IsDefined(typeof(IssueAction), value)){
                m_issueAction = (IssueAction) Enum.Parse(typeof(IssueAction), value);
            }
        }

        public string Description
        {
            get { return m_description; }
            set
            {
                m_description = value;
                // clear out full description, since it's now invalid
                m_fullDescription = null;
            }
        }

        private string GetOperatingSystemVersionName()
        {
            // http://stackoverflow.com/questions/860459/determine-os-using-the-environment-osversion-object-c/860489
            if (Environment.OSVersion.Version.Major == 5)
            {
                if (Environment.OSVersion.Version.Minor == 1)
                {
                    return "Windows XP";
                }
                else if (Environment.OSVersion.Version.Minor == 2)
                {
                    return "Windows XP 64-bit/Windows 2003";
                }
            }
            else if (Environment.OSVersion.Version.Major == 6)
            {
                if (Environment.OSVersion.Version.Minor == 0)
                {
                    return "Windows Vista";
                }
                else if (Environment.OSVersion.Version.Minor == 1)
                {
                    // windows 7 is version 6.1: http://windowsteamblog.com/blogs/windowsvista/archive/2008/10/14/why-7.aspx
                    return "Windows 7 or higher";
                }
            }

            return "Unknown windows version";

        }

        public string OperatingSystemInfo
        {
            get
            {
                if (m_operatingSystemInfo == null)
                {
                    m_operatingSystemInfo = Environment.NewLine + Environment.NewLine + "Operating system: " +GetOperatingSystemVersionName()
                        +Environment.NewLine + Environment.OSVersion;
                }
                return m_operatingSystemInfo;
            }
            set
            {
                m_operatingSystemInfo = value;
            }
        }

        public string FullDescription
        {
            get
            {
                if (m_fullDescription == null && CallStackTrace != null)
                {
                    m_fullDescription = CallStackHeader + Environment.NewLine;
                    m_fullDescription += CallStackTraceAsString;

                    m_fullDescription +=  OperatingSystemInfo;
                    m_fullDescription += LastLevelLoadedText();
                    m_fullDescription += GameModeText();

                    m_fullDescription += Environment.NewLine + Environment.NewLine + Description;
                }

                return m_fullDescription;
            }
        }

        private String LastLevelLoadedText(){
            if(LastLevelLoaded == null || LastLevelLoaded == String.Empty)
            {
                return String.Empty;
            }
            return Environment.NewLine + Environment.NewLine + "Last level loaded: " + LastLevelLoaded;
        }

        private String GameModeText(){
            if(GameMode == null || GameMode == String.Empty)
            {
                return String.Empty;
            }
            return Environment.NewLine + Environment.NewLine + "Game mode: " + GameMode;
        }

        public List<string> CallStackTrace
        {
            get { return m_callStackTrace; }
            set
            {
                m_callStackTrace = value;
            }
        }

        public string CallStackTraceAsString
        {
            get
            {
                string result = "";
                if (CallStackTrace != null)
                {
                    foreach (string line in CallStackTrace)
                    {
                        result += line + Environment.NewLine;
                    }
                }

                return result;
            }
        }

        private IEnumerable<string> UsefulCallStack
        {
            get
            {
                string lastLine = "";
                foreach (string rawLine in CallStackTrace)
                {
                    string line = GetStrippedCallStackEntry(rawLine);

                    // ignore recursion
                    if (line != lastLine)
                    {
                        yield return line;
                    }
                    lastLine = line;

                    if (IsLineAtEndOfStack(line))
                    {
                        break;
                    }
                }
            }
        }

        public bool IsEditorIssue
        {
            get { return m_isEditorIssue; }
        }

        public bool IsFullyLoaded
        {
            get { return m_fullyLoaded; }
            set { m_fullyLoaded = value; }
        }

        public String Platform
        {
            get { return m_platform; }
            set { this.m_platform = value;}
        }

        #region Private Members
        private List<string> m_attachmentNames;
        private List<string> m_attachmentData;

        private string m_buildFolder;
        private DateTime m_buildTime;
        private String m_lastLevelLoaded;

        private string m_errorLogFile;
        private string m_gameLogFile;
        private string m_buildName;
        private bool m_isEditorIssue;
        
        private string m_productVersion;
        private int m_duplicateNumber;
        private IssueAction m_issueAction;
        private string m_description;
        private string m_fullDescription;

        private List<string> m_callStackTrace;
        private String m_gameMode;

        private Log m_log;

        private String m_operatingSystemInfo;

        private bool m_fullyLoaded;
        private String m_errorText;
        private String m_platform;
        #endregion
    }
}
