﻿using System;
using System.Collections.Generic;
using System.Text;
using System.Text.RegularExpressions;
using System.IO;
using System.Net;


namespace CrashHandler
{
    public class DataExtractor
    {
        private CrashInfo m_crashInfo;
        private ExceptionHandler m_exceptionHandler;
        private Log m_log;
        
        public DataExtractor(CrashInfo crashInfo, ExceptionHandler exceptionHandler, Log log){
            this.m_crashInfo = crashInfo;
            this.m_exceptionHandler = exceptionHandler;
            this.m_log = log;
        }

        public void ProcessErrorLog()
        {
            String errorLogPath = m_crashInfo.BuildFolder + m_crashInfo.ErrorLogFile;
            
            String errorLogText = Util.GetFileText(errorLogPath);
            m_crashInfo.ProductVersion = GetProductVersion(errorLogText);
            m_crashInfo.CallStackTrace = GetCallStackTrace(errorLogText);
            m_crashInfo.ErrorText = errorLogText;
        }

        public string GetProductVersion(string errorText)
        {
            String result = String.Empty;
            try
            {
                result = GetBuildNumber(errorText);
                
            } catch (Exception){
                m_exceptionHandler.HandleFailedSubmit("Couldn't find product version in error log! CrashHandler needs to be updated.");
            }
            return result;
           
        }
        public static String GetBuildNumber(String errorText){
            Match match = Regex.Match(errorText, @"ProductVersion: (\d+)\.(\d+)\.(\d+)\.(\d+)");
            return match.Groups[4].Value;
        }

        private List<String> GetCallStackTrace(string errorText)
        {
            List<String> callStackTrace = new List<String>();

            string callStackHeader = CrashInfo.CallStackHeader;
            int callStackLocation = errorText.IndexOf(callStackHeader);
            if (callStackLocation < 0)
            {
                m_exceptionHandler.HandleFailedSubmit("Couldn't find a call stack in error.log! CrashHandler needs to be updated.");
            }

            errorText = errorText.Substring(callStackLocation + callStackHeader.Length);

            int callStackEnd = errorText.IndexOf("Suspended thread (");
            if (callStackEnd >= 0)
            {
                errorText = errorText.Substring(0, callStackEnd);
            }

            foreach (string line in errorText.Split(Environment.NewLine.ToCharArray(), StringSplitOptions.RemoveEmptyEntries))
            {
                callStackTrace.Add(line);
            }
            return callStackTrace;
        }

        public void ProcessGameLog()
        {
            String gameLogPath = m_crashInfo.BuildFolder+ m_crashInfo.GameLogFile;

            String gameLogText = Util.GetFileText(gameLogPath);
            m_crashInfo.BuildName = GetBuildName(gameLogText);
            m_crashInfo.LastLevelLoaded = GetLevelName(gameLogText);
            m_crashInfo.BuildTime = GetBuildTime(gameLogText);
            m_crashInfo.GameMode = GetGameMode(gameLogText);
        }

        public String GetLevelName(String logText){
            String levelName;
            if (m_crashInfo.IsEditorIssue)
            {
                levelName = FindLast(logText, @"Opening document (.*)");
            }
            else
            {
                levelName = FindLast(logText, @"\*LOADING: Loading Level (.*)");
            }
            return levelName;
        }

        public String GetBuildName(String editorText){
            String buildName =  String.Empty;
            Match match = Regex.Match(editorText, @"Executable: (.*)");
            if(match.Success){
                buildName = match.Groups[1].Value;
            }
            return buildName;
        }

        public String FindLast(String text, String pattern){
            Match prevMatch = FindLastMatch(text, pattern);

            if (!prevMatch.Success)
            {
                return String.Empty;
            }

            return prevMatch.Groups[1].Value.ToString();
            
        }

        private Match FindLastMatch(String text, String pattern)
        {
            Match match = Regex.Match(text, pattern);

            Match prevMatch = match;
            while (match.Success)
            {
                prevMatch = match;
                match = match.NextMatch();
            }
            return prevMatch;
        }

        private DateTime GetBuildTime(string logText)
        {
            Match match = Regex.Match(logText, @"BuildTime: (.+)");
            if (match.Success)
            {
                string dateString = match.Groups[1].Value.Trim();
                try
                {
                    return DateTime.Parse(dateString);
                }
                catch (System.Exception exception)
                {
                    m_log.Info("Error parsing build date: " + exception.Message);
                    // fall through
                }
            }

            // If we can't find a build number, assume it's a VERY recent build
            return DateTime.Now;
        }

        private String GetGameMode(String longText)
        {
            String gameMode = FindLast(longText, @"Starting in (.*) mode");
            if (gameMode.ToLower().Contains("client"))
            {
                gameMode += Environment.NewLine;
                gameMode += "Connected to server: ";
                gameMode += GetServerName(longText);
            }
            return gameMode;
        }

        private String GetServerName(String longText)
        {
            String address = FindLast(longText, @"ServerName: (.*)").Trim();
            if (address == String.Empty)
            {
                return "<no info>";
            } 
            else
            {
                //then try to get the dns name out of it..
                String name = Util.ReverseDNS(address, 1);
                if (name != address)
                {
                    address = String.Format("{0} ({1})", name, address);
                }
            }
            return address;
        }


        
    }
}
