﻿using System;
using System.Collections.Generic;
using System.Text;
using System.Net.Mail;
using System.IO;

namespace CrashHandler
{
    public class ExceptionHandler
    {
        Settings m_settings;
        Log m_log;
        CrashInfo m_crashInfo;
        const string s_failedCrashFile = "crash.txt";
        const string s_defaultFailedCrashFolder = "_Crash";

        public ExceptionHandler(Log log, CrashInfo crashInfo, Settings settings){
            m_log = log;
            m_crashInfo = crashInfo;
            m_settings = settings;
        }

        public void HandleFailedSubmit(string reason, Exception ex){
            StringBuilder completeMessage = new StringBuilder(reason);
            completeMessage.Append(Environment.NewLine);
            completeMessage.Append("Exception message:");
            completeMessage.Append(ex.Message);
            completeMessage.Append(Environment.NewLine);
            completeMessage.Append(Environment.NewLine);
            completeMessage.Append("Call stack:");
            completeMessage.Append(ex.StackTrace);
            HandleFailedSubmit(completeMessage.ToString());

        }
        public void HandleFailedSubmit(string reason)
        {
            
            try
            {
                m_log.Info("Error encountered: " + reason);

                if (SendErrorEmail(reason))
                {
                    m_log.Popup("Error creating jira bug! " + m_settings.ProgramMaintainerName
                        + " has been emailed, and he's on the case.");
                }
                else
                {
                    BackupCrashFiles(reason);

                }
            }
            catch (Exception exception)
            {
                m_log.Popup("Crash handler's crash handler crashed! " + exception.ToString());
                if(m_settings != null){
                    m_log.Popup("This is very bad. Please contact " + m_settings.ProgramMaintainerName + " at " + m_settings.ProgramMaintainerEmail);
                }
            }

            Environment.Exit(0);
        }

        private void BackupCrashFiles(string reason)
        {
            m_log.Info("Copying crash files...");
            string crashFolder = CreateCrashFolder();
            CopyCrashFiles(crashFolder);
            WriteErrorLog(reason, crashFolder);

            m_log.Popup("Error creating jira bug! See " + crashFolder + s_failedCrashFile + " for details.");
            m_log.Popup("Please send the contents of " + crashFolder + " to " + m_settings.ProgramMaintainerName + " at " + m_settings.ProgramMaintainerEmail);
        }

        private bool SendErrorEmail(string reason)
        {
            try
            {
                m_log.Info("Sending email to program maintainer " + m_settings.ProgramMaintainerName
                    + " <" + m_settings.ProgramMaintainerEmail + ">...");

                MailAddress emailFrom = new MailAddress(m_settings.EmailSenderEmailAddress, m_settings.EmailSenderName);
                MailMessage errorMail = new MailMessage();
                String userName = Environment.UserName;

                AddRecipientsToEmail(errorMail, m_settings.ProgramMaintainerEmail);
                errorMail.From = emailFrom;
                errorMail.Subject = "[" + userName+ "] Auto-email: Crash handler encountered an error :(";
                errorMail.Body = "This crash happened to: " + userName+ Environment.NewLine + "Using the build:" + m_crashInfo.BuildName + Environment.NewLine +
                    "Couldn't submit crash for build number " + m_crashInfo.ProductVersion
                    + Environment.NewLine + Environment.NewLine
                    + reason
                    + Environment.NewLine + Environment.NewLine
                    + m_crashInfo.FullDescription;

                foreach (string fileName in m_crashInfo.RawAttachmentNames)
                {
                    errorMail.Attachments.Add(new Attachment(m_crashInfo.BuildFolder + fileName));
                }

                errorMail.Attachments.Add(new Attachment(m_crashInfo.BuildFolder + m_settings.LogFileName));

                SmtpClient smtpClient = new SmtpClient(m_settings.SmtpHost);
                smtpClient.Send(errorMail);
                return true;
            }
            catch (Exception exception)
            {
                m_log.Info("Couldn't send email to program maintainer: " + exception.Message);
                return false;
            }
        }

        private void AddRecipientsToEmail(MailMessage message, string[] emails){
            foreach (String email in emails)
            {
                message.To.Add(email);
            }
        }


        private string CreateCrashFolder()
        {
            // find a unique crash folder

            string defaultCrashFolder = m_crashInfo.BuildFolder + s_defaultFailedCrashFolder;
            string crashFolder = defaultCrashFolder;
            for (int i = 1; true; ++i)
            {
                if (!Directory.Exists(crashFolder))
                {
                    Directory.CreateDirectory(crashFolder);
                    return crashFolder + Path.DirectorySeparatorChar;
                }

                crashFolder = defaultCrashFolder + i.ToString();
            }
        }

        private void CopyCrashFiles(string crashFolder)
        {
            foreach (string file in m_crashInfo.RawAttachmentNames)
            {
                string sourceFile = m_crashInfo.BuildFolder + file;
                string destFile = crashFolder + file;
                CopyFile(sourceFile, destFile);
            }

            CopyFile(m_log.LogPath, crashFolder + m_settings.LogFileName);
        }

        private void WriteErrorLog(string reason, string crashFolder)
        {
            StreamWriter outputFile = new StreamWriter(crashFolder + s_failedCrashFile);
            outputFile.WriteLine(reason);
            outputFile.WriteLine();
            if (m_crashInfo != null)
            {
                outputFile.WriteLine("Couldn't submit crash for build number " + m_crashInfo.ProductVersion);
                outputFile.WriteLine("With log " + m_crashInfo.GameLogFile);
                outputFile.WriteLine();
                outputFile.WriteLine();
                outputFile.WriteLine(m_crashInfo.FullDescription);
            }
            outputFile.Close();
        }


        private void CopyFile(string sourceFile, string destFile)
        {
            try
            {
                File.Copy(sourceFile, destFile, true);
            }
            catch (Exception e)
            {
                m_log.Info("Couldn't copy file " + sourceFile + " to " + destFile + ": " + e.ToString());
            }
        }

        public static void StartupException(String message, Exception e)
        {
            System.Windows.Forms.MessageBox.Show(message + Environment.NewLine + Environment.NewLine + e.Message);
        }
    }
}
