﻿using System;
using System.Collections.Generic;
using System.Text;

namespace CrashHandler
{
    public class JiraConnector
    {
        Log m_log;
        Settings m_settings;
        CrashHandlerSoapService m_crashHandlerSoapService;
        ExceptionHandler m_exceptionHandler;
        JiraSoapService m_jiraSoapService;
        string m_jiraToken;
        string m_jiraCrashHandlerPluginToken;
        string m_reporterJiraAccount;
        RemoteIssue m_issue;
        CrashInfo m_crashInfo;
        string m_reporterWindowsAccount;
        String m_errors;

        public JiraConnector(Log log, ExceptionHandler exceptionHandler, Settings settings, CrashInfo crashInfo, string reporterWindowsAccount){
            m_errors = "";
            m_log = log;
            m_settings = settings;
            m_crashInfo = crashInfo;
            m_reporterWindowsAccount = reporterWindowsAccount;
            m_exceptionHandler = exceptionHandler;
        }

        public JiraIssueData ReportIssue()
        {
            try
            {
                // Don't access jira/mysql until we need them so our sessions don't timeout
                AccessServers();

                if (!AddIssueToJira())
                    throw new Exception("couldn't add the issue to Jira, an error occurred");

                if (!AddAttachmentsToIssue())
                    throw new Exception("couldn't add attachments on issue " + BugUrl + "on Jira");

                m_log.Info("Issue sucessfully reported as " + m_issue.key);
                return new JiraIssueData(BugUrl, BugStatus);
            }
            catch (Exception exception)
            {
                m_exceptionHandler.HandleFailedSubmit("Problem reporting issue to jira: " + exception.Message);
                return null;
            }
        }

        private void AccessServers()
        {
            AccessJira();
            FindReporterJiraAccount();
        }

        private bool AddIssueToJira()
        {
            try
            {
                m_log.Info("Starting request to jira");

                string issueType = GetIssueType();
                DateTime buildTime = m_crashInfo.BuildTime.ToUniversalTime();
                DateTime requestStartTime = DateTime.Now;
                JiraCrashInfo jiraCrashInfo = m_crashHandlerSoapService.registerCrash(m_jiraCrashHandlerPluginToken, m_settings.Project,
                    m_crashInfo.ErrorText, issueType,
                    m_crashInfo.Summary, m_crashInfo.FullDescription + m_errors,
                    m_reporterJiraAccount, m_settings.Assignee, m_crashInfo.ProductVersion, buildTime, m_crashInfo.Platform);
                
                if (jiraCrashInfo == null)
                {
                    m_exceptionHandler.HandleFailedSubmit("RegisterCrash returned null!");
                    return false;
                }
                m_log.Info("Request ended successfully in: " + (DateTime.Now - requestStartTime).Milliseconds + "ms");
                m_issue = jiraCrashInfo.issue;
                m_crashInfo.DuplicateNumber = jiraCrashInfo.crashCount;
                m_crashInfo.SetIssueActionFromString(jiraCrashInfo.actionPerformed);
                return true;

            }
            catch (Exception exception)
            {
                m_log.Info(exception.Message);
                m_exceptionHandler.HandleFailedSubmit("Couldn't add issue to jira due to exception: " + exception.ToString());
                return false;
            }
        }


        private string BugStatus
        {
            get
            {
                switch (m_crashInfo.IssueActionTaken)
                {
                    case CrashInfo.IssueAction.Create:
                        return "This is a new issue!";
                    case CrashInfo.IssueAction.Reopen:
                        return "This is an existing issue that we couldn't reproduce, or we thought was fixed." + Environment.NewLine + "The bug has been reopened.";
                    case CrashInfo.IssueAction.Ignore:
                        return "This issue has been flagged as KS/AD." + Environment.NewLine + "Talk to the project manager if you think this bug should be reopened.";
                    case CrashInfo.IssueAction.FixedInNewerBuild:
                        return "We think this issue has been fixed!" + Environment.NewLine + "Please run the latest build and see if this problem still happens.";
                    case CrashInfo.IssueAction.Comment:
                        String head = m_issue.assignee == null ? "This issue is unassigned. " : "This issue is assigned to " + m_issue.assignee + ".";
                        return head + Environment.NewLine + "Its status is " + GetStatusName() + ".";
                    default:
                        {
                            string errorMessage = "Unknown issue action " + m_crashInfo.IssueActionTaken;
                            m_exceptionHandler.HandleFailedSubmit(errorMessage);
                            return "";
                        }
                }
            }
        }

        private string BugUrl
        {
            get { return m_settings.JiraServer + "browse/" + m_issue.key; }
        }

        private string GetStatusName()
        {
            foreach (RemoteStatus status in m_jiraSoapService.getStatuses(m_jiraToken))
            {
                if (status.id == m_issue.status)
                {
                    return status.name;
                }
            }

            return "(status unknown: status id " + m_issue.status + ")";
        }

        private string GetIssueType()
        {
            RemoteProject project = m_jiraSoapService.getProjectByKey(m_jiraToken, m_settings.Project);
            return GetIssueType(project);
        }


        private string GetIssueType(string desiredIssueType, RemoteProject project)
        {
            RemoteIssueType[] issueTypes = m_jiraSoapService.getIssueTypesForProject(m_jiraToken, project.id);
            foreach (RemoteIssueType issueType in issueTypes)
            {
                if (issueType.name.Equals(desiredIssueType))
                {
                    return issueType.id;
                }
            }

            m_log.Info("Couldn't find issue type " + desiredIssueType);

            return null;
        }

        private string GetIssueType(RemoteProject project)
        {
            if (m_crashInfo.IsEditorIssue)
            {
                string issueType = GetIssueType(m_settings.EditorIssueType, project);
                if (issueType != null)
                {
                    return issueType;
                }
            }

            return GetIssueType(m_settings.NormalIssueType, project);
        }


        private bool AddAttachmentsToIssue()
        {
            if (m_crashInfo.IssueActionTaken == CrashInfo.IssueAction.Ignore)
            {
                return true;
            }
            try
            {
                DateTime requestStartTime = DateTime.Now;
                m_log.Info("Uploading attachments...");

                m_jiraSoapService.addBase64EncodedAttachmentsToIssue(m_jiraToken, m_issue.key, m_crashInfo.AttachmentNames, m_crashInfo.AttachmentData);

                m_log.Info("Finished Uploading attachments. (" + (DateTime.Now - requestStartTime).Milliseconds + "ms)");
                
                return true;
            }
            catch (Exception exception)
            {
                // keep going, since a crash log is still useful without attachments
                m_log.Info("Couldn't add attachments to jira issue " + m_issue.key + ": " + exception.ToString());
                return false;
            }
        }

        private void AccessJira()
        {
            try
            {
                m_log.Info("Accessing Jira...");
                System.Net.ServicePointManager.Expect100Continue = false;

                m_jiraSoapService = new JiraSoapService();
                m_jiraToken = m_jiraSoapService.login(m_settings.Username, m_settings.Password);

                m_crashHandlerSoapService = new CrashHandlerSoapService();
                m_crashHandlerSoapService.Timeout = m_settings.SoapServiceTimeoutInSeconds * 1000;
                m_jiraCrashHandlerPluginToken = m_crashHandlerSoapService.crashHandlerLogin(m_settings.Username, m_settings.Password);
            }
            catch (Exception exception)
            {
                m_exceptionHandler.HandleFailedSubmit("Cannot connect to jira: Check if the crashHandler jira plugin has been properly installed, the jira url needs to be "+m_settings.JiraServer+", and " +m_settings.JiraServer+ "rpc/soap/crashhandlersoapservice?wsdl should show an xml, Error logging to jira "+ Environment.NewLine + exception.ToString());
            }
        }

        private void FindReporterJiraAccount()
        {
            try
            {
                RemoteUser user = m_jiraSoapService.getUser(m_jiraToken, m_reporterWindowsAccount);
                m_reporterJiraAccount = user.name;
            }
            catch (Exception)
            {
                m_errors = Environment.NewLine + Environment.NewLine + "This crash was reported by \"" + m_reporterWindowsAccount +
                    "\", but the crash handler couldn't find a matching jira account :(";

                m_log.Info("Couldn't find jira account for crash reporter!");
            }
        }
    }
}
