using System;
using System.Windows.Forms;

namespace CrashHandler.UI
{
    public partial class EditCrashDetail : UserControl, CrashProgressListener
    {
        CrashHandler m_crashHandler;
        LogListener m_statusListener;
        UI.Common m_mainWindow;
        bool m_submitRequested;

        public EditCrashDetail(CrashHandler crashHandler, UI.Common mainWindow)
        {
            m_mainWindow = mainWindow;
            m_crashHandler = crashHandler;
            m_statusListener = mainWindow;
            InitializeComponent();

            m_mainWindow.SetContents(this);

            ((CrashProgressListener)this).DisplayCallstack();
            PrepopulateDescription();
            CreateToolTip();
        }

        private void ReportIssue()
        {
            m_crashHandler.crashInfo.Description = errorDescription.Text;
            JiraIssueData jiraIssueData = m_crashHandler.ReportIssue();
            if (jiraIssueData != null)
            {
                // Switch to the "bug submitted" window
                BugSubmitted bugSubmitted = new BugSubmitted(jiraIssueData.Url, jiraIssueData.BugStatus);
                m_mainWindow.SetContents(bugSubmitted);
            }
            else
            {
                if(m_statusListener != null)
                {
                    m_statusListener.SetStatus("Bug submission failed!");
                }
                SetSubmitRequested(false);
            }
        }

        public void SetWarningMessage(String message)
        {
            this.warningLabel.Text = message;
        }

        void SetSubmitRequested(bool requested)
        {
            m_submitRequested = requested;
            errorDescription.ReadOnly = requested;
        }

        private void submitButton_Click(object sender, EventArgs e)
        {
            SetSubmitRequested(true);

            UpdateSubmitStatus();
            if(m_crashHandler.crashInfo.IsFullyLoaded) // if not, we'll do this later
            {
                ReportIssue();
            }
        }

        void CrashProgressListener.DisplayCallstack()
        {
            callStack.Text = "Call Stack:" + Environment.NewLine;
            if (m_crashHandler.crashInfo != null && !String.IsNullOrEmpty(m_crashHandler.crashInfo.CallStackTraceAsString))
            {
                callStack.Text += Environment.NewLine + m_crashHandler.crashInfo.CallStackTraceAsString;
            }
            callStack.Refresh();
        }

        private void PrepopulateDescription()
        {
            errorDescription.Text = m_crashHandler.crashInfo.Description;
            UpdateSubmitStatus();
        }

        void CreateToolTip()
        {
            toolTip.SetToolTip(errorDescription,
                "Write what happened before you crashed, like this:" + Environment.NewLine
                + "1) Jump in game" + Environment.NewLine
                + "2) Get hit by a train" + Environment.NewLine
                + "3) Crash! :(");
        }

        private void UpdateSubmitStatus()
        {
            submitButton.Enabled = !m_submitRequested && (errorDescription.Text.Length != 0);
        }

        private void onDescriptionChanged(object sender, EventArgs e)
        {
            UpdateSubmitStatus();
        }

        void CrashProgressListener.NotifyCrashLoaded()
        {
            UpdateSubmitStatus();
            if (m_submitRequested)
            {
                ReportIssue();
            }
            else if(m_statusListener != null)
            {
                m_statusListener.SetStatus(submitButton.Enabled ? "Ready to submit!" : "Please type an error description");
            }
        }
    }
}

