package com.crytek.jira.plugins.crashHandler;

import java.util.ArrayList;
import java.util.Date;

import com.atlassian.core.util.StringUtils;
import com.atlassian.jira.ComponentManager;
import com.atlassian.jira.issue.CustomFieldManager;
import com.atlassian.jira.issue.Issue;
import com.atlassian.jira.issue.fields.CustomField;
import com.atlassian.jira.issue.link.IssueLinkTypeManager;
import com.atlassian.jira.rpc.soap.service.IssueService;
import com.crytek.jira.plugins.crashHandler.signature.AbstractCallStackComparer;
import com.onresolve.jira.crytek.settings.Settings;
import com.opensymphony.user.User;

public class CrashMatcher
{
	// TODO: settings!

	static final String crashSignatureFieldName = "Crash_Signature";

	private final IssueService issueService;

	private final IssueUpdater issueUpdater;

	private final CustomField crashSignatureField;
	private final SignatureInfoBuilder signatureInfoBuilder;
	private final Settings settings;

	public CrashMatcher(IssueService issueService, IssueLinkTypeManager issueLinkTypeManager)
	{
		this.issueService = issueService;
		this.settings = Settings.getSettings();

		CustomFieldManager customFieldManager = ComponentManager.getInstance().getCustomFieldManager();
		crashSignatureField = customFieldManager.getCustomFieldObjectByName(crashSignatureFieldName);

		this.issueUpdater = new IssueUpdater(issueService, issueLinkTypeManager, crashSignatureField, settings);
		signatureInfoBuilder = new SignatureInfoBuilder(issueUpdater);

	}

	private String findFirstIssueInProject(ArrayList<String> issues, String project)
	{
		if (issues == null)
		{
			return null;
		}

		for (String issueKey : issues)
		{
			if (issueKey.startsWith(project))
			{
				return issueKey;
			}
		}

		return null;
	}

	public JiraCrashInfo registerCrash(User user, String project, String errorLogContents, String issueType, String summary, String description,
			String reporter, String assignee, String buildNumber, Date buildTime, String platform) throws Exception
	{
		String bestIssueKey = null;
		SignatureInfo signature = signatureInfoBuilder.fromErrorLog(errorLogContents);
		ArrayList<String> matchingIssuesRaw = new ArrayList<String>();

		if (!StringUtils.contains(errorLogContents, settings.getNewCrashPattern()))
		{
			matchingIssuesRaw = getMatchingIssues(user, signature);
		}

		String[] matchingIssues = new String[] {};
		if (matchingIssuesRaw != null && matchingIssuesRaw.size() > 0)
		{
			bestIssueKey = findFirstIssueInProject(matchingIssuesRaw, project);
			if (bestIssueKey != null)
			{
				matchingIssuesRaw.remove(bestIssueKey);
			}
			matchingIssues = new String[matchingIssuesRaw.size()];
			matchingIssuesRaw.toArray(matchingIssues);
		}

		JiraCrashInfo jiraCrashInfo;

		if (bestIssueKey == null)
		{
			bestIssueKey = issueUpdater.createNewIssue(user, project, signature, issueType, summary, description, reporter, assignee, buildNumber, platform);
			jiraCrashInfo = new JiraCrashInfo(issueService.getIssue(user, bestIssueKey), JiraCrashInfo.Create, 1);
		}
		else
		{
			jiraCrashInfo = issueUpdater.updateExistingIssue(user, bestIssueKey, description, reporter, buildNumber, buildTime, platform);
		}

		issueUpdater.linkIssues(user, bestIssueKey, matchingIssues);

		return jiraCrashInfo;
	}

	public ArrayList<String> getMatchingIssues(User user, SignatureInfo signature) throws Exception
	{
		Issue[] issues = new JiraSearch().getAllCrashes(user);

		double bestIssueScore = 0;
		ArrayList<String> matchingIssues = new ArrayList<String>();
		AbstractCallStackComparer callStackComparer = settings.getComparer().get(signature);

		for (Issue currentIssue : issues)
		{
			SignatureInfo currentIssueSignature = signatureInfoBuilder.fromIssue(currentIssue, crashSignatureField);

			double matchScore = callStackComparer.rateSimilarity(currentIssueSignature);

			if (Utils.lessThan(matchScore, settings.getMatchPercentage()))
			{
				continue;
			}
			else if (Utils.greaterThan(matchScore, bestIssueScore))
			{
				matchingIssues.clear();
				matchingIssues.add(currentIssue.getKey());
				bestIssueScore = matchScore;
			}
			else if (Utils.equals(matchScore, bestIssueScore))
			{
				matchingIssues.add(currentIssue.getKey());
			}
		}

		return matchingIssues;
	}
}
