package com.crytek.jira.plugins.crashHandler;

import java.util.Collection;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Map;

import org.apache.log4j.Logger;
import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;

import com.atlassian.jira.ComponentManager;
import com.atlassian.jira.issue.CustomFieldManager;
import com.atlassian.jira.issue.Issue;
import com.atlassian.jira.issue.IssueManager;
import com.atlassian.jira.issue.IssueUtilsBean;
import com.atlassian.jira.issue.ModifiedValue;
import com.atlassian.jira.issue.MutableIssue;
import com.atlassian.jira.issue.comments.CommentManager;
import com.atlassian.jira.issue.customfields.CustomFieldType;
import com.atlassian.jira.issue.fields.CustomField;
import com.atlassian.jira.issue.link.IssueLinkManager;
import com.atlassian.jira.issue.link.IssueLinkType;
import com.atlassian.jira.issue.link.IssueLinkTypeManager;
import com.atlassian.jira.issue.link.LinkCollection;
import com.atlassian.jira.issue.util.DefaultIssueChangeHolder;
import com.atlassian.jira.issue.util.IssueChangeHolder;
import com.atlassian.jira.rpc.soap.beans.RemoteFieldValue;
import com.atlassian.jira.rpc.soap.beans.RemoteIssue;
import com.atlassian.jira.rpc.soap.service.IssueService;
import com.onresolve.jira.crytek.settings.Settings;
import com.opensymphony.user.User;
import com.opensymphony.workflow.loader.ActionDescriptor;

public class IssueUpdater
{
	private final IssueLinkManager issueLinkManager;
	private final IssueManager issueManager;
	private final IssueService issueService;
	private final CustomFieldManager customFieldManager;

	private Long issueLinkTypeId;
	private final CustomField crashSignatureField;
	private final CustomField crashCountField;
	protected Settings settings;
	private static final Logger log = Logger.getLogger(IssueUpdater.class);

	protected IssueUpdater()
	{
		this.issueLinkManager = null;
		this.issueManager = null;
		this.issueService = null;
		this.customFieldManager = null;
		this.crashSignatureField = null;
		this.crashCountField = null;
		this.settings = null;
	}

	public IssueUpdater(IssueService issueService, IssueLinkTypeManager issueLinkTypeManager, CustomField crashSignatureField, Settings settings)
	{
		this.issueLinkManager = ComponentManager.getInstance().getIssueLinkManager();
		this.issueManager = ComponentManager.getInstance().getIssueManager();
		this.issueService = issueService;
		this.customFieldManager = ComponentManager.getInstance().getCustomFieldManager();
		this.crashCountField = customFieldManager.getCustomFieldObjectByName(settings.getCrashCountCustomFieldName());
		this.settings = settings;

		Collection<IssueLinkType> issueLinkTypes = issueLinkTypeManager.getIssueLinkTypesByName("Duplicate of bug");
		if (issueLinkTypes != null && issueLinkTypes.size() > 0)
		{
			IssueLinkType issueLinkType = issueLinkTypes.iterator().next();
			issueLinkTypeId = issueLinkType.getId();
		}
		this.crashSignatureField = crashSignatureField;
	}

	public String createNewIssue(User user, String project, SignatureInfo callStack, String issueType, String summary, String description, String reporter,
			String assignee, String buildNumber, String platform) throws Exception
	{
		RemoteIssue issue = new RemoteIssue();
		issue.setProject(project);
		issue.setAssignee(assignee);
		issue.setType(issueType);
		issue.setSummary(summary);
		issue.setDescription("Crash id: #1" + Utils.newLine + Utils.newLine + description);

		issue = issueService.createIssue(user, issue);

		MutableIssue mutableIssue = issueManager.getIssueObject(issue.getKey());
		mutableIssue.setPriorityId(settings.getPriorityId());
		mutableIssue.store();

		updateCustomField(mutableIssue, settings.getBuildCustomFieldName(), buildNumber);
		updateCollectionCustomField(mutableIssue, settings.getPlatformCustomFieldName(), platform);
		updateCustomField(mutableIssue, settings.getCrashCountCustomFieldName(), "1");
		updateCustomField(mutableIssue, settings.getSignatureCustomFieldName(), callStack.getIssueSignature());

		RemoteFieldValue reporterField = new RemoteFieldValue("reporter", new String[] { reporter });
		issueService.updateIssue(user, issue.getKey(), new RemoteFieldValue[] { reporterField });

		return issue.getKey();
	}

	public void updateIssueSignature(Issue issue, SignatureInfo info) throws Exception
	{
		MutableIssue mutableIssue = issueManager.getIssueObject(issue.getKey());
		if (mutableIssue == null)
		{
			log.error("cannot update issue signature for issue:" + issue.getKey());
			return;
		}
		updateCustomField(mutableIssue, crashSignatureField, info.getIssueSignature());
	}

	private void updateCustomField(MutableIssue issue, String customFieldName, String value) throws Exception
	{
		CustomField customField = customFieldManager.getCustomFieldObjectByName(customFieldName);
		if (customField == null)
		{
			throw new Exception("The custom field " + customFieldName + "doesn't exist on jira");
		}
		updateCustomField(issue, customField, value);
	}

	private void updateCustomField(MutableIssue issue, CustomField customField, String value) throws Exception
	{
		try
		{
			CustomFieldType customFieldType = customField.getCustomFieldType();
			Object valueToAdd = customFieldType.getSingularObjectFromString(value);
			ModifiedValue newValue = new ModifiedValue(issue.getCustomFieldValue(customField), valueToAdd);
			IssueChangeHolder dummyChangeHolder = new DefaultIssueChangeHolder();

			customField.updateValue(null, issue, newValue, dummyChangeHolder);
			issue.setCustomFieldValue(customField, value);
		}
		catch (Exception e)
		{
			throw new Exception("An error occurred, the " + customField.getName() + " cannot be updated", e);
		}
	}

	private void updateCollectionCustomField(MutableIssue issue, String customFieldName, String value) throws Exception
	{

		CustomField customField = customFieldManager.getCustomFieldObjectByName(customFieldName);
		if (customField == null)
		{
			throw new Exception("The custom field " + customFieldName + " doesn't exist on jira");
		}
		HashSet<String> set = new HashSet<String>();
		set.add(value);
		updateCollectionCustomField(customField, issue, set);
	}

	private void updateCollectionCustomField(CustomField customField, MutableIssue issue, Collection<String> value) throws Exception
	{
		try
		{
			ModifiedValue newValue = new ModifiedValue(issue.getCustomFieldValue(customField), value);
			IssueChangeHolder dummyChangeHolder = new DefaultIssueChangeHolder();
			customField.updateValue(null, issue, newValue, dummyChangeHolder);
			issue.setCustomFieldValue(customField, value);
		}
		catch (Exception e)
		{
			throw new Exception("An error occurred, the " + customField.getName() + " cannot be updated", e);
		}

	}

	public JiraCrashInfo updateExistingIssue(User user, String issueKey, String description, String reporter, String buildNumber, Date buildTime,
			String platform) throws Exception
	{
		MutableIssue issue = issueManager.getIssueObject(issueKey);

		// even if the following command fail continue...
		JiraCrashInfo result = tryReopenIssue(user, issue, buildTime);

		if (result.getActionPerformed() == JiraCrashInfo.Ignore)
		{
			return result;
		}

		issue = issueManager.getIssueObject(issueKey);

		int crashCount = increaseCrashCount(issue);
		result.setCrashCount(crashCount);

		commentOnIssue(user, issue, description, reporter, buildNumber, platform);
		RemoteIssue remoteIssue = issueService.getIssue(user, issue.getKey());
		result.setIssue(remoteIssue);
		return result;
	}

	private int increaseCrashCount(MutableIssue issue) throws Exception
	{
		try
		{
			int crashCount = getCrashCount(issue);
			crashCount++;
			updateCustomField(issue, crashCountField, Integer.toString(crashCount));
			return crashCount;

		}
		catch (Exception e)
		{
			throw new RuntimeException("Cannot increase crash count for issue: " + issue.getKey(), e);
		}
	}

	private int getCrashCount(Issue issue)
	{
		String crashCountString = null;
		Object crashCountCustomFieldValue = issue.getCustomFieldValue(crashCountField);
		if (crashCountCustomFieldValue == null)
		{
			return 0;
		}

		crashCountString = crashCountCustomFieldValue.toString();
		int crashCount = 0;
		if (crashCountString != null && !crashCountString.trim().equals(""))
		{
			crashCount = (int) Double.parseDouble(crashCountString);

		}
		return crashCount;
	}

	private JiraCrashInfo tryReopenIssue(User user, MutableIssue issue, Date buildTime) throws Exception
	{
		RemoteIssue remoteIssue = issueService.getIssue(user, issue.getKey());
		if (isKsAdIssue(issue))
		{
			return new JiraCrashInfo(remoteIssue, JiraCrashInfo.Ignore, getCrashCount(issue));
		}

		ActionDescriptor openAction = getOpenActionsForCurrentIssue(issue, user);
		if (openAction == null)
		{
			notifyIfIssueActionIsNeeded(issue);
			return new JiraCrashInfo(remoteIssue, JiraCrashInfo.Comment);
		}

		if (hasBeenFixed(issue, buildTime))
		{
			return new JiraCrashInfo(remoteIssue, JiraCrashInfo.FixedInNewerBuild);
		}

		try
		{
			// it seems we need to pass a description otherwise crash
			RemoteFieldValue descriptionField = new RemoteFieldValue("description", new String[] { issue.getDescription() });
			RemoteFieldValue[] params = new RemoteFieldValue[] { descriptionField };
			issueService.progressWorkflowAction(user, issue.getKey(), Integer.toString(openAction.getId()), params);
		}
		catch (Exception e)
		{
			throw new Exception("An Error occurred trying to change issue status " + issue.getKey() + " using the action: " + openAction.getName() + "("
					+ openAction.getId() + ")", e);
		}
		return new JiraCrashInfo(remoteIssue, JiraCrashInfo.Reopen);
	}

	private void notifyIfIssueActionIsNeeded(Issue issue)
	{
		List<String> openActions = settings.getOpenIssueStatuses();
		if (!openActions.contains(issue.getStatusObject().getName().trim()))
		{
			ExceptionHelper.sendErrorEmail("jira couldn't find the action to re-open the following issue " + issue.getKey());
		}
	}

	protected boolean hasBeenFixed(Issue issue, Date buildTime)
	{
		String issueStatusName = issue.getStatusObject().getName().trim();
		List<String> fixedIssueStatuses = settings.getFixedIssueStatuses();

		if (fixedIssueStatuses.contains(issueStatusName))
		{
			// TODO: the Issue.getUpdated() returns the time the last comment
			// has been added, we need to check the last time the status has
			// been changed instead check if
			// ComponentManager.getInstance().getWorklogManager() can help

			new DateTime(buildTime.getTime(), DateTimeZone.UTC);

			DateTime lastUpdatedTime = new DateTime(issue.getUpdated()).plusHours(settings.getHoursDifference());
			if (lastUpdatedTime.isAfter(buildTime.getTime()))
			{
				return true;
			}
		}
		return false;
	}

	private boolean isKsAdIssue(Issue issue)
	{
		return settings.getKsAdIssueStatuses().contains(issue.getStatusObject().getName());
	}

	private ActionDescriptor getOpenActionsForCurrentIssue(MutableIssue issue, User user)
	{
		Collection<ActionDescriptor> availableActions = getAvailableActions(issue, user);
		List<String> actions = settings.getOpenActions();

		for (ActionDescriptor action : availableActions)
		{
			if (actions.contains(action.getName()))
			{
				return action;
			}
		}

		return null;
	}

	private Collection<ActionDescriptor> getAvailableActions(MutableIssue issue, User user)
	{

		ComponentManager manager = ComponentManager.getInstance();
		manager.getJiraAuthenticationContext().setUser(user);

		IssueUtilsBean issueUtilsBean = new IssueUtilsBean(manager.getIssueManager(), manager.getWorkflowManager(), manager.getJiraAuthenticationContext(),
				manager.getPluginAccessor());
		Map<Integer, ActionDescriptor> availableActionMap = issueUtilsBean.loadAvailableActions(issue);

		Collection<ActionDescriptor> availableActions = availableActionMap.values();
		return availableActions;
	}

	private void commentOnIssue(User user, Issue issue, String description, String reporter, String buildNumber, String platform) throws Exception
	{
		CommentManager commentManager = ComponentManager.getInstance().getCommentManager();
		String body = getCommentBody(description, buildNumber, platform, getCrashCount(issue));

		try
		{
			commentManager.create(issue, reporter, body, false);
		}
		catch (Exception exception)
		{
			String message = "This crash happened to " + reporter + ", but the crash handler couldn't find a matching jira account :(";

			ExceptionHelper.sendErrorEmail(message, exception);

			body += Utils.newLine + Utils.newLine + message;
			commentManager.create(issue, user.getName(), body, false);
		}
	}

	private String getCommentBody(String description, String buildNumber, String platform, int crashCount)
	{
		return "Crash id: #" + crashCount + Utils.newLine + Utils.newLine + description + Utils.newLine + Utils.newLine + platform + " build number"
				+ buildNumber;
	}

	public void linkIssues(User user, String bestIssueKey, String[] matchingIssues) throws Exception
	{
		Issue sourceIssue = issueManager.getIssueObject(bestIssueKey);
		if (sourceIssue == null)
		{
			log.error("cannot load issue:" + bestIssueKey);
			return;
		}
		Long sourceIssueId = sourceIssue.getId();
		Collection<Issue> linkedIssues = getLinkedIssues(user, sourceIssue);

		for (String issueKey : matchingIssues)
		{
			Issue matchingIssue = issueManager.getIssueObject(issueKey);
			if (isNewLink(linkedIssues, matchingIssue))
			{
				Long matchingIssueId = matchingIssue.getId();

				Long sequence = null;
				issueLinkManager.createIssueLink(sourceIssueId, matchingIssueId, issueLinkTypeId, sequence, user);
			}

		}
	}

	private Collection<Issue> getLinkedIssues(User user, Issue issue)
	{

		LinkCollection links = issueLinkManager.getLinkCollection(issue, user);
		Collection<Issue> result = links.getAllIssues();
		return result;
	}

	private boolean isNewLink(Collection<Issue> linkedIssues, Issue matchingIssue)
	{
		return matchingIssue != null && !linkedIssues.contains(matchingIssue);
	}
}
