package com.crytek.jira.plugins.crashHandler;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.log4j.Logger;

import com.atlassian.gzipfilter.org.apache.commons.lang.StringUtils;
import com.atlassian.jira.issue.Issue;
import com.atlassian.jira.issue.attachment.Attachment;
import com.atlassian.jira.issue.fields.CustomField;
import com.atlassian.jira.util.AttachmentUtils;

public class SignatureInfoBuilder
{
	public static final int CURRENT_CALL_STACK_VERSION_ID = 4;
	public static final String errorLogAttachmentName = "error";
	public static final String errorLogExtension = ".log";
	public static final long maximumFilesize = 100000;
	private static final Logger log = Logger.getLogger(SignatureInfoBuilder.class);

	private final IssueUpdater issueUpdater;
	private final Pattern callStackWithFilenamePattern;
	private final Pattern callStackWithFunctionsPattern;
	private final Pattern callStackWithFunctionsIndexPattern;

	public SignatureInfoBuilder(IssueUpdater issueUpdater)
	{
		this.issueUpdater = issueUpdater;
		final String anySpaces = "\\s*";
		final String numberAndParenthesis = anySpaces + "\\d+\\)" + anySpaces;
		final String functionNameMatching = "(.+)";
		final String parenthesis = "\\(\\)";
		final String moduleName = "\\[.+\\:\\d+\\]";
		final String hexadecimalFunctionMatching = "function=(0x[A-Fa-f0-9]*)";

		callStackWithFilenamePattern = Pattern.compile(numberAndParenthesis + functionNameMatching + parenthesis + anySpaces + moduleName + anySpaces);
		callStackWithFunctionsPattern = Pattern.compile(numberAndParenthesis + functionNameMatching + parenthesis + anySpaces);
		callStackWithFunctionsIndexPattern = Pattern.compile(numberAndParenthesis + hexadecimalFunctionMatching + anySpaces);
	}

	public SignatureInfo fromIssue(Issue issue, CustomField crashSignatureField) throws Exception
	{
		if (isValidIssue(crashSignatureField, issue))
		{
			String issueSignature = (String) issue.getCustomFieldValue(crashSignatureField);
			int versionNumber = getStackVersionFromSignature(issueSignature);
			String exceptionDescription = extractValue(SignatureInfo.EXCEPTION_DESCRIPTION_LINE_HEADER, issueSignature);
			String signature = extractValue(SignatureInfo.SIGNATURE_LINE_HEADER, issueSignature);
			return new SignatureInfo(versionNumber, exceptionDescription, signature);
		}
		String attachmentText = getErrorLogAttachment(issue);
		if (attachmentText != null && attachmentText != "")
		{
			SignatureInfo result = fromErrorLog(attachmentText);
			issueUpdater.updateIssueSignature(issue, result);
			return result;
		}

		sendError(issue);
		SignatureInfo result = new SignatureInfo(CURRENT_CALL_STACK_VERSION_ID, "", "");
		issueUpdater.updateIssueSignature(issue, result);
		return result;
	}

	protected void sendError(Issue issue)
	{
		log.info("Cannot find issue signature for issue:" + issue.getKey() + " it doesn't contain a valid attachment");
	}

	private boolean isValidIssue(CustomField crashSignatureField, Issue issue)
	{
		boolean result = true;
		String issueSignature = (String) issue.getCustomFieldValue(crashSignatureField);
		if (issueSignature == null || issueSignature.trim() == "" || getStackVersionFromSignature(issueSignature) != CURRENT_CALL_STACK_VERSION_ID)
		{
			result = false;
		}
		return result;
	}

	private int getStackVersionFromSignature(String signature)
	{
		String signatureValue = extractValue(SignatureInfo.CALL_STACK_VERSION_ID_LINE_HEADER, signature);
		if (signatureValue.matches("^\\s*\\d+\\s*$"))
		{
			String signatureNoBlank = signatureValue.replaceAll("\\s", "");
			return Integer.parseInt(signatureNoBlank);
		}
		return -1;
	}

	private String extractValue(String valueName, String from)
	{
		String[] splittedString = from.split(Utils.newLine);
		for (String line : splittedString)
		{
			if (line.startsWith(valueName))
			{
				return line.replaceFirst("^" + valueName, "");
			}
		}
		return "";
	}

	protected String getErrorLogAttachment(Issue issue)
	{
		String result = "";
		try
		{
			@SuppressWarnings("unchecked")
			List<Attachment> attachments = (List) issue.getAttachments();
			for (Attachment attachment : attachments)
			{
				if (isValidAttachment(attachment))
				{
					result = readAttachmentContent(attachment);
					return result;
				}
			}
		}
		catch (Exception e)
		{
			ExceptionHelper.sendErrorEmail("An exception occurred trying to get the error log from attachments for issue " + issue.getKey(), e);
		}
		return null;
	}

	private boolean isValidAttachment(Attachment attachment)
	{
		String filename = attachment.getFilename();
		if (!filename.contains(errorLogAttachmentName))
		{
			return false;
		}
		if (!filename.endsWith(errorLogExtension))
		{
			return false;
		}
		if (attachment.getFilesize() > maximumFilesize)
		{
			return false;
		}
		return AttachmentUtils.getAttachmentFile(attachment).exists();
	}

	private String readAttachmentContent(Attachment attachment) throws FileNotFoundException, IOException
	{
		File file = AttachmentUtils.getAttachmentFile(attachment);
		FileReader fr = new FileReader(file);
		BufferedReader br = new BufferedReader(fr);
		StringBuilder builder = new StringBuilder();
		String line = br.readLine();
		while (line != null)
		{
			builder.append(line);
			builder.append(Utils.newLine);
			line = br.readLine();
		}
		fr.close();
		br.close();
		return builder.toString();
	}

	public SignatureInfo fromErrorLog(String errorLogText) throws Exception
	{
		if (errorLogText == null || errorLogText.trim() == "")
		{
			throw new Exception("the error.log passed as parameter is not valid" + errorLogText);
		}
		String exceptionDescription = getExceptionDescriptionFromCallStack(errorLogText);
		String[] splittedCallStack = Utils.splitInLines(errorLogText);
		String[] strippedCallStack = stripCallStack(splittedCallStack);

		String signature = StringUtils.join(strippedCallStack, SignatureInfo.CALL_STACK_SEPARATOR);

		return new SignatureInfo(CURRENT_CALL_STACK_VERSION_ID, exceptionDescription, signature);
	}

	private String getExceptionDescriptionFromCallStack(String callStack)
	{
		Pattern pattern = Pattern.compile("^" + SignatureInfo.EXCEPTION_DESCRIPTION_LINE_HEADER + "\\s*([^,]*)", Pattern.MULTILINE);
		Matcher matches = pattern.matcher(callStack);
		if (matches.find())
		{
			return matches.group(1);

		}
		return "";

	}

	private String[] stripCallStack(String[] callStack)
	{
		ArrayList<String> strippedCallStack = stripCallStackByPattern(callStack, callStackWithFilenamePattern);
		if (strippedCallStack.isEmpty())
		{
			strippedCallStack = stripCallStackByPattern(callStack, callStackWithFunctionsPattern);
		}
		if (strippedCallStack.isEmpty())
		{
			strippedCallStack = stripCallStackByPattern(callStack, callStackWithFunctionsIndexPattern);
		}

		String[] result = new String[strippedCallStack.size()];
		strippedCallStack.toArray(result);
		return result;
	}

	private ArrayList<String> stripCallStackByPattern(String[] callStack, Pattern pattern)
	{
		ArrayList<String> strippedCallStack = new ArrayList<String>();
		String lastLine = "";
		for (String line : callStack)
		{
			String strippedLine = stripCallStackLineByPattern(line, pattern);

			// ignore recursion
			if (strippedLine != lastLine && strippedLine != "")
			{
				strippedCallStack.add(strippedLine);
			}
			lastLine = strippedLine;

			if (isLineAtEndOfStack(line))
			{
				break;
			}
		}
		return strippedCallStack;
	}

	private String stripCallStackLineByPattern(String callStackLine, Pattern pattern)
	{
		Matcher matcher = pattern.matcher(callStackLine);
		if (matcher.matches())
		{
			return matcher.group(1);
		}
		return "";
	}

	private boolean isLineAtEndOfStack(String line)
	{
		if (line.contains("__tmainCRTStartup"))
		{
			return true;
		}

		if (line.contains("CrySimpleThread<CryRunnable>::RunThis"))
		{
			return true;
		}
		if (line.matches("^\\s*1\\).*$"))
		{
			return true;
		}

		return false;
	}

}
