package com.crytek.jira.plugins.crashHandler.signature;

import com.crytek.jira.plugins.crashHandler.ExceptionHelper;
import com.crytek.jira.plugins.crashHandler.SignatureInfo;
import com.onresolve.jira.crytek.settings.Settings;

/**
 * @author Ken Matches two callstacks. This class used the Optimized
 *         EditDistanceComparer logic. (uses only one row + a variable to create
 *         the edit distance matrix)
 * 
 *         The main concept here is to calculate the weight depending on the
 *         position the change is located. This has been made in order to have
 *         the top differences of the callstack weight more than the bottom
 *         ones.
 * 
 */
public class DistanceWeightedEditComparer extends AbstractCallStackComparer
{
	private final double[] editDistanceBuffer;
	protected final String[] callStackLines;
	private final double differenceRange;
	protected int excludeLines;

	public DistanceWeightedEditComparer(SignatureInfo callstack, Settings setting)
	{
		differenceRange = setting.getDistanceWeightedMatchRange();
		this.callStackLines = callstack.getCrashSignatureLines();
		editDistanceBuffer = new double[callStackLines.length + 1];
		excludeLines = 0;
	}

	@Override
	public double rateSimilarity(SignatureInfo signature)
	{
		String[] otherCallStackLines = signature.getCrashSignatureLines();
		if (callStackLines.length == 0 || otherCallStackLines.length == 0)
		{
			return 0;
		}
		for (int index = 0; index < Math.min(Math.min(callStackLines.length, otherCallStackLines.length), excludeLines); ++index)
		{
			if (!callStackLines[index].equals(otherCallStackLines[index]))
			{
				return 0;
			}
		}
		int maxSize = Math.max(callStackLines.length, otherCallStackLines.length);
		double editDistance = 0;
		try
		{
			editDistance = optimizedLevenshteinDistance(otherCallStackLines);
		}
		catch (Exception e)
		{
			ExceptionHelper.sendErrorEmail("An Exception occurred using DistanceWeightenedEditComparer", e);
		}
		return 1d - (editDistance / maxSize);
	}

	/**
	 * Levenshtein algorithm, calculates the number of changes needs to be made
	 * to transform string 1 in string 2 in this case we are comparing the lines
	 * in the call stack instead of characters.
	 * http://en.wikipedia.org/wiki/Levenshtein_distance
	 * 
	 * it's optimized in terms of memory. allocates only the current callstack
	 * length + 1 rather than creating a new n*m array each time
	 * 
	 * It returns a different value depending on the settings and where the
	 * difference is found in the distance matrix
	 * 
	 * @param otherCallstack
	 *            The other callstack needed for the comparison
	 * @return The edit distance weightened depending on the position of the
	 *         cange
	 */
	public double optimizedLevenshteinDistance(String[] otherCallstackLines)
	{
		double maximumWeight = (sqr(otherCallstackLines.length - 1)) + (sqr(callStackLines.length - 1));
		maximumWeight = Math.sqrt(maximumWeight);
		editDistanceBuffer[0] = 0.;
		for (int columnIndex = 1; columnIndex < editDistanceBuffer.length; ++columnIndex)
		{
			editDistanceBuffer[columnIndex] = editDistanceBuffer[columnIndex - 1] + calculateWeight(columnIndex - 1, 0, maximumWeight, differenceRange);
		}

		for (int lineIndex = 0; lineIndex < otherCallstackLines.length; ++lineIndex)
		{
			double lineValue = editDistanceBuffer[0] + calculateWeight(0, lineIndex, maximumWeight, differenceRange);
			double prev = lineValue;
			for (int columnIndex = 0; columnIndex < callStackLines.length; ++columnIndex)
			{
				double weight = calculateWeight(columnIndex, lineIndex, maximumWeight, differenceRange);
				double editCost = weight;
				if (callStackLines[columnIndex].equals(otherCallstackLines[lineIndex]))
				{
					editCost = 0;
				}
				double insertion = editDistanceBuffer[columnIndex + 1] + weight;
				double deletion = prev + weight;
				double edit = editDistanceBuffer[columnIndex] + editCost;

				editDistanceBuffer[columnIndex] = prev;
				prev = Math.min(edit, Math.min(insertion, deletion));
			}
			editDistanceBuffer[0] = lineValue;
			editDistanceBuffer[editDistanceBuffer.length - 1] = prev;
		}

		return editDistanceBuffer[editDistanceBuffer.length - 1];
	}

	private static double sqr(double d)
	{
		return d * d;
	}

	public double calculateWeight(int ai, int bi, double callstacksSize, double range)
	{
		double distanceFromOrigin = Math.sqrt(sqr(ai) + sqr(bi));
		double normalizedDistance = distanceFromOrigin / callstacksSize;
		return 1. + (range / 2. - range * normalizedDistance);
	}
}
