package com.onresolve.jira.crytek.settings;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;

import org.joda.time.Duration;

import com.atlassian.jira.ComponentManager;
import com.atlassian.jira.issue.Issue;
import com.atlassian.jira.issue.fields.CustomField;
import com.atlassian.jira.issue.link.IssueLinkTypeManager;
import com.atlassian.jira.issue.link.LinkCollection;
import com.atlassian.jira.rpc.soap.service.IssueService;
import com.atlassian.jira.web.action.IssueActionSupport;
import com.crytek.jira.plugins.crashHandler.IssueUpdater;
import com.crytek.jira.plugins.crashHandler.JiraSearch;
import com.crytek.jira.plugins.crashHandler.SignatureInfo;
import com.crytek.jira.plugins.crashHandler.SignatureInfoBuilder;
import com.crytek.jira.plugins.crashHandler.Utils;
import com.crytek.jira.plugins.crashHandler.signature.AbstractCallStackComparer;
import com.crytek.jira.plugins.crashHandler.signature.ComparerFactory;
import com.opensymphony.user.User;

public class CrashHandlerAlgorithmsTestPage extends IssueActionSupport
{
	/**
	 * 
	 */
	private static final long serialVersionUID = -9121248483988287756L;
	private String comparer;
	private String exceptionMessage;
	private final ArrayList<CrashMatch> matches;
	private final Settings settings;
	private User admin;
	private final IssueUpdater updater;
	private final SignatureInfoBuilder signatureBuilder;
	private final CustomField crashSignatureField;
	private Duration executionTime;
	private long totalIssues;

	public CrashHandlerAlgorithmsTestPage()
	{
		settings = Settings.getSettings();
		matches = new ArrayList<CrashMatch>();
		IssueLinkTypeManager issueLinkTypeManager = ComponentManager.getComponentInstanceOfType(IssueLinkTypeManager.class);
		crashSignatureField = ComponentManager.getInstance().getCustomFieldManager().getCustomFieldObjectByName(settings.getSignatureCustomFieldName());
		IssueService service = ComponentManager.getComponentInstanceOfType(IssueService.class);
		updater = new IssueUpdater(service, issueLinkTypeManager, crashSignatureField, settings);
		signatureBuilder = new SignatureInfoBuilder(updater);
	}

	@Override
	protected String doExecute() throws Exception
	{
		if (this.comparer == null)
		{
			this.exceptionMessage = "error, no data submitted";
			return super.doExecute();
		}
		this.exceptionMessage = GatherData();
		return super.doExecute();
	}

	private String GatherData()
	{
		try
		{
			admin = Utils.getAdministator();

			Issue[] allCrashes = new JiraSearch().getAllCrashes(admin);
			totalIssues = allCrashes.length;
			findAllMatches(allCrashes);
		}
		catch (NullPointerException e)
		{
			return "Cannot find Administrator for the current jira configuration for the search";
		}
		catch (Exception e)
		{
			return "An exception occurred " + e.getMessage();
		}
		return "";
	}

	private void findAllMatches(Issue[] allCrashes) throws Exception
	{
		long start = System.currentTimeMillis();
		for (Issue issue : allCrashes)
		{
			CrashMatch match = new CrashMatch(issue.getKey());
			AbstractCallStackComparer comparerForCurrentIssue = getComparerFactory().get(signatureBuilder.fromIssue(issue, crashSignatureField));
			Collection<Issue> linkedIssues = getLinkedIssues(issue);
			for (Issue otherIssue : allCrashes)
			{
				// don't compare the current issue with itself
				if (otherIssue == issue)
				{
					continue;
				}
				SignatureInfo otherObjSignature = signatureBuilder.fromIssue(otherIssue, crashSignatureField);
				double proximity = comparerForCurrentIssue.rateSimilarity(otherObjSignature);
				if (Utils.greaterThan(proximity, settings.getMatchPercentage()))
				{
					match.addMatch(otherIssue.getKey(), proximity, linkedIssues.contains(otherIssue) ? CrashMatch.Status.EXISTING : CrashMatch.Status.NEW);
				}
				else if (linkedIssues.contains(otherIssue))
				{
					match.addMatch(otherIssue.getKey(), proximity, CrashMatch.Status.DELETED);
				}
			}
			matches.add(match);
		}
		executionTime = new Duration(start, System.currentTimeMillis());
	}

	private Collection<Issue> getLinkedIssues(Issue issue)
	{
		LinkCollection links = ComponentManager.getInstance().getIssueLinkManager().getLinkCollection(issue, admin);
		Collection<Issue> result = links.getAllIssues();
		return result;
	}

	public void setComparer(String comparer)
	{
		this.comparer = comparer;
	}

	public String getExceptionMessage()
	{
		return exceptionMessage;
	}

	public ArrayList<CrashMatch> getMatches()
	{
		Collections.sort(matches);
		return matches;
	}

	public ComparerFactory[] getAllComparers()
	{
		return ComparerFactory.values();
	}

	public String getTotalExecutionTime()
	{
		return Long.toString(executionTime.getMillis());
	}

	public String getIssueExecutionTime()
	{
		return Double.toString((double) executionTime.getMillis() / (double) totalIssues);
	}

	public ComparerFactory getComparerFactory()
	{
		return ComparerFactory.valueOf(comparer);
	}

}
