package com.onresolve.jira.crytek.settings;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.util.Arrays;
import java.util.List;

import org.apache.log4j.Logger;
import org.apache.xml.security.exceptions.Base64DecodingException;
import org.apache.xml.security.utils.Base64;

import com.atlassian.jira.ComponentManager;
import com.atlassian.jira.config.ConstantsManager;
import com.atlassian.jira.config.properties.PropertiesManager;
import com.crytek.jira.plugins.crashHandler.ExceptionHelper;
import com.crytek.jira.plugins.crashHandler.Utils;
import com.crytek.jira.plugins.crashHandler.signature.ComparerFactory;

/**
 * @author ken Changing this class can affect settings saved in previous
 *         versions of the plugin, please have a look here:
 *         http://java.sun.com/j2se
 *         /1.3/docs/guide/serialization/spec/version.doc7.html and here:
 *         http://
 *         java.sun.com/j2se/1.3/docs/guide/serialization/spec/version.doc8.html
 */
// do not remove final from this class unless you really know what you are doing
// (effective java 2)
public final class Settings implements Serializable
{
	/**
	 * Auto generated serialVersionUID
	 */
	private static final long serialVersionUID = 3538708836110511127L;
	private static final String settingsPropertyKey = "com.onresolve.jira.crytek.crashHandler.Settings";
	private static final Logger log = Logger.getLogger(Settings.class);
	private static final String[] DEFAULT_OPEN_ACTIONS = { "Update", "Re-Open", "Fix Failed", "Fix failed" };
	private static final String[] DEFAULT_FIXED_ISSUE_STATUSES = { "Closed", "Claimed Fixed" };
	private static final String[] DEFAULT_OPEN_ISSUE_STATUSES = { "Open", "Accepted ( Not yet in progress )", "Assigned", "In Progress" };
	private static final String[] DEFAULT_KS_AD_ISSUE_STATUSES = { "KS/AD" };
	private static final String[] EMPTY_ARRAY = {};
	private static final ComparerFactory DEFAULT_COMPARER = ComparerFactory.PREFIX_COMPARER;
	public static final Double DEFAULT_DISTANCE_WEIGHTENED_MATCH_RANGE = 0.8;
	public static final String DEFAULT_PRIORITY = "40";

	private transient ConstantsManager constantsManager;

	private String maintainersEmail;
	private String emailFrom;
	private String signatureCustomFieldName;
	private String buildCustomFieldName;
	private String platformCustomFieldName;
	private String crashCountCustomFieldName;
	private String defaultPriorityId;

	private List<String> openActions;
	private List<String> fixedIssueStatuses;
	private List<String> openIssueStatuses;
	private List<String> ksAdIssueStatuses;

	private Integer hoursDifference;
	private Double matchPercentage;
	private Double distanceWeightenedMatchRange;
	private String emailSubject;

	private List<String> newCrashPattern;
	private ComparerFactory comparer;

	private Settings()
	{

	}

	private Settings(ConstantsManager constantsManager)
	{
		this.constantsManager = constantsManager;
	}

	private ConstantsManager getConstantsManager()
	{
		if (this.constantsManager == null)
		{
			this.constantsManager = ComponentManager.getInstance().getConstantsManager();
		}
		return constantsManager;
	}

	public static Settings getDefaultSettings()
	{
		return getDefaultSettings(ComponentManager.getInstance().getConstantsManager());
	}

	/**
	 * This function is created in order to be able to test passing a mock
	 * instead of the default constantsManager
	 * 
	 * @param constantsManager
	 * @return default settings
	 */
	public static Settings getDefaultSettings(ConstantsManager constantsManager)
	{
		Settings result = new Settings(constantsManager);
		initialiseWithDefault(result);
		return result;
	}

	public static Settings getSettings()
	{
		log.debug("start getting settings");
		Settings settings = null;
		try
		{
			settings = deserializeSettings();
		}
		catch (Exception e)
		{
			log.error("An exception occurred trying to deserialise the settings:" + e.getMessage());
		}

		if (settings == null)
		{
			settings = new Settings();
		}

		// do not move this call in the readObject method!!
		// (effective java 2nd edition item 17)
		initialiseWithDefault(settings);
		return settings;
	}

	// This method have 2 meanings:
	// - When there are no info in the database the object is initialized with
	// default values;
	// - When a new field is introduced, the deserialisation will produce an
	// empty value for the new field. here a default value is set
	private static void initialiseWithDefault(Settings settings)
	{
		settings.setMaintainersEmail(defaultWhenNull(settings.getMaintainersEmail(), "ken@crytek.com;alexmc@crytek.com"));
		settings.setEmailFrom(defaultWhenNull(settings.getEmailFrom(), "ken@crytek.com"));
		settings.setEmailSubject(defaultWhenNull(settings.getEmailSubject(), "[CrashHandlerPlugin] Exception in jira plugin"));

		settings.setSignatureCustomFieldName(defaultWhenNull(settings.getSignatureCustomFieldName(), "Crash_Signature"));
		settings.setBuildCustomFieldName(defaultWhenNull(settings.getBuildCustomFieldName(), "Build #"));
		settings.setPlatformCustomFieldName(defaultWhenNull(settings.getPlatformCustomFieldName(), "Platform/s"));
		settings.setCrashCountCustomFieldName(defaultWhenNull(settings.getCrashCountCustomFieldName(), "Crashcount"));

		settings.setOpenActions(defaultWhenNull(settings.getOpenActions(), Arrays.asList(DEFAULT_OPEN_ACTIONS)));
		settings.setFixedIssueStatuses(defaultWhenNull(settings.getFixedIssueStatuses(), Arrays.asList(DEFAULT_FIXED_ISSUE_STATUSES)));
		settings.setOpenIssueStatuses(defaultWhenNull(settings.getOpenIssueStatuses(), Arrays.asList(DEFAULT_OPEN_ISSUE_STATUSES)));
		settings.setKsAdIssueStatuses(defaultWhenNull(settings.getKsAdIssueStatuses(), Arrays.asList(DEFAULT_KS_AD_ISSUE_STATUSES)));
		settings.validatePriority(DEFAULT_PRIORITY);

		settings.setNewCrashPattern(defaultWhenNull(settings.getNewCrashPattern(), Arrays.asList(EMPTY_ARRAY)));
		settings.setHoursDifference(defaultWhenNull(settings.getHoursDifference(), 2));
		settings.setMatchPercentage(defaultWhenNull(settings.getMatchPercentage(), 0.7));
		settings.setDistanceWeightedMatchRange(defaultWhenNull(settings.getDistanceWeightedMatchRange(), DEFAULT_DISTANCE_WEIGHTENED_MATCH_RANGE));
		settings.setComparer(defaultWhenNull(settings.getComparer(), DEFAULT_COMPARER));
	}

	private void validatePriority(String defaultPriority)
	{
		if (isValidPriority(getPriorityId()))
		{
			return;
		}
		else if (isValidPriority(defaultPriority))
		{
			this.setPriorityId(defaultPriority);
		}
		else
		{
			this.setPriorityId(getConstantsManager().getDefaultPriorityObject().getId().toString());
		}
	}

	private boolean isValidPriority(String priorityId)
	{
		try
		{
			getConstantsManager().getPriorityName(priorityId);
			return true;
		}
		catch (Exception e)
		{
			return false;
		}
	}

	private static Settings deserializeSettings() throws IOException, ClassNotFoundException, Base64DecodingException
	{
		String propertyKey = settingsPropertyKey;
		String serializedForm = PropertiesManager.getInstance().getPropertySet().getText(propertyKey);
		if (serializedForm == null)
		{
			log.info("no settings found in the database");
			return null;
		}

		// the Base64.decode method is used to avoid the problem described here:
		// http://forums.sun.com/thread.jspa?threadID=479051&messageID=2228486
		ByteArrayInputStream st = new ByteArrayInputStream(Base64.decode(serializedForm));
		ObjectInputStream ois = new ObjectInputStream(st);
		Settings settings = null;
		try
		{
			settings = (Settings) ois.readObject();
		}
		catch (IOException e)
		{
			log.error("An IOException occurred during the setting deserialisation. :" + Utils.newLine + e.getMessage());
			ExceptionHelper.sendErrorEmail("Cannot read property " + settingsPropertyKey + " during deserialisation of settings, an IO error occurred: ", e);
			throw e;
		}
		catch (ClassNotFoundException e)
		{
			log.error("An error occurred trying to deserialise the Class: " + Utils.newLine + e.getMessage());
			throw e;
		}
		finally
		{
			ois.close();
		}
		return settings;
	}

	private static <T> T defaultWhenNull(T value, T defaultString)
	{
		return value == null ? defaultString : value;
	}

	public static void saveSettings(Settings settings)
	{
		PropertiesManager propertyManager = PropertiesManager.getInstance();
		String settingToRemove = settingsPropertyKey;
		try
		{
			deleteSettings();
		}
		catch (Exception e)
		{
			log.warn("Cannot remove property: " + settingToRemove + " it's probably the first time the settings has been saved");
		}
		try
		{
			serializeSettings(settings, propertyManager);
		}
		catch (Exception e)
		{
			log.error("An error occurred trying to save the settings: " + e.getMessage());
			ExceptionHelper.sendErrorEmail("An error occurred trying to save settings", e);
		}
	}

	static void deleteSettings()
	{
		try
		{
			PropertiesManager.getInstance().getPropertySet().remove(settingsPropertyKey);
		}
		catch (Exception e)
		{
			log.error("There's no setting to delete");
		}
	}

	private static void serializeSettings(Settings settings, PropertiesManager propertyManager) throws IOException
	{
		ByteArrayOutputStream st = new ByteArrayOutputStream();
		ObjectOutputStream so = new ObjectOutputStream(st);
		try
		{
			so.writeObject(settings);
		}
		catch (IOException e)
		{
			log.error("An error occurred during object serialization:" + e.getMessage());
		}
		finally
		{
			so.close();
		}

		// the Base64.encode method is used to avoid the problem described here:
		// http://forums.sun.com/thread.jspa?threadID=479051&messageID=2228486
		String serializedForm = new String(Base64.encode(st.toByteArray()));
		propertyManager.getPropertySet().setText(settingsPropertyKey, serializedForm);
	}

	private void writeObject(ObjectOutputStream outputStream) throws IOException
	{
		outputStream.defaultWriteObject();
	}

	private void readObject(ObjectInputStream inputStream) throws IOException, ClassNotFoundException
	{
		inputStream.defaultReadObject();
	}

	public String getMaintainersEmail()
	{
		return maintainersEmail;
	}

	// the setters are with the default accessibility. visible only from the
	// same package
	void setMaintainersEmail(String maintainersEmail)
	{
		this.maintainersEmail = maintainersEmail;
	}

	public String getEmailFrom()
	{
		return emailFrom;
	}

	void setEmailFrom(String emailFrom)
	{
		this.emailFrom = emailFrom;
	}

	public String getEmailSubject()
	{
		return emailSubject;
	}

	void setEmailSubject(String emailSubject)
	{
		this.emailSubject = emailSubject;
	}

	public String getSignatureCustomFieldName()
	{
		return signatureCustomFieldName;
	}

	void setSignatureCustomFieldName(String signatureCustomFieldName)
	{
		this.signatureCustomFieldName = signatureCustomFieldName;
	}

	public String getPlatformCustomFieldName()
	{
		return platformCustomFieldName;
	}

	void setPlatformCustomFieldName(String platformCustomFieldName)
	{
		this.platformCustomFieldName = platformCustomFieldName;
	}

	public String getBuildCustomFieldName()
	{
		return buildCustomFieldName;
	}

	void setBuildCustomFieldName(String buildCustomFieldName)
	{
		this.buildCustomFieldName = buildCustomFieldName;
	}

	public String getCrashCountCustomFieldName()
	{
		return crashCountCustomFieldName;
	}

	void setCrashCountCustomFieldName(String crashCountCustomFieldName)
	{
		this.crashCountCustomFieldName = crashCountCustomFieldName;
	}

	public List<String> getOpenActions()
	{
		return openActions;
	}

	public String getPriorityId()
	{
		return defaultPriorityId;
	}

	public void setPriorityId(String priority)
	{
		defaultPriorityId = priority;
	}

	void setOpenActions(List<String> openActions)
	{
		this.openActions = openActions;
	}

	public List<String> getFixedIssueStatuses()
	{
		return fixedIssueStatuses;
	}

	void setFixedIssueStatuses(List<String> fixedActions)
	{
		this.fixedIssueStatuses = fixedActions;
	}

	public List<String> getOpenIssueStatuses()
	{
		return openIssueStatuses;
	}

	void setOpenIssueStatuses(List<String> openIssueStatuses)
	{
		this.openIssueStatuses = openIssueStatuses;
	}

	public Integer getHoursDifference()
	{
		return hoursDifference;
	}

	void setHoursDifference(Integer hoursDifference)
	{
		this.hoursDifference = hoursDifference;
	}

	public List<String> getKsAdIssueStatuses()
	{
		return ksAdIssueStatuses;
	}

	void setKsAdIssueStatuses(List<String> ksAdName)
	{
		this.ksAdIssueStatuses = ksAdName;
	}

	public Double getMatchPercentage()
	{
		return matchPercentage;
	}

	void setMatchPercentage(Double matchPercentage)
	{
		this.matchPercentage = matchPercentage;
	}

	void setNewCrashPattern(List<String> newCrashPattern)
	{
		this.newCrashPattern = newCrashPattern;
	}

	public List<String> getNewCrashPattern()
	{
		return this.newCrashPattern;
	}

	void setDistanceWeightedMatchRange(Double range)
	{
		this.distanceWeightenedMatchRange = range;
	}

	public Double getDistanceWeightedMatchRange()
	{
		return distanceWeightenedMatchRange;
	}

	void setComparer(ComparerFactory comparer)
	{
		this.comparer = comparer;
	}

	public ComparerFactory getComparer()
	{
		return this.comparer;
	}

}
