package com.crytek.jira.plugins;

import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;

import java.sql.Timestamp;
import java.util.Date;

import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;

import com.atlassian.jira.config.ConstantsManager;
import com.atlassian.jira.issue.Issue;
import com.atlassian.jira.issue.status.Status;
import com.crytek.jira.plugins.crashHandler.IssueUpdater;
import com.onresolve.jira.crytek.settings.Settings;

public class UnitTestIssueUpdater
{

	private TestingIssueUpdater issueUpdater;
	private Issue issue;
	private Status status;
	private Timestamp today;
	private Timestamp lastMonth;
	private Timestamp buildTimeToday;
	private Timestamp buildTimeLastMonth;

	@Before
	public void Before()
	{
		ConstantsManager constantsManager = mock(ConstantsManager.class);
		when(constantsManager.getPriorityName(null)).thenReturn("");
		issueUpdater = new TestingIssueUpdater(Settings.getDefaultSettings(constantsManager));
		issue = mock(Issue.class);
		status = mock(Status.class);

		today = new Timestamp(new DateTime().getMillis());
		lastMonth = new Timestamp(new DateTime().minusMonths(1).getMillis());
		buildTimeToday = today;
		buildTimeLastMonth = lastMonth;
	}

	@Test
	public void hasBeenFixed_WhenClosedRecently_True()
	{
		withIssueStatusName("Closed");
		withIssueUpdated(today);

		boolean result = issueUpdater.testHasBeenFixed(issue, buildTimeLastMonth);
		assertTrue(result);

	}

	@Test
	public void hasBeenFixed_WhenClosedAndBuildSameTime_True()
	{
		withIssueStatusName("Closed");
		withIssueUpdated(today);

		boolean result = issueUpdater.testHasBeenFixed(issue, buildTimeToday);
		assertTrue(result);
	}

	@Test
	public void hasBeenFixed_WhenClaimedFixedRecently_True()
	{
		withIssueStatusName("Claimed Fixed");
		withIssueUpdated(today);

		boolean result = issueUpdater.testHasBeenFixed(issue, buildTimeLastMonth);
		assertTrue(result);

	}

	@Test
	public void hasBeenFixed_WhenIssueStillExist_False()
	{
		withIssueStatusName("Claimed Fixed");
		withIssueUpdated(lastMonth);

		boolean result = issueUpdater.testHasBeenFixed(issue, buildTimeToday);
		assertFalse(result);
	}

	@Test
	public void hasBeenFixed_WhenIssueStatusOpen_False()
	{
		withIssueStatusName("Open");

		boolean result = issueUpdater.testHasBeenFixed(issue, buildTimeToday);
		assertFalse(result);
	}

	/**
	 * Testing the library behavior
	 */
	@Test
	public void dateTimeIsAfter_SameDateWithDifferentTimezones_False()
	{
		DateTime utcNow = new DateTime(DateTimeZone.UTC);
		assertFalse(utcNow.isAfter(utcNow));

		DateTime differentTimeZoneNow = new DateTime(utcNow, DateTimeZone.forOffsetHours(5));
		assertFalse(utcNow.isAfter(differentTimeZoneNow));
		assertFalse(differentTimeZoneNow.isAfter(utcNow));
		assertTrue(differentTimeZoneNow.getHourOfDay() != utcNow.getHourOfDay());
	}

	private void withIssueUpdated(Timestamp reference)
	{
		when(issue.getUpdated()).thenReturn(reference);
	}

	private void withIssueStatusName(String description)
	{
		when(issue.getStatusObject()).thenReturn(status);
		when(status.getName()).thenReturn(description);
	}

	@After
	public void After()
	{

	}

	/**
	 * this class is needed to test protected method over issue updater
	 * 
	 * @author ken
	 */
	private class TestingIssueUpdater extends IssueUpdater
	{

		public TestingIssueUpdater(Settings settings)
		{
			this.settings = settings;
		}

		// TODO:add the code to check the build time
		public boolean testHasBeenFixed(Issue issue, Date buildTime)
		{
			return hasBeenFixed(issue, buildTime);
			// return isClosedAndBuildNumberOlder(issue, buildTime);
		}
	}
}
