package com.crytek.jira.plugins;

import static org.junit.Assert.assertArrayEquals;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;
import static org.mockito.Matchers.eq;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;

import com.atlassian.jira.issue.Issue;
import com.atlassian.jira.issue.fields.CustomField;
import com.crytek.jira.plugins.crashHandler.IssueUpdater;
import com.crytek.jira.plugins.crashHandler.SignatureInfo;
import com.crytek.jira.plugins.crashHandler.SignatureInfoBuilder;
import com.crytek.jira.plugins.crashHandler.Utils;

public class UnitTestSignatureInfo
{
	private static final String expectedDescription = "EXCEPTION_ACCESS_VIOLATION";
	private static String realCallStack;
	private static String attachmentWithoutErrorTerminator;
	private static String attachmentWithoutErrorTerminatorExpectedSignatureInfo;
	private static String expectedSignature;
	private static String expectedIssueSignature;
	private static String functionOnlyErrorLog;
	private static String functionOnlyErrorLogExpected;
	private static String fNamesOnlyErrorLog;
	private static String fNamesOnlyErrorLogExpected;

	private Issue issue;
	private IssueUpdater issueUpdater;
	private CustomField crashSignatureField;
	private TestingSignatureInfoBuilder builder;

	@BeforeClass
	public static void beforeClass()
	{
		String callStackInfoHead = SignatureInfo.CALL_STACK_VERSION_ID_LINE_HEADER + SignatureInfoBuilder.CURRENT_CALL_STACK_VERSION_ID + Utils.newLine
				+ SignatureInfo.EXCEPTION_DESCRIPTION_LINE_HEADER + "EXCEPTION_ACCESS_VIOLATION" + Utils.newLine + SignatureInfo.SIGNATURE_LINE_HEADER;

		realCallStack = "Logged at Friday, August 21, 2009 19:29:38" + Utils.newLine + "FileVersion: 1.1.1.358" + Utils.newLine + "ProductVersion: 1.1.1.358"
				+ Utils.newLine + "" + Utils.newLine + "Exception Code: 0xC0000005" + Utils.newLine + "Exception Addr: 0x0023:0x35198B65" + Utils.newLine
				+ "Exception Module: CryPhysics.dll" + Utils.newLine
				+ "Exception Description: EXCEPTION_ACCESS_VIOLATION, Attempt to read from address 0x0FFE2DEC" + Utils.newLine + "" + Utils.newLine
				+ "The memory could not be \"read\"" + Utils.newLine + "" + Utils.newLine + "Call Stack Trace:" + Utils.newLine
				+ "14) CTriMesh::Intersect()  [trimesh.cpp:2751]" + Utils.newLine + "13) CRopeEntity::CheckCollisions()  [ropeentity.cpp:1513]" + Utils.newLine
				+ "12) CRopeEntity::Step()  [ropeentity.cpp:1923]" + Utils.newLine
				+ "11) CPhysicalWorld::ProcessNextEngagedIndependentEntity()  [physicalworld.cpp:1646]" + Utils.newLine
				+ "10) CPhysicalWorld::TimeStep()  [physicalworld.cpp:2016]" + Utils.newLine + " 9) CPhysicsThreadTask::Run()  [system.cpp:875]"
				+ Utils.newLine + " 8) CPhysicsThreadTask::OnUpdate()  [system.cpp:832]" + Utils.newLine
				+ " 7) CThreadTask_Thread::SingleUpdate()  [threadtask.cpp:77]" + Utils.newLine + " 6) CThreadTask_Thread::Run()  [threadtask.cpp:101]"
				+ Utils.newLine + " 5) CrySimpleThread<CryRunnable>::RunThis()  [crythread_windows.h:187]" + Utils.newLine + " 4) endthreadex()"
				+ Utils.newLine + " 3) endthreadex()" + Utils.newLine + " 2) FlsSetValue()" + Utils.newLine + " 1) FlsSetValue()" + Utils.newLine + ""
				+ Utils.newLine + "" + Utils.newLine + "Suspended thread (Main):" + Utils.newLine + " 2) FlsSetValue()" + Utils.newLine + " 1) FlsSetValue()"
				+ Utils.newLine + "" + Utils.newLine + "" + Utils.newLine + "Suspended thread (Particles):" + Utils.newLine + " 2) FlsSetValue()"
				+ Utils.newLine + " 1) FlsSetValue()" + Utils.newLine;
		expectedSignature = "CTriMesh::Intersect CRopeEntity::CheckCollisions CRopeEntity::Step CPhysicalWorld::ProcessNextEngagedIndependentEntity CPhysicalWorld::TimeStep CPhysicsThreadTask::Run CPhysicsThreadTask::OnUpdate CThreadTask_Thread::SingleUpdate CThreadTask_Thread::Run CrySimpleThread<CryRunnable>::RunThis";

		expectedIssueSignature = callStackInfoHead + expectedSignature;

		attachmentWithoutErrorTerminator = "Logged at Thursday, August 20, 2009 17:13:34" + Utils.newLine + "FileVersion: 1.1.1.357" + Utils.newLine
				+ "ProductVersion: 1.1.1.357" + Utils.newLine + "" + Utils.newLine + "Exception Code: 0xC0000005" + Utils.newLine
				+ "Exception Addr: 0x001B:0x4CD82BFD" + Utils.newLine + "Exception Module: <Unknown>" + Utils.newLine
				+ "Exception Description: EXCEPTION_ACCESS_VIOLATION, Attempt to read from address 0x4CD82BFD" + Utils.newLine + "" + Utils.newLine
				+ "The memory could not be \"read\"" + Utils.newLine + "" + Utils.newLine + "Call Stack Trace:" + Utils.newLine + "30) function=0x4CD82BFD"
				+ Utils.newLine + "29) CPlayerMelee::DoMeleeEnd()  [playermelee.cpp:377]" + Utils.newLine + "28) CPlayerMelee::Update()  [playermelee.cpp:91]"
				+ Utils.newLine + "27) CPlayer::Update()  [player.cpp:893]" + Utils.newLine + "26) CGameObject::Update()  [gameobject.cpp:596]" + Utils.newLine
				+ "25) CEntity::Update()  [entity.cpp:318]" + Utils.newLine + "24) CEntitySystem::DoUpdateLoop()  [entitysystem.cpp:843]" + Utils.newLine
				+ "23) function=0x06DA6EA8" + Utils.newLine + "22) CEntitySystem::FindEntityByName()" + Utils.newLine + "21) CEntitySystem::PauseTimers()"
				+ Utils.newLine + "20) CEntitySystem::Init()  [entitysystem.cpp:183]" + Utils.newLine + "19) CEntitySystem::AddEntityEventListener()"
				+ Utils.newLine + "18) CEntitySystem::AddSink()" + Utils.newLine + "17) CEntitySystem::ChangeEntityName()" + Utils.newLine
				+ "16) CEntitySystem::ResetAreas()" + Utils.newLine + "15) CEntitySystem::RemoveSink()  [entitysystem.cpp:264]" + Utils.newLine
				+ "14) CEntitySystem::UpdateNotSeenTimeouts()" + Utils.newLine + "13) CEntitySystem::GetEntity()" + Utils.newLine
				+ "12) CEntitySystem::GetPhysicalEntitiesInBox()" + Utils.newLine + "11) CEntitySystem::IsIDUsed()" + Utils.newLine
				+ "10) CEntitySystem::DebugDraw()" + Utils.newLine + " 9) CEntitySystem::CreateEntityArchetype()" + Utils.newLine
				+ " 8) CEntitySystem::GetNumEntities()  [entitysystem.cpp:600]" + Utils.newLine + " 7) CEntitySystem::GetClassRegistry()" + Utils.newLine
				+ " 6) CEntitySystem::FindEntityByGuid()" + Utils.newLine + " 5) CEntitySystem::OnBeforeSpawn()" + Utils.newLine
				+ " 4) CEntitySystem::DoUpdateLoop()" + Utils.newLine + " 3) CEntitySystem::GetEntityIterator()" + Utils.newLine
				+ " 2) CEntitySystem::SendEventToAll()" + Utils.newLine + " 1) CEntityClass::SetFlags()" + Utils.newLine + "" + Utils.newLine + ""
				+ Utils.newLine + "Suspended thread (Physics):" + Utils.newLine + "10) CEntityClass::GetIEntityScript()  [entityclass.h:48]" + Utils.newLine
				+ " 9) CEntitySystem::GetMemoryStatistics()" + Utils.newLine + " 8) CEntitySystem::ActivatePrePhysicsUpdateForEntity()" + Utils.newLine
				+ " 7) CEntitySystem::OnEntityEvent()" + Utils.newLine + " 6) CEntitySystem::UnregisterEntityGuid()" + Utils.newLine
				+ " 5) CEntitySystem::LockSpawning()" + Utils.newLine + " 4) CEntitySystem::GetAreaManager()" + Utils.newLine
				+ " 3) CEntitySystem::CheckInternalConsistency()" + Utils.newLine + " 2) CEntitySystem::RemoveEntityEventListener()" + Utils.newLine
				+ " 1) CEntitySystem::LoadEntityArchetype()" + Utils.newLine + "" + Utils.newLine + "" + Utils.newLine + "Suspended thread (Particles):"
				+ Utils.newLine + "10) CEntitySystem::CreateEntityArchetype()  [entitysystem.cpp:1486]" + Utils.newLine + " 9) CEntitySystem::UpdateTimers()"
				+ Utils.newLine + " 8) CEntitySystem::~CEntitySystem()" + Utils.newLine + " 7) CEntityArchetypeManager::CEntityArchetypeManager()"
				+ Utils.newLine + " 6) function=0x20642520" + Utils.newLine + " 5) function=0x69746E65" + Utils.newLine + " 4) function=0x73656974"
				+ Utils.newLine + " 3) function=0x64252820" + Utils.newLine + " 2) function=0x74636120" + Utils.newLine + " 1) function=0x29657669"
				+ Utils.newLine;

		functionOnlyErrorLog = "Logged at Friday, September 25, 2009 10:49:01" + Utils.newLine + "FileVersion: 1.1.1.458" + Utils.newLine
				+ "ProductVersion: 1.1.1.458" + Utils.newLine +

				"Exception Code: 0xC0000005" + Utils.newLine + "Exception Addr: 0x001B:0x391CA8B1" + Utils.newLine + "Exception Module: <Unknown>"
				+ Utils.newLine + "Exception Description: EXCEPTION_ACCESS_VIOLATION, Attempt to read from address 0x0000006C" + Utils.newLine +

				"The memory could not be \"read\"" + Utils.newLine +

				"Call Stack Trace:" + Utils.newLine + "21) function=0x391CA8B1" + Utils.newLine + "20) function=0x391CAB03" + Utils.newLine
				+ "19) function=0x391887C4" + Utils.newLine + "18) function=0x3904D207" + Utils.newLine + "17) function=0x306131F1" + Utils.newLine
				+ "16) function=0x30513EDE" + Utils.newLine + "15) function=0x30515F12" + Utils.newLine + "14) function=0x305163BF" + Utils.newLine
				+ "13) function=0x33503E65" + Utils.newLine + "12) function=0x335067A8" + Utils.newLine + "11) function=0x33503B44" + Utils.newLine
				+ "10) function=0x3653457F" + Utils.newLine + " 9) function=0x307692FD" + Utils.newLine + " 8) function=0x39080B66" + Utils.newLine
				+ " 7) function=0x39002E71" + Utils.newLine + " 6) function=0x39006511" + Utils.newLine + " 5) function=0x37004CB7" + Utils.newLine
				+ " 4) function=0x37004F79" + Utils.newLine + " 3) function=0x37007526" + Utils.newLine + " 2) function=0x7C817067" + Utils.newLine
				+ " 1) function=0x7C817067" + Utils.newLine +

				"Suspended thread (Physics):" + Utils.newLine + " 2) function=0x7C817067" + Utils.newLine + " 1) function=0x7C817067" + Utils.newLine +

				"Suspended thread (Particles):" + Utils.newLine + " 2) function=0x7C817067" + Utils.newLine + " 1) function=0x7C817067" + Utils.newLine;
		functionOnlyErrorLogExpected = "0x391CA8B1 0x391CAB03 0x391887C4 0x3904D207 0x306131F1 0x30513EDE 0x30515F12 0x305163BF 0x33503E65 0x335067A8 0x33503B44 0x3653457F 0x307692FD 0x39080B66 0x39002E71 0x39006511 0x37004CB7 0x37004F79 0x37007526 0x7C817067 0x7C817067";

		fNamesOnlyErrorLog = "Logged at Friday, September 25, 2009 10:49:01" + Utils.newLine + "FileVersion: 1.1.1.458" + Utils.newLine
				+ "ProductVersion: 1.1.1.458" + Utils.newLine +

				"Exception Code: 0xC0000005" + Utils.newLine + "Exception Addr: 0x001B:0x391CA8B1" + Utils.newLine + "Exception Module: <Unknown>"
				+ Utils.newLine + "Exception Description: EXCEPTION_ACCESS_VIOLATION, Attempt to read from address 0x0000006C" + Utils.newLine +

				"The memory could not be \"read\"" + Utils.newLine +

				"Call Stack Trace:" + Utils.newLine + "15) RtlZombifyActivationContext()" + Utils.newLine + "14) DeactivateActCtx()" + Utils.newLine
				+ "13) function=0x00000000786E04AE" + Utils.newLine + "12) GetSystemMetrics()" + Utils.newLine + "11) PeekMessageW()" + Utils.newLine
				+ "10) CallWindowProcA()" + Utils.newLine + " 9) CXTPHookManager::HookWndProc() " + Utils.newLine + " 8) GetSystemMetrics()" + Utils.newLine
				+ " 7) GetMessageW()" + Utils.newLine + " 6) function=0x000000007873DCAA" + Utils.newLine + " 5) function=0x000000007873E577" + Utils.newLine
				+ " 4) function=0x00000000786F52F4" + Utils.newLine + " 3) __tmainCRTStartup() " + Utils.newLine + " 2) BaseProcessStart()" + Utils.newLine
				+ " 1) BaseProcessStart()" + Utils.newLine;

		fNamesOnlyErrorLogExpected = "RtlZombifyActivationContext DeactivateActCtx GetSystemMetrics PeekMessageW CallWindowProcA CXTPHookManager::HookWndProc GetSystemMetrics GetMessageW __tmainCRTStartup";
		attachmentWithoutErrorTerminatorExpectedSignatureInfo = callStackInfoHead
				+ "CPlayerMelee::DoMeleeEnd CPlayerMelee::Update CPlayer::Update CGameObject::Update CEntity::Update CEntitySystem::DoUpdateLoop CEntitySystem::Init CEntitySystem::RemoveSink CEntitySystem::GetNumEntities";

	}

	@Before
	public void before()
	{
		issue = mock(Issue.class);
		crashSignatureField = mock(CustomField.class);
		issueUpdater = mock(IssueUpdater.class);
		builder = new TestingSignatureInfoBuilder(issueUpdater);
	}

	@Test
	public void fromCallStack_returnsValidSignatureInfo() throws Exception
	{
		SignatureInfo signatureInfo = builder.fromErrorLog(realCallStack);

		assertEquals(SignatureInfoBuilder.CURRENT_CALL_STACK_VERSION_ID, signatureInfo.getCallStackVersionId());
		assertEquals(expectedSignature, signatureInfo.getSignature());
		assertEquals(expectedDescription, signatureInfo.getExceptionDescription());
	}

	@Test
	public void fromCallStack_returnsFunctionAddresses_WhenFunctionOnlyCallStack() throws Exception
	{
		SignatureInfo signatureInfo = builder.fromErrorLog(functionOnlyErrorLog);
		assertEquals(functionOnlyErrorLogExpected, signatureInfo.getSignature());
	}

	@Test
	public void fromCallStack_returnsFunctionNames_WhenFNamesOnlyCallStack() throws Exception
	{
		SignatureInfo signatureInfo = builder.fromErrorLog(fNamesOnlyErrorLog);
		assertEquals(fNamesOnlyErrorLogExpected, signatureInfo.getSignature());
	}

	@Test
	public void fromIssue_whenIssueHaveSignature_shouldReturnValidSignatureInfo() throws Exception
	{
		issueReturnsSignature(issue, expectedIssueSignature);

		SignatureInfo info = builder.fromIssue(issue, crashSignatureField);
		assertEquals(expectedIssueSignature, info.getIssueSignature());
	}

	@Test
	public void fromIssue_noSignatureButAttachment_shouldReturnValidSignatureInfo() throws Exception
	{
		issueReturnsSignature(issue, "");
		builder.setErrorLogAttachment(attachmentWithoutErrorTerminator);

		SignatureInfo info = builder.fromIssue(issue, crashSignatureField);

		verify(issueUpdater).updateIssueSignature(eq(issue), eq(info));
		assertEquals(attachmentWithoutErrorTerminatorExpectedSignatureInfo, info.getIssueSignature());
	}

	@Test
	public void fromIssue_onlyDescription_shouldReturnEmptySignatureInfo() throws Exception
	{
		issueReturnsSignature(issue, "");
		builder.setErrorLogAttachment("");

		SignatureInfo info = builder.fromIssue(issue, crashSignatureField);

		verify(issueUpdater).updateIssueSignature(eq(issue), eq(info));

		assertTrue(builder.isErrorSent());
		assertEquals("", info.getSignature());
		assertEquals("", info.getExceptionDescription());
		assertEquals(SignatureInfoBuilder.CURRENT_CALL_STACK_VERSION_ID, info.getCallStackVersionId());
	}

	@Test
	public void SignatureInfo_withNullCallstack_shouldReturnEmpty()
	{
		SignatureInfo signature = new SignatureInfo(SignatureInfoBuilder.CURRENT_CALL_STACK_VERSION_ID, null, null);
		assertArrayEquals(new String[] {}, signature.getCrashSignatureLines());
	}

	private void issueReturnsSignature(Issue issue, String signature)
	{
		when(issue.getCustomFieldValue(crashSignatureField)).thenReturn(signature);
	}

	private class TestingSignatureInfoBuilder extends SignatureInfoBuilder
	{
		public TestingSignatureInfoBuilder(IssueUpdater issueUpdater)
		{
			super(issueUpdater);
		}

		String errorLogAttachmentResult;
		boolean errorSent = false;

		public boolean isErrorSent()
		{
			return errorSent;
		}

		public void setErrorLogAttachment(String result)
		{
			errorLogAttachmentResult = result;
		}

		@Override
		public String getErrorLogAttachment(Issue issue)
		{
			return errorLogAttachmentResult;
		}

		@Override
		protected void sendError(Issue issue)
		{
			errorSent = true;
		}
	}
}
