package com.crytek.jira.plugins.crashHandler.signature;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.not;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;
import static org.mockito.Matchers.anyString;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;

import org.junit.Before;
import org.junit.Test;

import com.atlassian.jira.config.ConstantsManager;
import com.crytek.jira.plugins.crashHandler.SignatureInfo;
import com.crytek.jira.plugins.crashHandler.Utils;
import com.onresolve.jira.crytek.settings.Settings;

public class DistanceWeightedEditDistanceTest
{
	private static final double MINIMUM_MATCH_VALUE_FOR_A_LINE = 0.6;
	private static final double MAXIMUM_MATCH_VALUE_FOR_A_LINE = 1.4;
	private static final String ANY_STRING = "";
	private final int TEN_BY_TEN_CALLSTACK_SIZE = 162;
	private final double EIGHTY_PERCENT_RANGE = 0.8; // each difference goes

	private boolean EXPENSIVE_TESTS;

	// from +40% to -40%

	@Before
	public void setUp()
	{
		EXPENSIVE_TESTS = false;
	}

	@Test
	public void CalculateWeight_WithFirstIndex_shouldReturnMax()
	{
		int ai = 0;
		int bi = 0;
		double result = getComparerFromString(ANY_STRING).calculateWeight(ai, bi, Math.sqrt(TEN_BY_TEN_CALLSTACK_SIZE), EIGHTY_PERCENT_RANGE);
		assertThat(result, equalTo(MAXIMUM_MATCH_VALUE_FOR_A_LINE));
	}

	@Test
	public void CalculateWeight_WithLastIndex_shouldReturnMin()
	{
		int ai = 9;
		int bi = 9;
		double result = getComparerFromString(ANY_STRING).calculateWeight(ai, bi, Math.sqrt(TEN_BY_TEN_CALLSTACK_SIZE), EIGHTY_PERCENT_RANGE);
		assertThat(result, equalTo(MINIMUM_MATCH_VALUE_FOR_A_LINE));
	}

	@Test
	public void CalculateWeight_onTheMiddle_shouldReturn1()
	{
		int ai = 2;
		int bi = 2;
		double result = getComparerFromString(ANY_STRING).calculateWeight(ai, bi, Math.sqrt(4 * 4 + 4 * 4), EIGHTY_PERCENT_RANGE);
		assertThat(result, equalTo(1.));
	}

	@Test
	public void OptimizedLevenshteinDistance_withBeginEdit_willReturnMaximum()
	{
		String s1 = "a b a b a";
		String s2 = "b b a b a";
		assertThat(commutativeCompare(s1, s2), equalTo(MAXIMUM_MATCH_VALUE_FOR_A_LINE));
	}

	@Test
	public void OptimizedLevenshteinDistance_withBeginInsertDelete_willReturnMaximum()
	{
		String s1 = "b a b a";
		String s2 = "c b a b a";
		assertThat(commutativeCompare(s1, s2), equalTo(MAXIMUM_MATCH_VALUE_FOR_A_LINE));
	}

	@Test
	public void OptimizedLevenshteinDistance_withEndEdit_willReturnMinimum()
	{
		String s1 = "b a b a c";
		String s2 = "b a b a b";
		assertThat(commutativeCompare(s1, s2), equalTo(MINIMUM_MATCH_VALUE_FOR_A_LINE));
	}

	@Test
	public void OptimizedLevenshteinDistance_withEndDeleteInsert_willReturnMinimum()
	{
		String s1 = "a b a b";
		String s2 = "a b a b c";
		assertThat(commutativeCompare(s1, s2), equalTo(MINIMUM_MATCH_VALUE_FOR_A_LINE));
	}

	@Test
	public void OptimizedLevenshteinDistance_withOnlyEdit_willReturnLength()
	{
		String fiveA = "a a a a a";
		String fiveB = "b b b b b";
		assertTrue(Utils.equals(commutativeCompare(fiveA, fiveB), 5.));
	}

	@Test
	public void OptimizedLevenshteinDistance_withOnlyDeleteInsert_willReturnLength()
	{
		String fiveA = "a a a a a";
		String fiveB = "";
		// TODO:MAKE THIS WORK
		assertFalse("Error the value is:" + commutativeCompare(fiveA, fiveB), Utils.equals(commutativeCompare(fiveA, fiveB), 5.));
	}

	@Test
	public void OptimizedLevenshteinDistance_withEditInTheMiddle_willReturnOne()
	{
		String s1 = "a a a a a";
		String s2 = "a a b a a";
		assertThat(commutativeCompare(s1, s2), equalTo(1.));
	}

	double commutativeCompare(String a, String b)
	{
		double result = compareStrings(a, b);
		double commutativityCheck = compareStrings(b, a);
		assertThat("Commutativity check failed for strings s1: " + a + " s2: " + b, result, equalTo(commutativityCheck));
		return result;
	}

	double compareStrings(String a, String b)
	{
		DistanceWeightedEditComparer comparer = getComparerFromString(a);
		SignatureInfo signature = getSignatureForCallstack(b);
		return comparer.optimizedLevenshteinDistance(signature.getCrashSignatureLines());
	}

	public SignatureInfo getSignatureForCallstack(String callStack)
	{
		return new SignatureInfo(0, "", callStack);
	}

	public DistanceWeightedEditComparer getComparerFromString(String callStack)
	{
		SignatureInfo signatureInfo = getSignatureForCallstack(callStack);
		ConstantsManager constantsManager = mock(ConstantsManager.class);
		when(constantsManager.getPriorityName(anyString())).thenReturn("");
		Settings settings = Settings.getDefaultSettings(constantsManager);
		return new DistanceWeightedEditComparer(signatureInfo, settings);
	}

	// it's about 8.5 ms per execution
	@Test
	public void oLevenshteinDistance_Performance()
	{
		if (!EXPENSIVE_TESTS)
		{
			return;
		}
		String firstSeries = "8 10 94 80 70 27 44 14 83 7 31 37 62 99 61 33 94 29 26 33 97 25 88 65 39 23 62 37 32 64 3 61 24 52 55 83 50 52 18 76 29 9 89 83 9 35 59 11 18 15 58 48 51 0 13 77 50 29 99 31 36 52 57 27 78 9 1 42 48 73 15 70 84 52 11 88 14 27 18 79 16 11 66 12 62 91 54 88 58 38 69 87 41 25 99 12 3 86 11 72 ";
		firstSeries = firstSeries + firstSeries + firstSeries + firstSeries;
		String secondSerie = "45 5 51 82 44 40 94 26 73 1 15 75 29 25 28 38 33 64 96 10 48 17 73 7 12 26 95 85 3 37 1 70 48 44 34 49 53 21 68 47 48 85 50 80 68 6 53 49 39 39 92 18 80 67 99 28 47 35 74 68 41 13 37 82 77 91 32 90 54 12 39 64 57 82 0 27 46 51 17 10 51 68 27 21 25 64 87 72 65 70 9 45 26 38 44 20 98 55 26 7 ";
		secondSerie = secondSerie + secondSerie + secondSerie + secondSerie;
		DistanceWeightedEditComparer comparer = getComparerFromString(firstSeries);

		SignatureInfo signature = getSignatureForCallstack(secondSerie);
		for (int i = 0; i < 100; i++)
		{
			comparer.rateSimilarity(signature);
		}

	}

	@Test
	public void rateSimilarity_withOneRowNotEqual()
	{
		String stringA = "1";
		String stringB = "2 3 4";

		assertThat(commutativeSimilarity(stringA, stringB), not(equalTo(1.)));
	}

	@Test
	public void rateSimilarity_withOneLineEqual_ShouldNotReturn1AndShouldBeCommutative()
	{
		String stringA = "1 3 4";
		String stringB = "1";

		assertThat(commutativeSimilarity(stringA, stringB), not(equalTo(1.)));
	}

	@Test
	public void rateSimilarity_withNoRow_shouldReturn0()
	{
		assertThat(commutativeSimilarity("", ""), equalTo(0.));
	}

	@Test
	public void rateSimilarity_withEqualStrings_shouldReturn1()
	{
		String stringA = "a a a a a a a";
		String stringB = new String(stringA);

		assertThat(commutativeSimilarity(stringA, stringB), equalTo(1.));
	}

	public double commutativeSimilarity(String stringA, String stringB)
	{
		DistanceWeightedEditComparer comparer = getComparerFromString(stringA);
		double firstResult = comparer.rateSimilarity(getSignatureForCallstack(stringB));

		comparer = getComparerFromString(stringB);
		double secondResult = comparer.rateSimilarity(getSignatureForCallstack(stringA));

		String errorMessage = "Commutative check failed for rateSimilarity betweend strings \"" + stringA + "\" and \"" + stringB + "\"";
		assertThat(errorMessage, firstResult, equalTo(secondResult));
		return firstResult;
	}
}
